package objects;

import helpers.CombatStatistics;

import java.awt.Point;

import utils.Utils;

public class Ship extends CombatObject {
	
	protected int dirX;
	
	protected int dirY;
	
	protected int angle;
	
	protected double vX;
	
	protected double vY;
	
	protected double posXv;
	
	protected double posYv;
	
	private Shootable[] torpedos = new Shootable[Utils.MAX_SIMULTANEOUS_SHOTS];
	
	private int lastTriggeredTorpedo = -1;
	
	private int lastTriggeredTorpedoTime = -1;
	
	private long awayCount;
	
	public Ship(int x, int y, int dirX, int dirY)
	{
		super(x, y);
		vX = 0;
		vY = 0;
		posXv = x;
		posYv = y;
		this.dirX = dirY;
		this.dirY = dirY;
		angle = 0;
		isAlive = false;
		width = 24 + 2;
		height = 18 + 4;
		prepareTorpedos();
		awayCount = 0;
	}
	
	public boolean isLost()
	{
		return awayCount > Utils.HYPERSPACE_DELAY;
	}
	
	public boolean isMoving()
	{
		return Math.abs(moveX) >= 0.125 || Math.abs(moveY) >= 0.125;
	}
	
	public void spawn(int x, int y)
	{
		super.spawn(x, y);
		posXv = x;
		posYv = y;
		vX = 0;
		vY = 0;
		if(isLost())
		{
			prepareTorpedos();
		}
		awayCount = 0;
	}
	
	public void onTorpedoTriggered(int hitTime)
	{
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			if(torpedos[i].isAlive() == false)
			{
				torpedos[i] = new PreShot(hitTime);
				lastTriggeredTorpedo = i;
				lastTriggeredTorpedoTime = CombatStatistics.TIME;
				return;
			}
		}
	}
	
	public void onTorpedoFired(Shot torpedo)
	{
		if(lastTriggeredTorpedo > -1) {
			torpedo.setTimeToDestination(torpedos[lastTriggeredTorpedo].getTimeToDestination() - 2);
			torpedos[lastTriggeredTorpedo] = torpedo;
			lastTriggeredTorpedo = -1;
		}
	}
	
	public boolean isTorpedoReady()
	{
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			if(torpedos[i].isAlive() == false)
			{
				return true;
			}
		}
		return false;
	}
	
	public int getAvailableTorpedos()
	{
		int result = 0;
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			if(torpedos[i].isAlive() == false)
			{
				++result;
			}
		}
		return result;
	}
	
	public Shootable getFastestTorpedo(){
		Shootable result = null;
		int fastest = Integer.MAX_VALUE;
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			int ttd = torpedos[i].getTimeToDestination();
			if(torpedos[i].isAlive() && ttd < fastest)
			{
				fastest = ttd;
				result = torpedos[i];
			}
		}
		return result;
	}
	
	public int getTimeToNextTorpedo()
	{
		int result = Integer.MAX_VALUE;
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			int ttd = torpedos[i].getTimeToDestination();
			if(ttd < result)
			{
				result = ttd;
			}
		}
		return result;
	}
	
	public void prepareTorpedos()
	{
		for(int i = 0; i < Utils.MAX_SIMULTANEOUS_SHOTS; i++)
		{
			torpedos[i] = new PreShot(0, false);
		}
	}
	
	public String toString()
	{
		String result = "Ship #%d {%d, %d / %.3f, %.3f / %d}";
		return String.format(result, id, posX , posY, moveX, moveY, angle);
	}
	
	public int getAngleIndex()
	{
		return angle;
	}
	
	public double getMovementAngle()
	{
		return Utils.TABLES.getAngle(angle);
	}
	
	public void updateAway(int frameDif)
	{
		awayCount += frameDif;
	}
	
	public void update(int newX, int newY, int frameDif) {
		posX = newX;
		posY = newY;
		
		if(lastTriggeredTorpedo > -1)
		{
			// Check and correct...in case of packet loss
			if(CombatStatistics.TIME - lastTriggeredTorpedoTime > 2)
			{				
				torpedos[lastTriggeredTorpedo].destroy();
				lastTriggeredTorpedo = -1;
			}			
		}
	}
	
	public int getRadius() {
		return Math.max(width, height) / 2;
	}
	
	public double getSpeedX()
	{
		return vX;
	}
	
	public double getSpeedY()
	{
		return vY;
	}
	
	public void updateMovement()
	{		
		moveX = Math.floor(vX) / 8.0;
		moveY = Math.floor(vY) / 8.0;
		posXv += moveX;
		posYv += moveY;
	}
	
	public void performAcceleration()
	{	
		if(Math.abs(vX) < 64)
		{
			vX += Utils.TABLES.astCos(angle) / 128.0;
		}
		if(Math.abs(vY) < 64)
		{
			vY += Utils.TABLES.astSin(angle) / 128.0;
		}
		updateMovement();
	}
	
	public void performDeceleration()
	{
		if(vX > 0)
		{
			vX = Math.max(0, vX - Math.floor(vX) / 128.0 - 1 / 256.0);
		} else if (vX < 0) {
			vX = Math.min(0, vX - Math.floor(vX) / 128.0);
		}
		if(vY > 0)
		{
			vY = Math.max(0, vY - Math.floor(vY) / 128.0 - 1 / 256.0);
		} else if(vY < 0)
		{
			vY = Math.min(0, vY - Math.floor(vY) / 128.0);
		}
		updateMovement();
	}
	
	public void performTurnLeft()
	{
		angle = Utils.incByteAngle(angle);
		
		// Check and correct...in case of packet loss
		Point directionCheck = Utils.TABLES.getDirection(angle);
		if(directionCheck.x != dirX)
		{
			while(directionCheck.x != dirX || directionCheck.y != dirY)
			{
				angle = Utils.incByteAngle(angle);
				directionCheck = Utils.TABLES.getDirection(angle);
			}
		}
	}
	
	public void performTurnRight()
	{	
		angle = Utils.decByteAngle(angle);
		
		// Check and correct...in case of packet loss
		Point directionCheck = Utils.TABLES.getDirection(angle);
		if(directionCheck.x != dirX)
		{
			while(directionCheck.x != dirX || directionCheck.y != dirY)
			{
				angle = Utils.decByteAngle(angle);
				directionCheck = Utils.TABLES.getDirection(angle);
			}
		}
	}
	
	public void setDirection(int directionX, int directionY)
	{
		dirX = directionX;
		dirY = directionY;
	}

	public int getDirectionX() {
		return dirX;
	}
	
	public int getDirectionY() {
		return dirY;
	}
	
	public int getMaximumSpeed()
	{
		return 8;
	}
}
