package objects;

import helpers.CombatStatistics;
import utils.Utils;


public class Shot extends CombatObject implements Shootable {

	protected CombatObject owner;
	
	protected TargetObject target;
	
	protected int timeToLive;
	
	protected int timeToHit;
	
	protected boolean hasMissed = false;
	
	public Shot(int x, int y, CombatObject owner, TargetObject target, int frame) {
		super(x, y);
		this.owner = owner;
		scale = NO_SCALE;
		width = 2;
		height = 2;
		timeToLive = Utils.PHOTON_TORPEDO_MIN_ALIVE_TIME;
		timeToHit = Utils.PHOTON_TORPEDO_MIN_ALIVE_TIME;
		if(target != null)
		{			
			this.target = target;
			target.onFiredUpon(this);
		}
		if(owner instanceof Ship) {
			((Ship)owner).onTorpedoFired(this);
		}
	}
	
	public void update(int newX, int newY, int frameDif) {
		super.update(newX, newY, frameDif);
		timeToLive -= frameDif;
		timeToHit -= frameDif;
		if(timeToHit == -2 && !hasMissed && target != null)
		{
			hasMissed = true;
			target.onMissed(this);
		}
	}
	
	public void destroy() {
		super.destroy();
		if(target != null)
		{
			if(!hasMissed)
			{
				hasMissed = true;
				target.onMissed(this);
			}							
		}
		if(owner instanceof Ship) {
			if(timeToLive < 1)
			{
				++CombatStatistics.SHOTS_MISSED;
			}				
		}
		timeToLive = 0;
		timeToHit = 0;
	}
	
	public void setTimeToDestination(int hitTime)
	{
		this.timeToHit = hitTime;
	}
	
	public int getTimeToDestination()
	{
		if(timeToHit < -1)
		{
			return Math.max(0, timeToLive);
		} else {
			return Math.max(0, timeToHit);
		}
	}
	
	public void reAssignTarget(TargetObject newTarget, int hitTime)
	{
		if(target != null)
		{	
			target.onMissed(this);
		}
		target = newTarget;
		timeToHit = hitTime;
		if(target != null)
		{	
			target.onTargeted();
			target.onFiredUpon(this);
		}
	}
	
	public TargetObject getTarget()
	{
		return target;
	}
	
	public CombatObject getOwner()
	{
		return owner;
	}
	
	public String toString()
	{
		String result = "Shot #%d {%d, %d / %.3f, %.3f / %s, %s / %d}";
		if(owner instanceof Ship)
		{
			//return "Shot\t"+moveX+"\t"+moveY;
			return String.format(result, id, posX, posY, moveX, moveY, "Ship", target, movePrecision);
		} else {
			return String.format(result, id, posX, posY, moveX, moveY, "Saucer", target, movePrecision);
		}
	}
	
	public int getMaximumSpeed()
	{
		return 10;
	}
}
