package de.wens02;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.JPanel;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;

import de.caff.asteroid.analysis.DatagramDumper;

public class AsteroidApp
{
	public static final byte	KEY_HYPERSPACE	= 1;
	public static final byte	KEY_FIRE				= 2;
	public static final byte	KEY_THRUST			= 4;
	public static final byte	KEY_RIGHT				= 8;
	public static final byte	KEY_LEFT				= 16;
	public static final byte	KEY_START				= 32;
	
	public static final int		MAX_X						= 1023;			// 0...1023
	public static final int		MAX_Y						=  767;			// 0...767
	public static final int		OFFSET_Y				=  128;			// Vektor-y beginnt bei 128, wird auf 0.. normiert

	public static final int		EXTENT_X				= 1024;
	public static final int		EXTENT_Y				=  768; 		// oben/unten 128 Pixel Rand wegen 4:3 Bildschirm 

	public static final int		FRAMES_5MIN			=  17999;		// Spieldauer in Frames: 5min * 60sec * 60f/sec
	public static int					FRAME_INTERVAL	=  60000;		// Alle #n Frames den Bildschirminhalt zeichnen
	public static int					MAX_FRAME_LOOKAHEAD = 65;		// default: 80 Soviel Frames vorausberechnen

	public static boolean			withMySQL				= false;
	public static boolean			withLog4j				= false;
	public static boolean			withDisplay			= false;
	public static boolean			withDumper			= false;
	private static boolean		withLatency1		= false;

	private DatagramDumper		dumper					= null;
	private boolean						dumper_active		= false;
	private Thread						dumper_thread		= null;
	private int								game_id					= 0;

	private DatagramSocket		socket;
	private InetAddress				address;
	private byte[]						sendbuf					= new byte[] { 'c', 't', 'm', 'a', 'm', 'e', 0, 0 };
	private byte[]						rcvbuf					= new byte[1026];
	private byte							last_frame			= 0;
	private static int				my_ping					= 0;
	private static int				server_ping			= 0;

	private DatagramPacket		send_packet			= null;
	private DatagramPacket		rcv_packet			= null;

	// Indiziert ber eigenen PING-Wert (0..255)
	private static byte[]			keys_send				= new byte[256];
	private static int[]			winkelbyte			= new int[256];
	
	/** Schnittstelle zu SQL, falls man es denn braucht :-) */
	private MySQL							sql							= null;

	/** Liste aller potentiellen Schussziele (Asteroiden, UFO) */
	private Vector<Target>		targets					= new Vector<Target>(27);
	private Vector<Target>		tmpTargets			= new Vector<Target>(27);
	private Vector<Target>		hardTargets			= new Vector<Target>(27);

	/** Nchstes Target auf Kollisionskurs */
	private Target						tCollision			= null;
	
	/** Nchster Asteroid */
	private Asteroid					aNearest				= null;

	private UFO								ufo							= new UFO( 0, 0, 15, 0x929 );
	private Ship							ship						= new Ship( 0, 0, 0, 0 );
	private StringBuffer			textbuf					= new StringBuffer(50);
	private Vector<Text>			texts						= new Vector<Text>();

	/** Anzahl Asteroiden im vorherigen Frame */
	private int								prev_cnt_ast		= 0;

	/** Anzahl Asteroiden im aktuellen Frames */
	private int								curr_cnt_ast		= 0;

	/** Asteroiden des aktuellen Frames */
	private Vector<Asteroid>	asteroids				= new Vector<Asteroid>(30);
	
	/** Liste der Asteroiden und Explosionen aus dem Vektor-RAM,
	 * theoretisch <= 27, in der Praxis taucht aber auch Index >28 auf!
	 */
	private VectorObject[]		vram_objects		= new VectorObject[27];
	private VectorObject[]		vram_objects_new= new VectorObject[27];

	private Vector<Shot>			prev_shots			= new Vector<Shot>(10);
	private Vector<Shot>			shots						= new Vector<Shot>(10);
	
	private Point							tmpPoint				= new Point();
	
	public static int					frame_cnt				= 0;
	public static int					prev_latency		= 0; 
	public static int					latency					= 0;
	public static int					norm_latency		= 0;

	private int								shots_fired			= 0;
	private int								large_ast_hit		= 0;
	private int								medium_ast_hit	= 0;
	private int								small_ast_hit		= 0;
	private int								large_ufo_hit		= 0;
	private int								small_ufo_hit		= 0;

	private int								level						= 0;
	private int								level_shots			= 0;
	private int								level_key_fire	= 0;
	private int								level_ufos			= 0;
	private int								level_startscore= 0;
	private int								level_startframe= 0;
	private int								level_waitframes= 0;
	protected static int			level_part			= 0;	// 0: Beginn, 1: Mittelteil, 2:Ende

	private int								score						= 0;
	private int								scorewrap				= 0;
	private int								highscore				= 0;
	private int								ships_remain		= 0;
	private int								warp_cnt				= 0;

	private boolean						finishing				= false;
	
	public	JFrame						frame						= null;
	
	public AsteroidApp( String hostname )
	{
		sql = new MySQL( withMySQL );
		Log.init( withLog4j );

		try {
			socket	= new DatagramSocket();
			address	= InetAddress.getByName( hostname );

			send_packet	= new DatagramPacket( sendbuf, sendbuf.length, address, 1979 );
			rcv_packet	= new DatagramPacket( rcvbuf, rcvbuf.length );
			
			if ( withLatency1 ) {
				socket.send( send_packet );
			}
		}
		catch ( Exception e ) {
			System.err.println( "Exception: " + e );
			e.printStackTrace();
			System.exit(0);
		}
		for ( int i = 0; i < vram_objects_new.length; i++ ) {
			vram_objects_new[i] = new VectorObject( 0, 0, 0, 0 );
		}		
		if ( withDisplay ) {
			frame = new JFrame( "M.A.M.E. Asteroid Client" ) {
				public void paint( Graphics g )
				{
					try {
						// Mit anderen Worten: Die linke untere Ecke ist bei (0, 128), die rechte obere bei (1023, 895).
						g.clearRect( 0, 0, 1024, 1024 );
						g.drawRect( 0, 128, 1023, 895 - 128 + 1 );
				
						for ( Asteroid a : asteroids ) {
							if ( null == a ) continue;

							g.setColor( Color.darkGray );
				
							if ( a == tCollision ) {
								g.setColor( Color.red );
							}
							a.paint( (Graphics2D)g );
						}
						for ( Shot shot : Ship.shots ) {
							g.setColor( Color.blue );
							shot.paint( g );
						}
						for ( Shot shot : UFO.shots ) {
							g.setColor( Color.orange );
							shot.paint( g );
						}
						for ( int i = 0; i < texts.size(); i++ ) {
							Text d = texts.get( i );
							g.drawString( d.d, d.x, 1024 - d.y );
						}
						if ( ship.visible ) {
							ship.draw( (Graphics2D)g );
						}
						if ( ufo.visible ) {
							ufo.paint( (Graphics2D)g );
						}
					}
					catch( Exception e ) {
						// ignore
					}
				}				
			};
			frame.addWindowListener( new WindowAdapter() {
					public void windowClosing( WindowEvent e )
					{
						frame.dispose();
						System.exit( 0 );
					}
				}
			);
			frame.getContentPane().add( new JPanel() );
			
			frame.setLocation( 1280 + 256 - 16, 0 );
			frame.setSize( 1024 + 16, 1024 );
	
			frame.setVisible( true );
			frame.repaint();
		}
	}

	public void startNewGame( boolean force ) throws SQLException
	{
		if ( force || frame_cnt > 1000 ) {
			game_id = sql.addGame();
		}
		finishing				= false;
		frame_cnt				= 0;
		scorewrap				= 0;
		level						= 0;

		shots_fired			= 0;
		large_ast_hit		= 0;
		medium_ast_hit	= 0;
		small_ast_hit		= 0;
		large_ufo_hit		= 0;
		small_ufo_hit		= 0;
		ships_remain		= 0;
		warp_cnt				= 0;

		if ( null != dumper_thread ) {
			dumper_thread.stop();
			dumper_thread = null;
		}
		SimpleDateFormat df = new SimpleDateFormat( "yyyy-MM-dd_HHmmss" );
		String filename = df.format( new Date() );

		if ( game_id > 0 ) {
			filename = filename + String.format( "_%4d", game_id );
		}
		try {
			if ( withDumper ) {
				dumper = new DatagramDumper( filename );
				dumper_thread = new Thread( dumper, "dumper" );
				dumper_thread.start();
				
				dumper_active = false;
			}
			Log.setFile( filename + ".log" );
		}
		catch( IOException e ) {
			System.err.println( "Exception: " + e );
			System.exit(0);
		}
	}

	public int getRealScore()
	{
		int scorereal = scorewrap * 100000 + score;

		if ( scorereal > highscore ) {
			highscore = scorereal;
		}
		return scorereal;
	}
	
	/**
	 * Aus der Scoredifferenz berechnen, was abgeschossen wurde
	 *  Large Asteroids		20
	 *  Medium Asteroid	  50
	 *  Small Asteroids	 100
	 *  Large Saucer		 200
	 *  Small Saucer		1000
	 */
	public void updateScores( int new_score )
	{
		if ( score > 98000 && new_score < score ) {
			// Umbruch 99999 > 0
			scorewrap++;
		}
		int diff = new_score - score;

		while ( diff >= 1000 ) {
			small_ufo_hit++;
			diff -= 1000;
		}
		if ( score < 10000 && diff >= 200 ) {
			// Groes UFO nur bei Score < 10.000
			large_ufo_hit++;
			diff -= 200;
		}
		while ( diff >= 100 ) {
			small_ast_hit++;
			diff -= 100;
		}
		while ( diff >= 50 ) {
			medium_ast_hit++;
			diff -= 50;
		}
		while ( diff >= 20 ) {
			large_ast_hit++;
			diff -= 20;
		}
		score = new_score;
	}

  public static int byteToUnsigned( byte b )
  {
    return b < 0 ? b + 256 : b;
  }
  
  public static int getWinkelbyte()
  {
  	int wb = winkelbyte[ server_ping ];

  	for ( int i = 1; i <= norm_latency; i++ ) {
			if ( 0 != ( keys_send[ ( server_ping + i ) & 0xFF ] & KEY_LEFT ) ) {
				wb++;
			}
			if ( 0 != ( keys_send[ ( server_ping + i ) & 0xFF ] & KEY_RIGHT ) ) {
				wb--;
			}
  	}
  	return ( wb + 256 ) & 0xFF;
  }
  
  public byte send( byte command ) throws SQLException
	{
		keys_send[my_ping] = command;

		try {
			sendbuf[6] = command;
			sendbuf[7] = (byte)my_ping;

			if ( withLatency1 ) {
				socket.receive( rcv_packet );
				socket.send( send_packet );
			}
			else {
				socket.send( send_packet );
				socket.receive( rcv_packet );
			}
			if ( (byte)( rcvbuf[1024] - last_frame ) != 1 ) {
				Log.fatal( String.format(
						"Frame#%5d: Packet LOSS: last = %02x, this = %02x", frame_cnt, last_frame, rcvbuf[1024] ) );
			}
			if ( null != dumper && dumper_active ) {
				dumper.datagramReceived( rcv_packet, null );
				dumper.datagramSent( send_packet );
			}
			++frame_cnt;

			server_ping		= byteToUnsigned( rcvbuf[1025] );
			prev_latency	= latency;
			latency				= ( my_ping - server_ping + 256 ) & 0xff;

			if ( prev_latency != latency ) {
				Log.warn( String.format( "Frame#%5d: Latenz %d -> %d", frame_cnt, prev_latency, latency ) );
			}
			interpret();

			// Nachhalten der gedrckten Tasten
			if ( 0 != ( keys_send[server_ping] & KEY_LEFT ) ) {
				ship.rotate( true /* left */ );
			}
			if ( 0 != ( keys_send[server_ping] & KEY_RIGHT ) ) {
				ship.rotate( false );
			}
			winkelbyte[server_ping]	= Ship.winkelbyte;

			my_ping = ( my_ping + 1 ) & 0xFF;

			return last_frame = rcvbuf[1024];
		}
		catch ( Exception e ) {
			System.err.println( "Exception: " + e );
			e.printStackTrace();
			return 0;
		}
	}

	public void interpret() throws SQLException
	{
		if ( address.getHostName().equals( "asteroids.heise.de" ) ) {
			String header = new String( rcvbuf, 0, 4 );
	
			if ( header.equals( "game" ) || header.equals( "busy" ) ) {
				System.err.println( new String( rcvbuf ) );
				System.exit(5);
			}
		}
		// Ersten Befehl auf JMPL (0xe0 0xe2) testen:
		final int rb0 = ( (int)rcvbuf[0] & 0xFF );
		final int rb1 = ( (int)rcvbuf[1] & 0xFF );

		if ( 0xe0 != rb1 && 0xe2 != rb1 ) {
			// sollte nicht vorkommen; erster Befehl ist immer ein JMPL
			System.err.println( "interpret: 0x0e|0x02-Fehler:" );
			System.exit(5);
		}
		prev_cnt_ast = asteroids.size();
		asteroids.clear();

		Ship.shots.clear();
		UFO.shots.clear();

		Vector<Shot> tmpVec = prev_shots;

		prev_shots.clear();
		prev_shots = shots;
		shots = tmpVec;

		texts.clear();

		boolean ship_seen		= false;
		boolean saucer_seen = false;
		int			ships_seen	= 0;
		
		int			vx	= 0;
		int			vy	= 0;
		int			vs	= 0;
		int			v1x = 0;
		int			v1y	= 0;

		int			shipdetect	= 0;
		boolean finished		= false;
		boolean gameover		= false;

		int			text_x			= 0;
		int			text_y			= 0;
		boolean	textPending	= false;

		textbuf.setLength(0);

		/** Index fr das Matching der neuen zu den alten Objekten */
		int			vram_cnt		= 0;

		for ( int pc = 2; !finished && !gameover && pc < rcvbuf.length; pc += 2 ) {
			int b0 = ( (int)rcvbuf[pc] & 0xFF );
			int b1 = ( (int)rcvbuf[pc + 1] & 0xFF );
			int op = b1 >> 4;
			int a1 = ( b0 | ( b1 << 8 ) ) & 0xFFF;
			int a2 = 0;

			if ( op <= 0x0A ) {
				pc += 2;
				b0 = ( (int)rcvbuf[pc] & 0xFF );
				b1 = ( (int)rcvbuf[pc + 1] & 0xFF );
				a2 = ( b0 | ( b1 << 8 ) );
			}
			if ( op <= 0x09 ) {
				// 0...9: LVEC langer Vektor
				int dy = ( ( a1 & 0x400 ) == 0 ) ? ( a1 & 0x3FF ) : -( a1 & 0x3FF );
				int dx = ( ( a2 & 0x400 ) == 0 ) ? ( a2 & 0x3FF ) : -( a2 & 0x3FF );
				int vz = a2 >> 12;

				if ( dx == 0 && dy == 0 && vz == 15 ) {
					// Heller Vektor mit Ausdehnung 0 == PUNKT!
					tmpPoint.x = vx;
					tmpPoint.y = vy;

					Shot shot	= Shot.FindShot( prev_shots, tmpPoint );

					if ( null == shot ) {
						shot = new Shot( vx, vy );
						
						if ( ship.getSquaredTorusDistance( shot, 0 ) < 22.0 * 22.0 ) {
							// Winkelbyte des Schusses speichern, da der Schuss vor zwei Frames abgeschossen
							// wurde, wird hier mit ping - 2 gerechnet:
							shot.winkelbyte = winkelbyte[ ( server_ping + 254 ) & 0xFF ];
							shot.dx					= Ship.shot_delta[ shot.winkelbyte ][0];
							shot.dy					= Ship.shot_delta[ shot.winkelbyte ][1];

							ship.getShotPrediction( tmpPoint, 40, shot.winkelbyte );

							shot.projected_frames = 40;
							shot.projected_x			= tmpPoint.x;
							shot.projected_y			= tmpPoint.y;

							shots_fired++;
							level_shots++;
						}
						else {
							shot.saucer = true;
						}
					}
					shots.add( shot );

					if ( shot.saucer ) {
						UFO.shots.add( shot );
					}
					else {
						Ship.addShot( shot );
					}
				}
				else if ( op == 0x06 && vz == 12 && dx != 0 && dy != 0 ) {
					// Schiff-Detektierung anhand bestimmter Helligkeit und Opcode
					if ( 0 == shipdetect ) {
						v1x = dx;
						v1y = dy;
					}
					else if ( 1 == shipdetect ) {
						// Richtungsvektor eintragen:
						ship.x2	= vx + ( v1x - dx );
						ship.y2	= vy + ( v1y - dy );

						if ( !ship.visible ) {
							ship.setVisible( vx, vy );
						}
						else {
							// Schiff war schon sichtbar: Ggf. Bewegungsvektoren (dx,dy) ermitteln
							ship.update( vx, vy );
						}
						ship_seen = true;
					}
					shipdetect++;
				}
			}
			else {
				switch ( op ) {
					case 0x0A:
						// LABS
						vy = ( a1 & 0x3FF ) - OFFSET_Y;
						vx = ( a2 & 0x3FF );
						vs = ( a2 >> 12 );
						break;

					case 0x0B:
						// HALT
						finished = true;
						break;

					case 0x0C:
						// JSRL
						String s = Text.map[a1];

						if ( textPending = ( null != s ) ) {
							if ( 0 == textbuf.length() ) {
								text_x = vx;
								text_y = vy;
							}
							textbuf.append(s);
						}
						else {
							switch ( a1 ) {
								case 0x852:
									// System.err.println( "Copyright" );
									break;

								case 0x880: // Explosion XXL (3)
								case 0x896: // Explosion XL  (2)
								case 0x8B5: // Explosion L   (1)
								case 0x8D0: // Explosion S   (0)
									if ( ufo.x == vx && ufo.y == vy && 0x8b5 == a1 && 11 == vs ) {
										ufo.exploding = true;
									}
									else if ( ufo.exploding && ufo.x == vx && ufo.y == vy ) {
										// Explosion des UFOs ignorieren, nicht in die Objektliste mit aufnehmen
										// ... ignore
									}
									else {
										vram_objects_new[vram_cnt++].setExplosion( vx, vy, vs, a1 );
									}
									break;

								case 0x8f3:
								case 0x8ff:
								case 0x90d:
								case 0x91a:
									vram_objects_new[vram_cnt++].setAsteroid( vx, vy, vs, a1 );
									break;

								case 0x929:
									if ( Debug.ufo ) {
										Log.info( String.format(
												"Frame#%5d: UFO 0x%03x at (%3d,%3d) [vs=%3d, vram_cnt=%2d]",
												frame_cnt, a1, vx, vy, vs, vram_cnt ) );
									}
									if ( !ufo.visible ) {
										ufo.setAppearance( vx, vy, vs, 0x929 );
										level_ufos++;
									}
									else {
										tmpPoint.x = vx;
										tmpPoint.y = vy;

										ufo.update( tmpPoint );
									}
									saucer_seen = true;
									break;

								case 0xA6D:
									// System.err.println( "RAUMSCHIFF" );
									ships_seen++;
									break;

								default:
									System.err.println( "c a1 = " + Integer.toHexString( a1 ) );
									break;
							}
						}
						break;

					case 0x0D:
						// RTSL
						finished = true;
						break;

					case 0x0E:
						// JMPL
						finished = true;
						break;

					case 0x0F:
						// SVEC
						break;
				}
			}
			if ( !textPending && textbuf.length() > 0 ) {
				// Score ' 2280': x,y = 100,876
				// High  '33700': x,y = 400,876
				final String text = textbuf.toString();

				if ( text_x == 100 && text_y == ( 876 - OFFSET_Y ) ) {
					int new_score = Integer.parseInt( text.replaceAll( " ", "" ) );
					updateScores( new_score );
				}
				else if ( text.startsWith( "IHR ERGEBNIS" ) || text.startsWith( "STARTKN0EPFE DRUECKEN" ) ) {
					startNewGame( false );
					gameover = true;
				}
				if ( withDisplay ) {
					texts.add( new Text( text, text_x, text_y ) );
				}
				textbuf.setLength(0);
			}
			textPending = false;
		}
		VectorObject.sync( asteroids, vram_objects, vram_objects_new, vram_cnt );

		curr_cnt_ast = asteroids.size();

		if ( !ship_seen ) {
			ship.visible = false;
		}
		if ( !saucer_seen ) {
			ufo.visible = false;
		}
		if ( ships_seen > ships_remain ) {
			ships_remain = ships_seen;
		}
		if ( gameover ) {
			send( KEY_START );
		}
	}
	
	public void sendName()
	{
		if ( !address.getHostName().equals( "asteroids.heise.de" ) ) {
			return;
		}
								// 12345678901234567890123456789012345678
		String name = "ctname...THE DISMAL DREGS OF DEFEAT   ";

		try {
			DatagramPacket packet = new DatagramPacket( name.getBytes(), name.length(), address, 1979 );
			socket.send( packet );
		}
		catch ( Exception e ) {
			System.err.println( "Exception: " + e );
			e.printStackTrace();
		}
	}
	
	public void start() throws SQLException
	{
		startNewGame( true );

		send( (byte)0 );
		sendName();

		boolean	fireDelay	= false;
		int			waitFrames= 0;
		byte		keys			= 0;

		boolean hyperspace		= false;
		boolean hyperspaceWait= false;
		boolean	log_invisible = false;
		boolean log_visible		= true;

		int			measure_shot_at			= 0;
		int			measure_shot_frames	= 0;
		int			measure_expected		= 0;
		boolean measure_finished		= false;

		int			wbRange			= 0;
		
		Target bestTarget[] = new Target[2];

		while ( true ) {
			keys = 0;

			if ( frame_cnt == FRAMES_5MIN || ( Debug.stopAtLevel5 && 5 == level ) ) {
				Log.info( String.format( "Frame#%5d: Spielstand: %5d", frame_cnt, getRealScore() ) );

				sql.setScore5min( getRealScore() );
				sql.setScore10min( 0 );

//				sql.setScore( frame_cnt,
//											getRealScore(),
//											shots_fired,
//											large_ast_hit,
//											medium_ast_hit,
//											small_ast_hit,
//											large_ufo_hit,
//											small_ufo_hit,
//											ships_remain,
//											warp_cnt );
//
//				finishing = true;
//
//				while ( finishing ) {
//					send( (byte)0 );
//				}
			}
			if ( !ship.visible && log_invisible ) {
				Log.info( String.format( "Frame#%5d: Ship invisible [wb=%3d]", frame_cnt, Ship.winkelbyte ) );

				log_invisible = false;
				log_visible		= true;
				hyperspaceWait= true;
			}
			if ( ship.visible && log_visible ) {
				Log.info( String.format( "Frame#%5d: Ship visible   [wb=%3d]", frame_cnt, Ship.winkelbyte ) );

				log_visible		= false;
				log_invisible	= true;
				hyperspaceWait= false;
			}
			if ( prev_cnt_ast == 0 && curr_cnt_ast > 0 ) {
				if ( 1 == ++level ) {
					score			= 0;
					scorewrap = 0;	// Zur Sicherheit abschalten..., automatische Erkennung luft nicht immer
					frame_cnt	= 1;	// Wettbewerbs-MAME zhlt das "Spieler 1" nicht mit
					warp_cnt	= 0;

					dumper_active = true;
				}
				level_part				= 0;
				level_startframe	= frame_cnt;
				level_startscore	= getRealScore();
				level_ufos				= 0;
				level_shots				= 0;
				level_key_fire		= 0;
				
				Log.info( String.format(
						"Frame#%5d: Start Level %2d [%2d Asteroiden, %5d Punkte] [wait=%3d]",
						frame_cnt, level, curr_cnt_ast, getRealScore(), level_waitframes ) );
			}
			if ( prev_cnt_ast > 0 && curr_cnt_ast == 0 ) {
				int nFrames = frame_cnt - level_startframe;
				int lScore	= getRealScore() - level_startscore;

				Log.info( String.format(
						"Frame#%5d: Ende  Level %2d [%4d Frames] [%5d Punkte = %3d/100f, %d UFOs, %2d[%3d] Shots]",
						frame_cnt, level, nFrames, lScore,
						(int)( 100.0 * (double)lScore / (double)nFrames ),
						level_ufos, level_shots, level_key_fire ) );

				level_waitframes = 165;
			}
			if ( level_waitframes > 0 ) {
				--level_waitframes;

				// Schuss tritt in f+2 aus, trifft nicht im letzten Frame (-1) und
				// soll noch 1-2 Frames Chance haben, einen A. zu treffen (-1)
				ship.getShotPrediction( tmpPoint, Shot.getLifeSpan(), getWinkelbyte() );
				MovingObject.normalize( tmpPoint );

				final int xDistLsq = MovingObject.NormalDistSq( tmpPoint, 10,     		tmpPoint.y ) / 8;
				final int xDistRsq = MovingObject.NormalDistSq( tmpPoint, MAX_X - 10, tmpPoint.y ) / 8;

				final int yDistBsq = MovingObject.NormalDistSq( tmpPoint, tmpPoint.x, 10 ) / 8;
				final int yDistTsq = MovingObject.NormalDistSq( tmpPoint, tmpPoint.x, MAX_Y - 10 ) / 8;
					
				boolean preshot_position_reached = ( level_waitframes >= 65 )
																				&& ( ( xDistLsq < 900 || xDistRsq < 900 )
																					|| ( yDistBsq < 900 || yDistTsq < 900 ) );

				if ( 100 == level_waitframes ) {
					measure_shot_at			= frame_cnt;
					measure_shot_frames	= 0;
					measure_finished		= false;
					measure_expected		= Shot.getLifeSpan();

					if ( Debug.shot_measure ) {
						Log.debug( String.format(
								"Frame#%5d: Measure shot [frame mod 4 = %d], lifespan expected = %2d",
								frame_cnt, frame_cnt % 4, Shot.getLifeSpan() ) );
					}
					keys |= KEY_FIRE;
				}
				else if ( !ufo.visible && preshot_position_reached ) {
					final int remaining_shot_lifespan = 2 + Shot.getLifeSpan() - level_waitframes;

					if ( remaining_shot_lifespan >= 3 && remaining_shot_lifespan <= 7 ) {
						if ( !fireDelay ) {
							if ( Debug.preShots ) {
								Log.debug( String.format(
										"Frame#%5d: PRESHOT [%2d:%d] X(%5d|%5d) Y(%5d|%5d)",
										frame_cnt, level_waitframes, remaining_shot_lifespan,
										xDistLsq, xDistRsq, yDistBsq, yDistTsq ) );
							}
							keys |= KEY_FIRE;
						}
						keys |= KEY_RIGHT;
					}
				}
				else if ( !ufo.visible) {
					final Point edges[] = new Point[] {
								new Point(     0,  ship.y ),
								new Point( MAX_X,  ship.y ),
								new Point( ship.x, 0 ),
								new Point( ship.x, MAX_Y )
					};
					double	dist	= 1000;
					Point		edge	= null;

					for ( int i = 0; i < edges.length; i++ ) {
						int dx = ship.x - edges[i].x;
						int dy = ship.y - edges[i].y;
						double edge_dist = Math.sqrt( dx * dx + dy * dy );

						if ( null == edge || edge_dist < dist ) {
							edge = edges[i];
							dist = edge_dist;
						}
					}
					double alpha			= Math.toDegrees( Math.atan2( edge.x - ship.x, edge.y - ship.y ) );
					double ship_alpha	= Ship.shot_alpha[ Ship.winkelbyte ];

					if ( level_waitframes > 80 || Math.abs( alpha - ship_alpha ) > 6.0 ) {
						keys |= KEY_RIGHT;
					}
				}
				if ( level_waitframes > 20 && level_waitframes < 40 ) {
					// Wenn ein Schuss zwischen Frame 20 und 40 verpufft ist, war es der Messschuss:
					if ( prev_shots.size() == 1 && !measure_finished ) {
						measure_finished		= true;
						measure_shot_frames = prev_shots.firstElement().frames;
						
						if ( measure_shot_frames >= 69 && measure_shot_frames <= 72 ) {
							Shot.setLifeSpanOffset( measure_shot_at % 4, measure_shot_frames );

							if ( measure_expected != measure_shot_frames ) {
								Log.debug( String.format(
										"Frame#%5d: [Level %2d] expected shot frames #%2d != measured frames #%2d",
										frame_cnt, level, measure_expected, measure_shot_frames ) );
							}
						}
						if ( Debug.shot_measure ) {
							Log.debug( String.format(
									"Frame#%5d: Measure shot at %5d [frame mod 4 = %d], frames=%2d",
									frame_cnt, measure_shot_at, measure_shot_at % 4, measure_shot_frames ) );
						}
					}
				}
			}
			if ( ship.visible ) {
				for ( Shot shot : UFO.shots ) {
					if ( shot.frames >= 4 ) {
						// Mgliche Kollision mit UFO-Schssen berechnen
						for ( int f = 4 + norm_latency; f >= 1; f-- ) {
							if ( ship.shotWillHit( shot, f ) ) {
								Log.warn( String.format( "Frame#%5d: hyperspace [UFO hit in %2d Frames]", frame_cnt, f ) );
								shot.killing	= true;
								hyperspace		= true;
								break;
							}
						}
					}
				}
			}
			targets.clear();

			if ( ufo.visible ) {
				targets.add( ufo );
			}
			int cnt_ast_large		= 0;
			int cnt_ast_medium	= 0;
			int cnt_ast_small		= 0;

			for ( Asteroid a : asteroids ) {
				switch( a.sizeType ) {
					case Asteroid.SIZE_LARGE  : cnt_ast_large++; break;
					case Asteroid.SIZE_MEDIUM : cnt_ast_medium++; break;
					case Asteroid.SIZE_SMALL  : cnt_ast_small++; break;
				}
				targets.add( a );
			}
			wbRange = Config.RotateRange[level_part];

			if ( 0 == level_part && cnt_ast_large <= 2 && asteroids.size() >= 15 ) {
				if ( Debug.level ) {
					Log.info( String.format(
							"Frame#%5d: [%3d] bergang zu Mittelspiel",
							frame_cnt, frame_cnt - level_startframe ) );
				}
				level_part++;
				wbRange = Config.RotateRange[level_part];
			}
			else if ( 1 == level_part && cnt_ast_medium <= 2 && asteroids.size() <= 10 ) {
				if ( Debug.level ) {
					Log.info( String.format(
							"Frame#%5d: [%3d] bergang zu Endspiel",
							frame_cnt, frame_cnt - level_startframe ) );
				}
				level_part++;
				wbRange = Config.RotateRange[level_part];
			}
			ship.checkShotHits( targets, frame_cnt - level_startframe );

			// Alle Targets, auch solche mit framewait>0, die auf Kollisionskurs innerhalb #Frames sind,
			// markieren und den nchsten innerhalb #n Frames, auf den noch nicht geschossen wurde, zurckgeben
			tCollision = ship.findCollisionTarget( targets, Config.MinCollisionFrame[level_part] );

			if ( ship.visible ) {
				for ( Target target : targets ) {
					if ( target.collision && target.collisionFrame <= 4 + norm_latency ) {
						Log.warn( String.format(
									"Frame#%5d: Asteroid hit in %2d Frames, dist=%5f",
									frame_cnt, target.collisionFrame, target.distance ) );
	
						hyperspace = true;
					}
				}
			}
			// Alle Targets mit framewait werden nun fr die weitere Betrachtung ignoriert.
			targets.clear();

			for ( Asteroid a : asteroids ) {
				if ( Config.Asteroid26Lock && ( asteroids.size() == 26 ) && ( a.framewait == 0 ) && ( a != tCollision ) && ( a.sizeType != Asteroid.SIZE_SMALL ) ) {
					// Falls schon 26 Asteroiden auf dem Bildschirm sind, werden alle Asteroiden
					// auer den kleinen und solchen auf Kollisionskurs fr Schsse deaktiviert
					continue;
				}
				if ( Config.AsteroidDistLock && a.distance > 150 && a.frames < 4 && a.framewait == 0 ) {
					// Asteroiden, die "weit" weg sind und noch keine genaue Distanz haben, werden ignoriert
					continue;
				}
				if ( 0 == a.framewait ) {
					targets.add( a );
				}
			}
			// Falls nur noch wenig Frames zu spielen sind, bleibt ein Asteroid stehen
			// und wir spielen nur noch UFO-Jger
			if ( ( frame_cnt > 17750 && frame_cnt < 17920 ) && targets.size() == 1 ) {
				Target aLast = targets.get(0);

				if ( !aLast.collision ) {
					targets.remove( aLast );
					aLast.framewait = 1;
				}
			}
			if ( ufo.visible && 0 == ufo.framewait ) {
				targets.add( ufo );
			}
			Target target = null;

			if ( null != tCollision ) {
				// Suche nach bester Drehrichtung nur fr die Asteroiden auf Kollisionskurs
				// Es wird ein temporrer Vektor genommen, damit die findTarget-Methode (siehe unten) unterwegs
				// trotzdem noch andere Ziele findet
				tmpTargets.clear();
				tmpTargets.add( tCollision );

				target = ship.markTargetReachability( bestTarget, tmpTargets, 45 );

				if ( Debug.collisions && null != target ) {
					Log.debug( String.format(
							"Frame#%5d: tCollision %2d Frames, L/R=%2d/%2d",
							frame_cnt, target.collisionFrame, target.leftMinHitInFramesKeys, target.rightMinHitInFramesKeys ) );
				}
			}
			if ( Config.MediumPreferAtEnd && ( 2 == level_part ) && ( null == target ) && ( cnt_ast_medium > 0 ) ) {
				// Beim bergang in das Endspiel zuerst auf die mittelgroen Asteroiden fokussieren:
				tmpTargets.clear();

				for ( Asteroid a : asteroids ) {
					if ( 0 == a.framewait && a.sizeType == Asteroid.SIZE_MEDIUM ) {
						tmpTargets.add( a );
					}
				}
				target = ship.markTargetReachability( bestTarget, tmpTargets, wbRange );
			}
			if ( null == target && targets.contains( ufo ) ) {
				if ( Config.LastAsteroidBeforeUFO && ufo.small && targets.size() == 2 ) {
					// Das kleine UFO wird temporr herausgenommen, um den Level schneller abzurumen
					targets.remove( ufo );
					target = ship.markTargetReachability( bestTarget, targets, wbRange );
					targets.add( ufo );
				}
				else if ( Config.SmallUFOfirst && ufo.small ) {
					// Sofort auf das kleine UFO halten
					tmpTargets.clear();
					tmpTargets.add( ufo );
					target = ship.markTargetReachability( bestTarget, tmpTargets, wbRange );
				}
				else if ( Config.LargeUFOLast && !ufo.small ) {
					// Groes UFO bis zum Levelende aufheben
					targets.remove( ufo );
					target = ship.markTargetReachability( bestTarget, targets, wbRange );
					targets.add( ufo );
				}
			}
			if ( null == target ) {
				target = ship.markTargetReachability( bestTarget, targets, wbRange );
			}
			if ( fireDelay ) {
				// Nicht gleich zum nchsten Target drehen, wenn nicht auf target geschossen werden kann
				bestTarget[1] = null;
			}
			if ( target != null && ship.visible ) {
				// In Richtung des nchsten Asteroiden bzw. des UFOs drehen
				keys |= ship.moveToTarget( target, bestTarget[1] );
			}
			if ( !fireDelay && ( Ship.shots.size() < 4 || ship.shotAvailable(2)) ) {
				// Nur schieen, wenn auch etwas innerhalb #Frames getroffen werden kann:
				Target dst = null;
				
				if ( null != target && 0 == target.leftMinHitInFramesKeys ) {
					dst = target;
				}
				else {
					// "en passant"-Ziel finden:
					for ( int f = 0; f <= Config.WaitFrames[level_part]; f++ ) {
						dst = ship.findTarget( targets, f );
						
						if ( null != dst ) {
							waitFrames = f;
	
							if ( waitFrames > 0 ) {
								if ( dst != target ) {
									Log.debug( String.format( "Frame#%5d: %d-wait Target, hitframes=%2d",
											frame_cnt, f, Math.min( dst.leftMinHitInFrames, dst.rightMinHitInFrames ) ) );
								}
								dst = null;
							}
							break;
						}
					}
				}
				if ( null != dst ) {
					if ( null == target ) {
						// kann vorkommen, wenn target z.B. innerhalb der Kollisionsasteroiden gesucht wurde,
						// dst aber innerhalb aller Targets
						target = dst;
					}
					// Target dst kann jetzt abgeschossen werden. Fr dst gilt:
					//     dst.left/rightMinHitInFramesKeys = 0
					// und dst.leftMinHitInFrames = dst.rightMinHitInFrames 
					final int shot_needed	= Math.min( target.leftMinHitInFramesKeys, target.rightMinHitInFramesKeys );

					boolean	shoot =	( dst == target )												// Immer auf das eigentliche Ziel schieen
												|| ( dst instanceof UFO )									// auf das fiese UFO immer schieen
												|| Ship.shots.size() <= 2									// es sind nur 2 Schsse oder weniger auf dem Schirm
												|| dst.leftMinHitInFrames < shot_needed		// En Passant-Target ist schneller abschiebar als Ziel erreichbar
												|| ship.shotAvailable( 2 + shot_needed );	// Es wird zwischenzeitlich ein anderer Schuss frei

					if ( dst != target && !(dst instanceof UFO) ) {
						if ( 1 == shot_needed ) {
							if ( Debug.shot_needed ) {
								Log.info( String.format(
										"Frame#%5d: shot_needed = 1, dst[%2d] != target[%2d]",
										frame_cnt,
										Math.min( dst.leftMinHitInFrames, dst.rightMinHitInFrames ),
										Math.min( target.leftMinHitInFrames, target.rightMinHitInFrames ) ) );
							}
						}
					}
					if ( shoot ) {
						if ( dst.shots_fired_at >= ship.getMaxShots( dst, 0 ) - 1 ) {
							// Der Schuss tritt (falls berhaupt) in 2 + latency Frames aus und wird dann berprft,
							// ob er auch wirklich trifft:
							dst.framewait = 2 + norm_latency;
						}
						keys |= KEY_FIRE;
					}
				}
			}
			else if ( Debug.shot_lock && !fireDelay && null != target && target.leftMinHitInFramesKeys == 0 ) {
				Log.debug( String.format(
						"Frame#%5d: Schusssperre, Wartezeit = %2d",
						frame_cnt, ship.nextShotAvailable() ) );
			}
			// Neu 28.05.08 21:16 Zu Levelbeginn nicht in ersten beiden Frames (ohne richtige Bewegungsinformationen) schieen:
			final int levelframe = frame_cnt - level_startframe;	// 0...

			if ( 1 == level && levelframe < 5 ) {
				// Winkelbyte ist u.U. nicht bekannt -> in gleiche Richtung drehen und ermitteln
				keys = KEY_LEFT;
			}
			else if ( levelframe < 2 ) {
				// Zwei Frames immer warten, ggf. schon drehen
				keys &= ~KEY_FIRE;
			}
			else if ( levelframe < 4 ) {
				if ( null != ( aNearest = ship.findNearestAsteroid( asteroids ) ) ) {
					if ( aNearest.distance > 100 ) {
						// Schon in Richtung des Targets drehen, aber noch nicht schieen
						keys &= ~KEY_FIRE;
					}
				}
			}
			if ( hyperspace ) {
				hyperspace = false;

				keys |= KEY_HYPERSPACE;
				log_invisible = true;

				warp_cnt++;

				Log.info( String.format( "Frame#%5d: hyperspace [wb=%3d]", frame_cnt, Ship.winkelbyte ) );
			}
			if ( 0 != ( keys & KEY_HYPERSPACE ) || waitFrames > 0 ) {
				keys &= ~KEY_LEFT;
				keys &= ~KEY_RIGHT;

				if ( waitFrames > 0 ) waitFrames--;
			}
			if ( hyperspaceWait ) {
				// Keine Tasten mehr drcken, wenn max. Framerate berschritten und Spiel
				// beendet werden soll oder das Schiff sich noch im "Hyperspace" befindet
				keys = 0;
			}
			if ( 0 != ( keys & KEY_FIRE ) ) {
				fireDelay = true;		// Einen Frame warten, bis wieder geschossen werden kann
				level_key_fire++;
			}
			else {
				fireDelay = false;
			}
			send( keys );

			if ( withDisplay ) {
				if ( frame_cnt % FRAME_INTERVAL == 0 ) {
					frame.repaint();
				}
			}
		}
	}

	public static void main( String arg[] )
	{
		Config.readConfig();

		for ( int i = 0; i < arg.length; i++ ) {
			if ( arg[i].equals( "-mysql" ) ) {
				withMySQL = true;
			}
			else if ( arg[i].equals( "-log4j" ) ) {
				withLog4j = true;
			}
			else if ( arg[i].equals( "-display" ) ) {
				withDisplay = true;
			}
			else if ( arg[i].equals( "-dumper" ) ) {
				withDumper = true;
			}
			else if ( arg[i].equals( "-latency1" ) ) {
				withLatency1 = true;
				norm_latency = 1;
			}
			else if ( arg[i].equals( "-normlatency" ) ) {
				i++;
				norm_latency = Integer.valueOf( arg[i] ).intValue();
			}
			else if ( !arg[i].startsWith( "-" ) ) {
				AsteroidApp client = new AsteroidApp( arg[i] );
				
				try {
					client.start();
				}
				catch( SQLException e ) {
					System.err.println( "SQL Exception: " + e );
				}
				System.exit(0);
			}
		}
	}
}
