/*
 *
 * Asteroids Player
 *
 * for the c't Competition 2008 (Creativ '08) 
 *
 * Copyright 2008, Volker Raum, Erlangen, Germany
 *
 */
package de.volkerraum.asteroids.player;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Observable;
import java.util.Observer;

import de.volkerraum.asteroids.HotShot;
import de.volkerraum.asteroids.model.AsteroidsModel;
import de.volkerraum.asteroids.model.BaseObject;
import de.volkerraum.asteroids.model.Command;
import de.volkerraum.asteroids.model.Helper;
import de.volkerraum.asteroids.model.Saucer;
import de.volkerraum.asteroids.model.Ship;
import de.volkerraum.asteroids.model.Target;

/**
 * @author Volker Raum (C) 2007
 */
public class Player implements Observer
{
   AsteroidsModel theModel;
   Command command = new Command();

   double lastScore = 0;
   int huntSaucerThreshold = 1; // Asteroids
   // until
   // we
   // hunt for the saucer

   int idleShotIntervall = 0;

   boolean shotLastTime = false;

   DecimalFormat df = null;

   double lastAngle = 0;
   double shotAngleMean = 0;
   double shotAngleMeanCount = 0;

   int lastShotsCount = 1000;

   Target nextTarget = null;

   int angleByte = 0;
   long analysedFrameCount = 0;

   boolean first = true;

   int asteroidsCountWhenShot = 0;

   int thrustCounter = 0;

   long maxRecursionCount = 0;

   public Player(AsteroidsModel model)
   {
      DecimalFormatSymbols symbols = DecimalFormatSymbols.getInstance();
      symbols.setDecimalSeparator('.');
      df = new DecimalFormat("0.000", symbols);

      theModel = model;
      theModel.addObserver(this);
      // Send a command so the Asteroid server knows us.
      System.err.println("FIRE FOR START");
      theModel.sendCommand(command.clearCommand().addAction(Command.ACTION_FIRE));
   }

   public void kill()
   {
   }

   public void update(Observable o, Object arg)
   {
      Integer updateMSG = (Integer) arg;

      analysedFrameCount++;
      if (updateMSG.equals(AsteroidsModel.MSG_GAME_STARTED))
      {
         return;
      }

      if (updateMSG.equals(AsteroidsModel.MSG_GAME_FINISHED))
      {
         System.err.println("END GAME");

         if (HotShot.singleGameWanted)
         {
            try
            {
               Thread.sleep(5000);
            }
            catch (InterruptedException e)
            {
            }
            System.exit(0);
         }

      }
      if (updateMSG.equals(AsteroidsModel.MSG_NEW_ROUND))
      {
         thrustCounter = 0;
      }

      command.clearCommand();
      Ship ship = theModel.getTheShip();
      Saucer saucer = theModel.getTheSaucer();

      if (!ship.present)
      {
         theModel.sendCommand(command);
         return;
      }
      checkStrategicMovementForCorner(ship, saucer);

      boolean doEvasiveAction = false;

      int stepsToImpact = Helper.calcStepsToShotImpact(theModel.getIdentifiedShots(), ship);
      if (stepsToImpact > 1 && stepsToImpact < 10 && theModel.getAsteroidCount() > 0) // >1 is a safetybecause the algorithm may deliver 1 => false alarm. Only jump if there are Asteroids around.
         // if we are alone with the saucer let the shot hit us => IS FASTER....
         doEvasiveAction = true;

      if (doEvasiveAction) // DAMN SAUCER !!!
      {
         nextTarget = null;
         command.addAction(Command.ACTION_HYPERSPACE);
         theModel.sendCommand(command);

         return;
      }

      double dist = Helper.getClosestDistanceToAnyObject(theModel.getAllObjectsInScene());
      int minimumImpact = Helper.getFastestImpact(theModel.getAllObjectsInScene());

      if (dist < AsteroidsModel.COLLISION_SAFETY || minimumImpact < 3)
      {
         nextTarget = null;

         command.addAction(Command.ACTION_HYPERSPACE);
         theModel.sendCommand(command);
         return;
      }

      if (asteroidsCountWhenShot != theModel.getAsteroidCount())
      {
         asteroidsCountWhenShot = theModel.getAsteroidCount();
         // nextTarget = null;
      }

      // Correction of the intercept Angle (Fallback Savety)
      if (nextTarget != null)
      {

         if (nextTarget.targetRef.interceptPossible && Math.abs(nextTarget.targetRef.interceptAngleByte - nextTarget.angleByte) < 7)
         {
            nextTarget.angleByte = nextTarget.targetRef.interceptAngleByte;
            nextTarget.angle = Helper.winkelByteToAngle.get(nextTarget.angleByte);

         }
         if (nextTarget.targetRef.interceptPossible && Math.abs(nextTarget.targetRef.interceptAngleByte - nextTarget.angleByte) > 6)
         {
            nextTarget = null;
         }

      }

      BaseObject potentialThreat = Helper.calcBiggestThreat(theModel.getAllObjectsInScene());
      if (potentialThreat != null)
      {
         // NO target => take threat
         if (nextTarget == null && potentialThreat.interceptPossible)
         {
            nextTarget = new Target(potentialThreat, potentialThreat.impactFrames);

         }
         else if (nextTarget != null && potentialThreat.interceptPossible)
         {

            if (!nextTarget.isThreat)
            {
               // target is NOT a threat => take threat
               nextTarget = new Target(potentialThreat, potentialThreat.impactFrames);
            }
            else if (nextTarget.impactFrames > potentialThreat.impactFrames)
            {
               // target is threat but impact is not as imminent as other =>
               // take new Threat
               nextTarget = new Target(potentialThreat, potentialThreat.impactFrames);
            }

         }
      }

      if (nextTarget == null)
      {

         BaseObject target = Helper.calcBestFreeTargetByReachInterceptAngleAndTargetDesity(theModel.getAllObjectsInScene(), ship, theModel);

         if (target != null)
         {
            nextTarget = new Target(target);

         }
      }
      if (nextTarget == null)
      {
         BaseObject target = Helper.calcBestFreeTargetByInterceptFrames(theModel.getAllObjectsInScene(), ship, theModel);
         if (target != null)
         {
            nextTarget = new Target(target);
         }
      }

      if (nextTarget == null)
      {
         BaseObject target = Helper.calcBestFreeTargetByInterceptAngle(theModel.getAllObjectsInScene(), ship, AsteroidsModel.SHOT_REACH, theModel);
         if (target != null)
         {
            nextTarget = new Target(target);
         }
      }

      // Take a look in the future
      int futureAngleByte = ship.angleByte;

      if (theModel.isLeftCommand())
      {
         // A Left Command was issued in the last
         // frame => We already may fire.
         futureAngleByte += 3;

      }

      if (theModel.isRightCommand())
      {
         // A Right Command was issued in the
         // last frame => We already may fire.
         futureAngleByte -= 3;
      }

      if (futureAngleByte < 0)
         futureAngleByte += 256;
      if (futureAngleByte > 255)
         futureAngleByte -= 256;

      double futureAngle = Helper.getAngleForAngleByte(futureAngleByte);

      if (nextTarget != null)
      {

         if (nextTarget.angleByte == futureAngleByte)
         {

            if (lastShotsCount > 2)
            {
               nextTarget.targetRef.shotsFiredAt++;
               nextTarget.targetRef.shotImpactFrames = nextTarget.interceptFrames;
               command.addAction(Command.ACTION_FIRE);
               asteroidsCountWhenShot = theModel.getAsteroidCount();

               nextTarget = null;
               lastShotsCount = 0;
            }
         }
         else if (Math.abs(nextTarget.angleByte - futureAngleByte) < 3)
         {
            // SYNC PROBLEM ANGLE COUNT
            command.addAction(Command.ACTION_FIRE);
            asteroidsCountWhenShot = theModel.getAsteroidCount();

            nextTarget = null;
         }
         else
         {
            BaseObject foundRandom = Helper.isInterceptPointInDirection(theModel.getAllObjectsInScene(), futureAngleByte);
            if (foundRandom != null && foundRandom.interceptPossible && foundRandom != nextTarget.targetRef)
            {
               foundRandom.shotsFiredAt++;
               foundRandom.shotImpactFrames = foundRandom.interceptFrames;
               asteroidsCountWhenShot = theModel.getAsteroidCount();

               command.addAction(Command.ACTION_FIRE);
            }
         }
      }

      // Rotation
      if (nextTarget != null)
      {

         double diff = Helper.calcAngleDifference(futureAngle, nextTarget.angle);

         if (diff < 0)
         {
            command.addAction(Command.ACTION_RIGHT);
         }
         else if (diff > 0)
         {
            command.addAction(Command.ACTION_LEFT);
         }
      }
      else
      {
         BaseObject defaultObject = Helper.calcClosestFreeTargetByRange(theModel.getAllObjectsInScene(), AsteroidsModel.SHOT_REACH);
         if (defaultObject != null)
         {
            double diff = Helper.calcAngleDifference(futureAngle, defaultObject.angleToShip);

            if (diff < 0)
            {
               command.addAction(Command.ACTION_RIGHT);
            }
            else if (diff > 0)
            {
               command.addAction(Command.ACTION_LEFT);
            }
         }
      }
      lastShotsCount++;

      if (command.shootActionGiven() && shotLastTime)
      {
         // sending fire constantly simply blocks the gun => release it
         command.removeAction(Command.ACTION_FIRE);
      }
      shotLastTime = command.shootActionGiven();

      theModel.sendCommand(command);

   }

   private void checkStrategicMovementForCorner(Ship ship, Saucer saucer)
   {
      if (!saucer.present && theModel.getAsteroidCount() == 0)
      {
         int preferredPositionX = 0;
         int preferredPositionY = 0;
         double distanceX = 100;
         double distanceY = 100;

         if (ship.x > 512)
            preferredPositionX = 824;
         else
            preferredPositionX = 200;

         if (ship.y > 384)
            preferredPositionY = 568;
         else
            preferredPositionY = 200;

         double distanceFromDestinationX = ship.x - preferredPositionX;
         double distanceFromDestinationY = ship.y - preferredPositionY;
         double distanceFromDestination = Math.sqrt(distanceFromDestinationX * distanceFromDestinationX + distanceFromDestinationY * distanceFromDestinationY);

         if (Math.abs(distanceFromDestinationX) > distanceX || Math.abs(distanceFromDestinationY) > distanceY)
         {
            double angleToShip = Math.atan2(distanceFromDestinationY / distanceFromDestination, distanceFromDestinationX / distanceFromDestination);

            double deltaAngle = Math.abs(Math.abs(ship.angleOfDirection - angleToShip) - 3.1415);

            if (deltaAngle < 0.3 && thrustCounter < 15 && Helper.canMoveToCurrentDirection(theModel.getAllObjectsInScene(), ship))
            {
               command.addAction(Command.ACTION_THRUST);
               thrustCounter++;
            }

            // No Asteroid around => steer the ship explicitly

            double threatDx = preferredPositionX - ship.x;
            double threatDy = preferredPositionY - ship.y;

            if (ship.dirX * threatDy - ship.dirY * threatDx > 0)
               command.addAction(Command.ACTION_LEFT);
            else
               command.addAction(Command.ACTION_RIGHT);
         }
      }
   }

   public String getObjectDescription(BaseObject target)
   {
      int i = -1;
      String result = "";
      for (BaseObject currObject : theModel.getAllObjectsInScene())
      {
         i++;
         if (currObject == target)
         {
            result = currObject.getClass().getSimpleName() + " # " + i + " - " + target.x + "/" + target.y;
         }
      }
      return result;
   }

   public BaseObject getNextTargetedObject()
   {
      if (nextTarget != null)
         return nextTarget.targetRef;
      else
         return null;
   }

   public Target getNextTarget()
   {
      return nextTarget;
   }

}
