package de.dkn.asteroids;

import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import de.caff.asteroid.Bullet;
import de.caff.asteroid.FrameInfo;
import de.caff.asteroid.FramePreparer;
import de.caff.asteroid.GameData;
import de.caff.asteroid.MovingGameObject;
import de.caff.asteroid.SpaceShip;
import de.caff.asteroid.Ufo;
import de.caff.asteroid.FrameInfo.Direction;

/**
 * <p>
 * Trifft die Entscheidung, was der DknAsteroidsPlayer tun soll. Die Entscheidung und Informationen
 * darber werden in einer Instanz von DknDecisionData, die wiederum an das aktuelle
 * FrameInfo-Objekt gehngt wird, gehalten. Damit kann alles im AnalysisDisplay angezeigt werden,
 * was das Debugging und Verbessern des Players erleichtern sollte.
 * </p>
 * <p>
 * HINWEIS: Die Domain de.dkn gehrt nicht mir, aber ich benutze das Krzel dkn schon sehr lange und
 * meine, bzw. unsere Domain de.familie-damken ist nicht Java-fhig und damit ist de.dkn aus meiner
 * Sicht genauso geeignet wie jede abweichende Schreibweise von de.familie-damken.
 * </p>
 * <p>
 * (c) 2008, by Uwe Damken ... basierend auf einem Framework von Rammi (rammi@caff.de)
 * </p>
 */
public class DknDecisionPreparer implements FramePreparer {

	/** Anzahl der Frames, die eine Bullet mindestens luft */
	private static final int MIN_BULLET_LIFETIME = 65;

	/** Wieviel Frames im voraus soll das Schiff vor dem nchsten Target geschtzt werden? */
	private static final int FRAMES_AHEAD_OBJECT_HYPERSPACE = 4;
	// Wenn das Schiff "im Verschwinden" noch getroffen wird, zerschellt es beim Wiedereinsatzort!

	/** Bis zu welcher Entfernung ist ein Schieen auf mittlere und groe Asteroiden zu gefhrlich? */
	private static final int FRAMES_AHEAD_TOO_DANGEROUS_TOO_FIRE = 3 * FRAMES_AHEAD_OBJECT_HYPERSPACE;

	/** Bis wieviele Frames nach dem erwarteten Treffer soll mit dem Beschuss gewartet werden? */
	private static final int DEATH_TOLERANCE_FRAMES = 1;

	/** Bis zu wieviel Frames bis zum Treffen auf das Schiff soll ein Target als gefhrlich gelten? */
	private static final int MAX_FRAMES_AHEAD_DANGEROUS_TARGET = 256 / 2 + FRAMES_AHEAD_OBJECT_HYPERSPACE;
	// Max HitInFrames ist MAX_FASTEST_SHOOTING_DIRECTION_DELTA + MIN_BULLET_LIFETIME = 200

	/** Wieviel Drehungbewegungen sollen links- und rechtsrum fr findFirst berechnet werden? */
	private static final int MAX_FIRST_SHOOTING_DIRECTION_DELTA = 256 / 2 + 2;
	// Einmal gab es ein UFO, das nur mit mehr als einer Drehung getroffen werden konnte

	/** Wieviel Drehungbewegungen sollen links- und rechtsrum fr findLastest berechnet werden? */
	private static final int MAX_FASTEST_SHOOTING_DIRECTION_DELTA = 256 / 3 / 2 + 2;

	/** Wieweit ist eine Schiffskugel mindestens von der "Schiffmitte" entfernt? */
	private static final int MIN_SHIP_BULLET_SQUARED_SHIP_DISTANCE = 300;

	/** Wieweit ist eine Schiffskugel hchstens von der "Schiffmitte" entfernt? */
	private static final int MAX_SHIP_BULLET_SQUARED_SHIP_DISTANCE = 500;

	/** "Kein Treffer" braucht nur einmal instanziiert zu werden (static nicht fr Inner Classes) */
	private final TargetHitResult THR_NO_HIT = new TargetHitResult(0, null, null);

	/** Switch between shooting and not shooting. */
	private boolean firedLastTime = false;

	/** Aktuelle Entscheidungsdaten */
	private DknDecisionData decision;

	/** Mit dieser virtuellen Bullet werden alle mglichen Treffer berechnet */
	private Bullet possibleBullet = new Bullet(0, 0, 0);

	/** Die nchste freie Id fr Bullets (Asteroiden und UFOs siehe DknVelocityPreparer) */
	private int nextId = 0;

	/** Wieviele Schiffskugeln drfen gleichzeitig unterwegs sein? */
	private static final int MAX_CONCURRENT_SHIP_BULLETS = 4;

	/**
	 * Prepare the frame(s).
	 */
	public void prepareFrames(LinkedList<FrameInfo> frameInfos) {
		if (frameInfos.size() >= 1) {
			FrameInfo frame = frameInfos.getLast();
			DknDecisionData prevDecision = (frameInfos.size() < 2) ? null : frameInfos.get(
					frameInfos.size() - 2).getDecisionData();
			decision = frame.getDecisionData();
			if (prevDecision != null) {
				decision.inherit(prevDecision);
			}
			DknStatistics.getInstance().setScore(frame.getScore());
			final SpaceShip ship = frame.getSpaceShip();
			if (ship == null) {
				// Hyperspace wurde erfolgreich ausgefhrt oder das Schiff ist sonstwie weg
				decision.hyperInProgress = false;
			} else {
				correctFailedShotIfNeededAndIdentifyShipBullets(frameInfos, frame, ship);
				setShipRelatedValuesInPotentiallyDangerousObjects(frame, ship); // inkl. Bullets
				cleanupShotTargets(frame);
				splitPotentialTargetsIntoGroups(frame, prevDecision); // Asteroiden & UFOs aufteilen
				if (escapeIfNecessary(frame, prevDecision)) {
					return; // Nach einer Flucht in den Hyperraum bleibt nichts mehr zu tun
				}
				if (prevDecision != null && prevDecision.isHitExpectedInNextFrame()) {
					// Beim letzten Mal wurde mit hitInFrames == 1 gedreht, so dass jetzt ein
					// Treffer zu erwarten ist. Ob es das gewnschte Target trifft oder nicht, weil
					// ein anderes dazwischen steht, ist unerheblich, vielleicht muss der Schussweg
					// ja freigerumt werden. Auf jeden Fall soll das korrekte Target als
					// "angeschossen" markiert werden.
					fireIfAnyTargetWillBeHit(frame, ship, decision.allNotShotTargets, 0);
				}
				boolean turned = false;
				if (!decision.dangerousTargets.isEmpty()) {
					turned = turnToHitAndFireLimited(frame, ship, decision.dangerousTargets, true);
				}
				if (!turned && !decision.highPrioTargets.isEmpty()) {
					turned = turnToHitAndFireLimited(frame, ship, decision.highPrioTargets, true);
				}
				if (!turned) {
					// Ungebremst schieen, Treffer aus der lowPrioTargets-Liste entfernen
					if (fireIfAnyTargetWillBeHit(frame, ship, decision.allNotShotTargets, 0)) {
						MovingGameObject shotTarget = decision.fireThr.getShotTargetInfo()
								.getShotTarget();
						decision.lowPrioTargets.remove(shotTarget);
					}
					// Trotzdem auch noch zum nchstbesten hindrehen
					turnToHit(frame, ship, decision.lowPrioTargets, true);
				}
			}
		}
	}

	/**
	 * Entfernt das Target des letzten Schusses aus decision.shotTargets, wenn der Schuss nicht der
	 * erwartete ist, um es zum erneuten Beschuss freizugeben. Kugeln, die wie erwartet kommen,
	 * werden auerdem durch Vergabe einer Id als Schiffskugeln markiert.
	 */
	private void correctFailedShotIfNeededAndIdentifyShipBullets(LinkedList<FrameInfo> frameInfos,
			FrameInfo frame, SpaceShip ship) {
		ExpectedBulletInfo ebi = null;
		// Die fr diesen Frame erwartete Bullet sollte in der Entscheidung des vorletzten Frames
		// (-3) oder - bei einem Frame-Drop - des letzten Frames (-2) enthalten ssein.
		for (int i = frameInfos.size() - 2; i >= 0 && i >= frameInfos.size() - 3 && ebi == null; i--) {
			FrameInfo fi = frameInfos.get(i);
			if (fi.getIndex() == frame.getIndex() - 2) {
				ebi = fi.getDecisionData().expectedBulletInfo;
			}
		}
		if (ebi != null) {
			// Ein Schuss erscheint im bernchsten Frame und sollte dann stimmen ...
			boolean expectedBulletOccuredExactly = false;
			boolean expectedBulletOccuredDeviant = false;
			for (Bullet bullet : frame.getBullets()) {
				if (bullet.getVelocityX() == 0.0 && bullet.getVelocityY() == 0.0) { // Neue Bullet?
					int distanceSq = (int) bullet.getLocation().distanceSq(ship.getLocation());
					if (MIN_SHIP_BULLET_SQUARED_SHIP_DISTANCE <= distanceSq
							&& distanceSq <= MAX_SHIP_BULLET_SQUARED_SHIP_DISTANCE) {
						expectedBulletOccuredDeviant = true;
						bullet.setIdentity(nextId++);
					}
					if (bullet.getLocation().equals(ebi.getExpectedLocation())) {
						expectedBulletOccuredExactly = true;
						expectedBulletOccuredDeviant = false;
						break;
					}
				}
			}
			if (!expectedBulletOccuredExactly) { // ... oder der Schuss erscheint nie mehr
				decision.shotTargets.remove(ebi.getTargetIdentity());
				// System.out.println(frame.getIndex() + "-" + frame.getReceiveTime() + ": "
				// + "Shoot failed: " + ebi.toString());
			}
			DknStatistics.getInstance().incrementBulletStatistics(expectedBulletOccuredExactly,
					expectedBulletOccuredDeviant);
		}
	}

	/**
	 * Setzt alle schiffsbezogenen Werte der potenziell gefhrlichen Objekte, d.h. quadrierte
	 * Entfernung zum Schiff, Frames bis zum Auftreffen auf das Schiff und ob ein Objekt fr das
	 * Schiff gefhrlich ist, und das fr Asteroiden, UFOs und Bullets.
	 */
	private void setShipRelatedValuesInPotentiallyDangerousObjects(FrameInfo frame, SpaceShip ship) {
		for (MovingGameObject o : frame.getPotentiallyDangerousObjects()) {
			o.setShipRelatedValues(ship, MAX_FRAMES_AHEAD_DANGEROUS_TARGET);
		}
	}

	/**
	 * Die "angeschossenen" Targets werden aufgerumt, indem sie umkopiert werden. Beim Umkopieren
	 * entfallen Targets, die nicht mehr existieren, die wohl doch nicht getroffen wurden und Ufos,
	 * die Richtung gendert haben und daher wohl nicht getroffen werden knnen.
	 */
	private void cleanupShotTargets(FrameInfo frame) {
		Map<Integer, ShotTargetInfo> newShotTargets = new HashMap<Integer, ShotTargetInfo>();
		for (MovingGameObject target : frame.getPotentialTargets()) {
			Integer id = target.getIdentity();
			ShotTargetInfo sti = decision.shotTargets.get(id);
			if (sti != null && !sti.targetShouldHaveBeenDeath(frame)
					&& !sti.targetIsUfoAndHasChangedDirection(target)) {
				newShotTargets.put(id, sti);
			}
		}
		decision.shotTargets = newShotTargets; // "mark and sweep" ist beendet
	}

	/**
	 * Teilt alle nicht "angeschossenen" Targets in zwei Gruppen auf, gefhrliche und ungefhrliche
	 * potenzielle Ziel. Bei Gruppen sind sortiert nach der quadrierten Entfernung zum Schiff. Wenn
	 * mehrere Targets in einer Schusslinie stehen, kann ohnehin nur das nchste Target getroffen
	 * werden. Es machte also keinen Sinn, auf ein "verdecktes" Target zu schieen.
	 */
	private void splitPotentialTargetsIntoGroups(FrameInfo frame, DknDecisionData prevDecision) {
		decision.dangerousTargets = new ArrayList<MovingGameObject>();
		decision.highPrioTargets = new ArrayList<MovingGameObject>();
		decision.lowPrioTargets = new ArrayList<MovingGameObject>();
		decision.allNotShotTargets = new ArrayList<MovingGameObject>();
		for (MovingGameObject target : frame.getPotentialTargets()) {
			if (target.getIdentity() != null
					&& decision.shotTargets.get(target.getIdentity()) == null) {
				if (target.isDangerous()) {
					decision.dangerousTargets.add(target);
				} else if (target instanceof Ufo) {
					decision.highPrioTargets.add(target);
				} else {
					decision.lowPrioTargets.add(target);
				}
				decision.allNotShotTargets.add(target);
			}
		}
		Collections.sort(decision.dangerousTargets, new BySquaredShipDistance());
		Collections.sort(decision.highPrioTargets, new BySquaredShipDistance());
		Collections.sort(decision.lowPrioTargets, new BySquaredShipDistance());
		Collections.sort(decision.allNotShotTargets, new BySquaredShipDistance());
		// log(frame.getIndex() + ": potentialTargets=" + frame.getPotentialTargets().toString());
		// log(frame.getIndex() + ": dangerousTargets=" + decision.dangerousTargets.toString());
		// log(frame.getIndex() + ": otherPotentialTargets="
		// + decision.lowPrioTargets.toString());
	}

	/**
	 * Gibt es irgendein potenziell gefhrliches Objekt, das das Schiff in zwei Frames treffen wird,
	 * dann entschwinde in den Hyperspace und gib in dem Fall true zurck.
	 */
	private boolean escapeIfNecessary(FrameInfo frame, DknDecisionData prevDecision) {
		for (MovingGameObject o : frame.getPotentiallyDangerousObjects()) {
			if (o.getHitShipInFrames() > 0
					&& o.getHitShipInFrames() <= FRAMES_AHEAD_OBJECT_HYPERSPACE) {
				for (MovingGameObject target : frame.getPotentialTargets()) {
					target.setDangerous(false);
				}
				if (prevDecision == null || !prevDecision.hyperInProgress) {
					// Zweimal Hyperspace fhren zur Sprengung, aber das dangerous-Flag muss
					// jedes Mal zurckgenommen werden, sonst ist es am Ende wieder an
					decision.hyper = true;
					decision.hyperInProgress = true;
				}
				break;
			}
		}
		return decision.isHyper();
	}

	/**
	 * Dreht zum erstbesten (findFirst) oder am schnellsten zu treffenden Target (findFastest) und
	 * schiet, wenn gedreht wurde, auf alle bisher nicht angeschossenen Targets, wenn sie vor der
	 * Flinte stehen und der Schuss krzer luft als die Drehzeit zum gewnschten Objekt.
	 */
	private boolean turnToHitAndFireLimited(FrameInfo frame, final SpaceShip ship,
			List<MovingGameObject> turnToTargets, boolean findFirst) {
		TargetHitResult thr = turnToHit(frame, ship, turnToTargets, findFirst);
		// FIXME -dkn- Eventuell schiee ich das, wo ich hier hindrehe, gleich noch ab und
		// verschenke damit eine Schussdrehung. Aber wie soll ich das verhindern, ich brauche erst
		// die Drehung fr die "Schusshemmung" ... hmmm
		int turns = thr.getTurns();
		boolean turned = turns > 0;
		if (turns == 1) {
			decision.hitExpectedInNextFrame = true;
		} else if (turned) {
			// Wenn noch ein Schuss zustzlich "frei" ist, kann ich unlimiert schieen, sonst
			// muss ich den verbleibenden restriktiv verweden und limitiert schieen.
			if (frame.getShipBulletCount() < MAX_CONCURRENT_SHIP_BULLETS - 1) {
				fireIfAnyTargetWillBeHit(frame, ship, decision.allNotShotTargets, 0);
			} else {
				fireIfAnyTargetWillBeHit(frame, ship, decision.allNotShotTargets, turns);
			}
		}
		return turned;
	}

	/**
	 * Prfen, ob ein Schuss bei der erwarteten Schussrichtung ein Target treffen wrde. Wenn ja,
	 * schieen und das Target als angeschossen eintragen und true zurckliefern. Wenn hitInFrames
	 * bis zum Treffer (inklusive Puffer) die maxHitFrames berschreitet, wird nicht geschossen, um
	 * nicht zu einem gefhrlichen Target gedreht zu haben, um dann zu sehen, dass keine
	 * Schussmglichkeit besteht, weil schon vier (weniger wichtige) Bullets unterwegs sind.
	 */
	private boolean fireIfAnyTargetWillBeHit(FrameInfo frame, final SpaceShip ship,
			List<MovingGameObject> targets, int maxHitInFrames) {
		boolean firedThisTime = false; // Vom Nicht-Schuss ausgehen
		// Es kann immer nur ein ums andere Mal geschossen werden und es knnen immer nur vier
		// eigene Kugeln gleichzeitig unterwegs sein
		if (!firedLastTime) { // && frame.getShipBulletCount() < MAX_CONCURRENT_SHIP_BULLETS) {
			TargetHitResult thr = findFirstTargetHit(frame, ship, 0, 0, targets);
			// TODO -dkn- Tuning mglich, wenn Begrenzung weiter unten stattfindet
			if (thr.getHitInFrames() > 0
					&& (maxHitInFrames == 0 || thr.getHitInFrames() < maxHitInFrames)) {
				decision.fireThr = thr;
				decision.expectedBulletInfo = thr.getExpectedBulletInfo();
				decision.shotTargets.put(thr.getShotTargetInfo().getTargetIdentity(), thr
						.getShotTargetInfo());
				decision.fire = true;
				firedThisTime = true;
			}
		}
		return firedLastTime = firedThisTime; // Dieses Mal ist bald das letzte mal
	}

	/**
	 * Dreht in die Richtung, die am ehesten einen Treffer auf eins der bergebenen Targets
	 * ermglicht (findFirst) oder auf das am schnellsten (Drehung + Schusslauf) zu treffenden
	 * Target (findFastest) und gibt die bentigte Anzahl von ShootingDir-Turns zurck.
	 */
	private TargetHitResult turnToHit(FrameInfo frame, final SpaceShip ship,
			List<MovingGameObject> targets, boolean findFirst) {
		TargetHitResult thr;
		if (findFirst) {
			thr = findFirstTargetHit(frame, ship, 1, MAX_FIRST_SHOOTING_DIRECTION_DELTA, targets);
		} else {
			thr = findFastestTargetHit(frame, ship, 1, MAX_FASTEST_SHOOTING_DIRECTION_DELTA, targets);
		}
		if (thr.getHitInFrames() > 0 && thr.getVectoredTurns() > 0) {
			decision.turnThr = thr;
			decision.left = true;
			frame.turnNextShootingDirectionLeft();
			return thr;
		}
		if (thr.getHitInFrames() > 0 && thr.getVectoredTurns() < 0) {
			decision.turnThr = thr;
			decision.right = true;
			frame.turnNextShootingDirectionRight();
			return thr;
		}
		return THR_NO_HIT;
	}

	/**
	 * Prfen, in welcher Schiffsrichtung (Angabe ber Anfang und ein von-bis-+/--Delta) ein Target
	 * mit mglichst geringer Entfernung vom Schiff getroffen werden kann, d.h. es wird erst ber
	 * die nach Entfernung sortierten Targets und dann ber die Schussrichtungen iteriert. Wenn ja,
	 * die Anzahl der Frames bis zum Treffer und Informationen ber die "genutzte" Bullet und das
	 * "getroffene" Target in Form von ExpectedBulletInfo und ShotTargetInfo zurckliefern.
	 */
	private TargetHitResult findFastestTargetHit(FrameInfo frame, final SpaceShip ship,
			int fromShootDirDelta, int toShootDirDelta, List<MovingGameObject> targets) {
		TargetHitResult fastestThr = null;
		for (MovingGameObject target : targets) {
			for (int i = -toShootDirDelta; i <= toShootDirDelta; i++) { // halb rum
				if (i <= -fromShootDirDelta || i >= fromShootDirDelta) {
					Direction shootDirLeft = FrameInfo.SHOOTING_DIRECTIONS[DknShootingDirPreparer
							.normalize(frame.getProbableShootingDir() + i)];
					final int bulletOccurDelay = Math.abs(i) + 2; // Turn + Bullet Occur Delay
					TargetHitResult thr = evaluateTargetHitWithoutDirection(frame, ship,
							shootDirLeft, bulletOccurDelay, target);
					if (thr.getHitInFrames() > bulletOccurDelay) {
						thr.setVectoredTurns(i);
						if (fastestThr == null
								|| thr.getHitInFrames() < fastestThr.getHitInFrames()) {
							fastestThr = thr;
						}
					}
				}
			}
		}
		if (fastestThr == null) {
			return THR_NO_HIT; // Kein Treffer mglich
		} else {
			return fastestThr;
		}
	}

	/**
	 * Prfen, in welcher Schiffsrichtung (Angabe ber Anfang und ein von-bis-+/--Delta) ein Target
	 * mit mglichst wenig Drehbewegungen getroffen werden kann, d.h. es wird erst ber ber die
	 * Schussrichtungen und dann die nach Entfernung sortierten Targets iteriert. Wenn ja, die
	 * Anzahl der Frames bis zum Treffer und Informationen ber die "genutzte" Bullet und das
	 * "getroffene" Target in Form von ExpectedBulletInfo und ShotTargetInfo zurckliefern.
	 */
	private TargetHitResult findFirstTargetHit(FrameInfo frame, final SpaceShip ship,
			int fromShootDirDelta, int toShootDirDelta, List<MovingGameObject> targets) {
		for (int i = fromShootDirDelta; i <= toShootDirDelta; i++) { // halb rum
			for (MovingGameObject target : targets) {
				TargetHitResult thr = evaluateTargetHitWithDirection(frame, ship, i, target);
				if (thr.getHitInFrames() > 0) {
					if (!tooDangerousToFire(target, thr.getHitInFrames())) {
						return thr;
					}
				}
			}
		}
		return THR_NO_HIT; // Kein Treffer mglich
	}

	/**
	 * Wenn ein grerer Asteroid zu nahe ist, schiee ich besser nicht auf ihn, sonst knnen mich
	 * die Trmmer treffen. Ausnahmen sind natrlich Asteroiden, die mir direkt gefhrlich werden.
	 */
	private boolean tooDangerousToFire(MovingGameObject target, int hitInFrames) {
		return !target.isDangerous() && !(target instanceof Ufo)
				&& (target.getSize() == 32 || target.getSize() == 16) && hitInFrames > 0
				&& hitInFrames < FRAMES_AHEAD_TOO_DANGEROUS_TOO_FIRE;
	}

	private TargetHitResult evaluateTargetHitWithDirection(FrameInfo frame, final SpaceShip ship,
			int i, MovingGameObject target) {
		Direction shootDirLeft = FrameInfo.SHOOTING_DIRECTIONS[DknShootingDirPreparer
				.normalize(frame.getProbableShootingDir() + i)];
		final int bulletOccurDelay = i + 2; // ShootingDir Turn Delay + Bullet Occur Delay
		TargetHitResult thrLeft = evaluateTargetHitWithoutDirection(frame, ship, shootDirLeft,
				bulletOccurDelay, target);
		if (thrLeft.getHitInFrames() > bulletOccurDelay) {
			thrLeft.setVectoredTurns(i);
			return thrLeft;
		}
		Direction shootDirRight = FrameInfo.SHOOTING_DIRECTIONS[DknShootingDirPreparer
				.normalize(frame.getProbableShootingDir() - i)];
		TargetHitResult thrRight = evaluateTargetHitWithoutDirection(frame, ship, shootDirRight,
				bulletOccurDelay, target);
		if (thrRight.getHitInFrames() > bulletOccurDelay) {
			thrRight.setVectoredTurns(-i);
			return thrRight;
		}
		return THR_NO_HIT; // Kein Treffer mglich
	}

	private TargetHitResult evaluateTargetHitWithoutDirection(FrameInfo frame, SpaceShip ship,
			Direction shootDir, int bulletOccurDelay, MovingGameObject target) {
		double bvx = shootDir.getBulletVelocity().getX();
		double bvy = shootDir.getBulletVelocity().getY();
		int bx = ship.getX() + (int) shootDir.getDisplacement().getX();
		int by = ship.getY() + (int) shootDir.getDisplacement().getY();
		if (target.getVelocityX() == 0.0 && target.getVelocityY() == 0.0) {
			// Geschwindigkeit des Targets ist noch unbekannt => ist schieen sinnlos
		} else {
			possibleBullet.setX((int) Math.round(bx - bulletOccurDelay * bvx));
			possibleBullet.setY((int) Math.round(by - bulletOccurDelay * bvy));
			possibleBullet.setVelocity(bvx, bvy);
			// Hier liee sich tunen, indem nicht 1-70 Frames gerechnet werden, sondern die
			// Frame-Range, die ich mathematisch ermitteln konnte (Code ist in Revision 42).
			int hitInFrames = target.hitInFrames(possibleBullet, MIN_BULLET_LIFETIME
					+ bulletOccurDelay); // kommt spter, luft rechnerisch daher lnger
			if (hitInFrames > 0) {
				return new TargetHitResult(hitInFrames, new ExpectedBulletInfo(frame, bx, by,
						target), new ShotTargetInfo(target, frame.getReceiveTime()
						+ (hitInFrames + DEATH_TOLERANCE_FRAMES) * 1000
						/ GameData.FRAMES_PER_SECOND));
			}
		}
		return THR_NO_HIT; // Kein Treffer mglich
	}

	/**
	 * Sammelt ein paar Werte zur berprfung der tatschlich erschienenen Bullet.
	 */
	public class ExpectedBulletInfo {

		private int frameIndex;

		private int expectedX;

		private int expectedY;

		private Integer targetIdentity;

		public ExpectedBulletInfo(FrameInfo frame, int x, int y, MovingGameObject target) {
			frameIndex = frame.getIndex();
			expectedX = x;
			expectedY = y;
			targetIdentity = target.getIdentity();
		}

		public Point getExpectedLocation() {
			return new Point(expectedX, expectedY);
		}

		public Integer getTargetIdentity() {
			return targetIdentity;
		}

		public String toString() {
			return String.format("EBI[frameIndex=%d,expectedX=%d,expectedY=%d,targetIdentity=%s]",
					frameIndex, expectedX, expectedY, targetIdentity);
		}

		public int getFrameIndex() {
			return frameIndex;
		}

	}

	/**
	 * Halter-Klasse fr die Ergebnisse von findPossibleTargetHit
	 */
	public class TargetHitResult {

		private int hitInFrames;

		private ExpectedBulletInfo expectedBulletInfo;

		private ShotTargetInfo shotTargetInfo;

		private int vectoredTurns;

		public TargetHitResult(int hitInFrames, ExpectedBulletInfo expectedBulletInfo,
				ShotTargetInfo shotTargetInfo) {
			this.hitInFrames = hitInFrames;
			this.expectedBulletInfo = expectedBulletInfo;
			this.shotTargetInfo = shotTargetInfo;
		}

		public int getHitInFrames() {
			return hitInFrames;
		}

		public ExpectedBulletInfo getExpectedBulletInfo() {
			return expectedBulletInfo;
		}

		public ShotTargetInfo getShotTargetInfo() {
			return shotTargetInfo;
		}

		public String toString() {
			if (expectedBulletInfo == null || shotTargetInfo == null) {
				return String.format("THR[hitInFrames=%d]", hitInFrames);
			} else {
				return String.format("THR[hitInFrames=%d,%s,%s]", hitInFrames, shotTargetInfo
						.toString(), expectedBulletInfo.toString());
			}
		}

		public int getTurns() {
			return Math.abs(vectoredTurns);
		}

		public int getVectoredTurns() {
			return vectoredTurns;
		}

		public void setVectoredTurns(int vectoredTurns) {
			this.vectoredTurns = vectoredTurns;
		}

	}

	/**
	 * Information ber "angeschossene" Asteroide oder UFOs.
	 */
	public class ShotTargetInfo {

		private Integer targetIdentity;

		private Point origin;

		private long expectedDeathTime;

		private MovingGameObject shotTarget; // FIXME -dkn- Nur fr "Nachschussaufrumen"

		public ShotTargetInfo(MovingGameObject givenTarget, long givenExpectedDeathTime) {
			targetIdentity = givenTarget.getIdentity();
			origin = givenTarget.getOrigin();
			expectedDeathTime = givenExpectedDeathTime;
			shotTarget = givenTarget;
		}

		public boolean targetShouldHaveBeenDeath(FrameInfo frame) {
			return frame.getReceiveTime() > expectedDeathTime;
		}

		public boolean targetIsUfoAndHasChangedDirection(MovingGameObject target) {
			return target instanceof Ufo && !origin.equals(target.getOrigin());
		}

		public Integer getTargetIdentity() {
			return targetIdentity;
		}

		public long getExpectedDeathTime() {
			return expectedDeathTime;
		}

		public String toString() {
			return String.format("STI[id=%d,expectedDeathTime=%d]", targetIdentity,
					expectedDeathTime);
		}

		public Point getOrigin() {
			return origin;
		}

		public MovingGameObject getShotTarget() {
			return shotTarget;
		}

	}

	public class BySquaredShipDistance implements Comparator<MovingGameObject> {

		public int compare(MovingGameObject o1, MovingGameObject o2) {
			int sd1 = o1.getSquaredDistanceToShip();
			int sd2 = o2.getSquaredDistanceToShip();
			return (sd1 < sd2) ? -1 : ((sd1 == sd2) ? 0 : 1);
		}

	}

}
