/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.research;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.client.Commands;
import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.game.objects.AngleByteRange;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * A simple bot that can be used to create the angle byte map found in 
 * the {@link Ship} class. Should be run against a fresh started mame
 * server and will turn three times on its own axis and then output
 * the map on the console. 
 * 
 * @author Henning Faber
 */
public class RecordAngleByte implements IAsteroidsBot {

    /**
     * The last game status received from the mame server. 
     */
    private GameStatus m_gameStatus;
    
    /**
     * The map with the recorded angle bytes. 
     */
    private Map<Double, Set<Integer>> m_angleMap 
        = new HashMap<Double, Set<Integer>>(128);
    
    /**
     * The initial angle, after the mame server has been started.
     */
    private int m_angle = -3;
    
    /**
     * Flag that indicates, if the task has yet been completed.  
     */
    private boolean m_done = false;

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.bot.IAsteroidsBot#compute()
     */
    public Commands compute() {
        Commands c = new Commands();
        Ship ship = m_gameStatus.getShip();
        
        if ((ship != null) && (!m_done)) {

            if (m_angle >= 0) {
                double angle = ship.getViewAngle();
                Set<Integer> angleByteSet = m_angleMap.get(angle);
                if (angleByteSet == null) {
                    angleByteSet = new HashSet<Integer>();
                    m_angleMap.put(angle, angleByteSet);
                }
                angleByteSet.add(Integer.valueOf(m_angle));
                
                m_angle = (m_angle + AngleByteRange.ANGLE_BYTE_STEP_SIZE) & 0xff;
                if (m_angle == 0) {
                    m_done = true;
                    System.out.println(dumpMap());
                }
            } else {
                m_angle = 0;
            }
            
            c.pressLeft(true);
        }
        
        return c;
    }
    
    private String dumpMap() {
        StringBuilder sb = new StringBuilder();
        
        sb.append("\nprivate static final Map<Double, AngleByteRange> ANGLE_BYTE_MAP =\n");
        sb.append("    new HashMap<Double, AngleByteRange>();\n");
        sb.append("static {\n");
        
        Set<Entry<Double, Set<Integer>>> entrySet = m_angleMap.entrySet();
        List<Entry<Double, Set<Integer>>> sortedEntrySet = new ArrayList<Entry<Double, Set<Integer>>>(
                entrySet);
        Collections.sort(sortedEntrySet, new Comparator<Entry<Double, Set<Integer>>>() {

            public int compare(Entry<Double, Set<Integer>> o1, Entry<Double, Set<Integer>> o2) {
                Set<Integer> s1 = o1.getValue();
                Set<Integer> s2 = o2.getValue();
                return Collections.min(s1) - Collections.min(s2);
            }
        });
        
        for (Entry<Double, Set<Integer>> entry : sortedEntrySet) {
            int lowAngle = Integer.MAX_VALUE;
            int highAngle = Integer.MIN_VALUE;
            for (int a : entry.getValue()) {
                if (a < lowAngle) {
                    lowAngle = a;
                }
                if (a > highAngle) {
                    highAngle = a;
                }
            }
            sb.append("    ANGLE_BYTE_MAP.put(");
            sb.append(entry.getKey());
            sb.append(", new AngleByteRange(");
            sb.append((lowAngle - AngleByteRange.ANGLE_BYTE_STEP_SIZE) & 0xff);
            sb.append(", ");
            sb.append((highAngle - AngleByteRange.ANGLE_BYTE_STEP_SIZE) & 0xff);
            sb.append("));\n");
        }
        sb.append("}\n");
        
        return sb.toString();
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public void gameStatusUpdated(GameStatusEvent e) {
        m_gameStatus = e.getGameStatus();
    }
}
