/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

package de.hfaber.asteroids.bot.sittingduck;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.field.Screen;
import de.hfaber.asteroids.game.objects.AngleByteRange;
import de.hfaber.asteroids.game.objects.Bullet;
import de.hfaber.asteroids.game.objects.GameObject;
import de.hfaber.asteroids.game.objects.ScaleableGameObject;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.objects.ShotVector;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * @author Henning Faber
 */
public class MoveTable {

    /**
     * The LOGGER instance.
     */
    private static final Logger LOGGER = Logger.getLogger(MoveTable.class);

    /**
     * The maximum number of calculations that are executed during a 
     * single compute cycle.
     */
    private static final int MAX_CALCULATIONS_PER_FRAME = 16384;

    /**
     * The maximum number of frames for which a possible collision of
     * a target with the ship is checked. Collisions are only checked
     * once, when a new target appears in the game. Thus, this value
     * should hold at least as many frames as are usually needed to
     * clear a complete level.   
     */
    private static final int MAX_COLLISION_LOOK_AHEAD = 2000;
    
    /**
     * Number of frames that are subtracted from the collision frame 
     * number that is calculated for a target. This allows to hit the 
     * target before the actual collision occurs. However, this value
     * should not be set too high, because targets that are already
     * within the safety range will not be recognized as collision
     * targets anymore.
     */
    private static final int COLLISION_SAFETY_FRAMES = 3;

    /**
     * Comparator for the move sets of the move table. If a shot fired 
     * at a certain frame with a certain angle crosses several targets, the 
     * nearest target will be the one that is actually hit.
     */
    private static final Comparator<Move> MOVE_TABLE_COMPARATOR = 
        new Comparator<Move>() {

            /* (non-Javadoc)
             * @see java.util.Comparator#compare(T, T)
             */
            public int compare(Move o1, Move o2) {
                // targets with lower impact frame number are hit first
                int result = o1.getImpactFrameNo() - o2.getImpactFrameNo();
                
                // since the move table consists of sets (which do not allow
                // duplicate elements), make sure that if there should ever
                // be two targets that are hit at exactly the same frame,
                // they are still not considered as equal
                if (result == 0) {
                    result = o1.getTargetId() - o2.getTargetId();
                }
                
                // return the result
                return result;
            }
        };
        
    /**
     * Table with all possible moves. The table is cyclic, i.e. the moves for
     * the current game status are not constantly at top of the table but move
     * along the table lines with subsequent frames.
     */
    private final TreeSet<Move>[][] m_moveTable;
    
    /**
     * The maximum number of lines that the table can hold.
     */
    private final int m_tableSize;
    
    /**
     * The line in the move table, where the moves for the current frame can 
     * be found.
     */
    private int m_tableStart;
    
    /**
     * The first line <em>relative to the table start</em>, which contains
     * valid calculated data. This will usually be zero, meaning that all
     * the data from the beginning of the table is valid.
     */
    private int m_tableDataStart;
    
    /**
     * The number of lines in the move table that contain valid moves.
     */
    private int m_tableLength;
    
    /**
     * The last game status that was used to update the move table.
     */
    private GameStatus m_prevGs;
    
    /**
     * Maps game object ids to the frame number, when the object will collide
     * with the ship. A value of {@link Move#NO_COLLISION} denotes that the
     * object is not on collision course with the ship. 
     */
    private final Map<Integer, Integer> m_collisionFrames;
    
    /**
     * Creates a move table.
     * 
     * @param tableSize the maximum number of frames that will be calculated
     *  ahead
     */
    public MoveTable(int tableSize) {
        super();
        
        m_tableSize = tableSize;

        // initialize move table
        m_moveTable = createMoveTable(tableSize);
        m_tableStart = 0;
        m_tableDataStart = 0;
        m_tableLength = 0;
        
        // initialize collision frame list
        m_collisionFrames = new HashMap<Integer, Integer>();
    }

    /**
     * Creates and initializes the data structure for the move table.
     * 
     * @param tableSize the size of the move table to create 
     * @return the move table
     */
    @SuppressWarnings("unchecked")
    private TreeSet<Move>[][] createMoveTable(int tableSize) {
        TreeSet<Move>[][] table = new TreeSet[tableSize]
                [AngleByteRange.ANGLE_BYTE_VALUE_COUNT];
        for (int frame = 0; frame < tableSize; frame++) {
            for (int angle = 0; angle < AngleByteRange.ANGLE_BYTE_VALUE_COUNT; 
                    angle++) {
                table[frame][angle] = new TreeSet<Move>(MOVE_TABLE_COMPARATOR);
            }
        }
        return table;
    }

    /**
     * Resets the move table by removing all currently available moves.
     */
    public final void reset() {
        m_tableLength = 0;
        m_collisionFrames.clear();
    }
    
    /**
     * Updates the move table. First checks, if and how much of the existing
     * data can be re-used. Then updates the re-usable data for the new game
     * status and finally calculates the missing data that is needed to fill 
     * up the table again.
     * 
     * @param gs the game status
     * @param nextAvailableShot the relative frame number, when the next 
     *  shot is available
     */
    public final void update(GameStatus gs, int nextAvailableShot) {
        if (m_prevGs != null) {

            // check, if existing data is still valid
            Ship prevShip = m_prevGs.getShip();
            Ship ship = gs.getShip();
            if (prevShip != null) {

                // reset move table & collisions, if ship has moved
                if (!ship.getLocation().equals(prevShip.getLocation())) {
                    m_tableLength = 0;
                    m_collisionFrames.clear();
                }
                
                // reset move table, if shot angle was not exactly synchronized
                if (!Move.isReachableFrom(m_prevGs.getFrameNo(), 
                        prevShip.getShotAngle(), gs.getFrameNo(), 
                        ship.getShotAngle())) {
                    LOGGER.debug("Ship rotation out of sync. Resetting move table.");
                    m_tableLength = 0;
                }
            }
            
            // roll the move table
            int frameGap = gs.getFrameNo() - m_prevGs.getFrameNo();
            if (frameGap > 0) {
                
                // move the table start forwards
                m_tableStart = (m_tableStart + frameGap) % m_tableSize;
                
                // the first line with valid data remains the same, but the
                // pointer is relative to the table start -> thus, decrease
                // the pointer
                m_tableDataStart = m_tableDataStart - frameGap;
                if (m_tableDataStart < 0) {
                    m_tableDataStart = 0;
                }
                
                // decrease the table length
                m_tableLength = m_tableLength - frameGap;
                if (m_tableLength < 0) {
                    m_tableLength = 0;
                }
            }
            
            // if the next shot is available earlier than it has been
            // expected in previous calculations, subsequently calculate
            // the missing data now
            if (nextAvailableShot < m_tableDataStart) {
                List<ScaleableGameObject> targetList = gs.getTargetList();
                for (int frame = nextAvailableShot; frame < m_tableDataStart; 
                        frame++) {
                    calculateCompleteFrame(gs, targetList, frame);
                }
                m_tableDataStart = nextAvailableShot;
            }

            // update existing data, if any
            if (m_tableLength > 0) {
                updateExistingData(gs, m_prevGs, nextAvailableShot);
            } 
        } else {
            
            // previous game status is null -> reset move table & collisions
            m_tableLength = 0;
            m_collisionFrames.clear();
        }
        
        // add missing lines to move table
        addNewData(gs, nextAvailableShot);
        
        // remember the game status for the next update
        m_prevGs = gs;
    }

    /**
     * <p>
     * Updates all moves that are stored in the current scope of the move 
     * table. The current scope of the move table is determined by the table 
     * start, the table length, the current angle of the ship and the frame 
     * when the next shot is available.
     * </p>
     * <p>
     * The update compares the target lists of the current game status and 
     * the previous game status. For each target on the lists, the result 
     * of the comparison may have the following possible outcomes:
     * </p>
     * <ul>
     * <li><b>Destroyed target</b><br>
     * If a target from the previous game status is no longer available in 
     * in the current game status, all moves that refer to that target are 
     * removed from the move table's scope as described above.
     * <li><b>New target</b><br>
     * If a target from the current game status was not yet available in the
     * previous game status, all possible moves for that target are calculated.
     * <li><b>Changed target</b><br>
     * If a target is available in the current and the previous game status, 
     * but the direction vector of the target has changed, all previous 
     * calculated moves for that target are removed from the move table and 
     * new moves are calculated.
     * </ul>
     * 
     * @param gs the game status
     * @param nextAvailableShot the relative frame number, when the next 
     *  shot is available
     */
    private void updateExistingData(GameStatus gs, GameStatus prevGs, 
            int nextAvailableShot) {
        // get a sorted list of targets from the previous frame
        List<ScaleableGameObject> prevTargets = prevGs.getTargetList();
        Collections.sort(prevTargets);

        // when a new level starts, fill the table incrementally,
        // as it would also be done after a hyperspace jump or at
        // the game start
        if (prevTargets.size() == 0) {
            m_tableLength = 0;
            return;
        } 

        // get a sorted list of targets for the current frame
        List<ScaleableGameObject> curTargets = gs.getTargetList();
        Collections.sort(curTargets);
   
        // compare both lists and find differences
        int prevPtr = 0;
        int curPtr = 0;
        while ((prevPtr < prevTargets.size()) 
                && (curPtr < curTargets.size())) {
            
            ScaleableGameObject prev = prevTargets.get(prevPtr);
            ScaleableGameObject cur = curTargets.get(curPtr);
            int prevId = prev.getId();
            int curId = cur.getId();
      
            // do not calculate anything for untracked objects,
            // since these objects do not have a valid direction vector
            if (prevId == GameObject.UNTRACKED) {
                prevPtr++;
                continue;
            }
            if (curId == GameObject.UNTRACKED) {
                curPtr++;
                continue;
            }
      
            if (prevId == curId) {
                
                // object from previous frame is still available in current
                // frame -> check, if direction vector has changed
                if (!prev.getDirection().equals(cur.getDirection())) {
                    removeMoves(gs, prevId, nextAvailableShot);
                    addMoves(gs, cur, nextAvailableShot);
                }
                
                prevPtr++;
                curPtr++;
                
            } else if (prevId < curId) {
                
                // object from previous frame is no longer available
                // in current frame
                removeMoves(gs, prevId, nextAvailableShot);
                prevPtr++;
                
            } else if (curId < prevId) {
                
                // object from current frame is new (was not available
                // in previous frame)
                addMoves(gs, cur, nextAvailableShot);
                curPtr++;
            }
        }
        
        // if target list from previous frame still contains objects,
        // these objects are altogether no longer available in the
        // current frame
        while (prevPtr < prevTargets.size()) {
            GameObject prev = prevTargets.get(prevPtr);
            int prevId = prev.getId();
            removeMoves(gs, prevId, nextAvailableShot);
            prevPtr++;
        }
      
        // if target list from current frame still contains objects,
        // these objects are altogether new objects
        while (curPtr < curTargets.size()) {
            ScaleableGameObject cur = curTargets.get(curPtr);
            if (cur.getId() != GameObject.UNTRACKED) {
                addMoves(gs, cur, nextAvailableShot);
            }
            curPtr++;
        }
    }

    /**
     * Calculates all possible moves within the current scope of the move 
     * table that may lead to the destruction of the given game object. The 
     * current scope of the move table is determined by the table start, the
     * table length, the current angle of the ship and the frame when the 
     * next shot is available.
     * 
     * @param gs the game status
     * @param o the game object for which the moves should be added
     * @param nextAvailableShot the relative frame number, when the next 
     *  shot is available
     */
    private void addMoves(GameStatus gs, ScaleableGameObject target,
            int nextAvailableShot) {
        // iterate through all frames of the current scope of the move table
        int frameNo = nextAvailableShot;
        while (frameNo < m_tableLength) {
            
            // consider each shot angle that is reachable from the
            // current position within the given number of frames
            for (int angle = -frameNo; angle <= frameNo; angle++) {

                // if move is possible, add it to the move table
                Move m = calculateMove(gs, frameNo, angle, target);
                if (m != null) {
                    TreeSet<Move> queue = getMoveQueue(gs, frameNo, angle);
                    queue.add(m);
                }
            }
            
            // go on to the next frame
            frameNo++;
        }
    }

    /**
     * Removes all moves from the current scope of the move table that 
     * reference a target with the given id from the move table. The 
     * current scope of the move table is determined by the table start, 
     * the table length, the current angle of the ship and the frame when 
     * the next shot is available.
     * 
     * @param gs the game status
     * @param id the id of the target, whose moves should be removed
     * @param nextAvailableShot the relative frame number, when the next 
     *  shot is available
     */
    private void removeMoves(GameStatus gs, int id, int nextAvailableShot) {
        // remove any calculated collisions for the given id 
        m_collisionFrames.remove(id);
        
        // iterate through all frames of the current scope of the move table
        int frameNo = nextAvailableShot;
        while (frameNo < m_tableLength) {

            // consider each shot angle that is reachable from the
            // current position within the given number of frames
            for (int angle = -frameNo; angle <= frameNo; angle++) {
                
                // check all moves for this frame and angle
                TreeSet<Move> queue = getMoveQueue(gs, frameNo, angle);
                Iterator<Move> it = queue.iterator();
                while (it.hasNext()) {
                    
                    // remove move, if it references the given game object
                    Move m = it.next();
                    if (m.getTargetId() == id) {
                        it.remove();
                    }
                }
            }
            
            // go on to the next frame
            frameNo++;
        }
    }

    /**
     * Calculates all possible moves for all available game objects for the
     * lines of the move table that are missing.
     * 
     * @param gs the game status
     * @param nextAvailableShot the relative frame number, when the next 
     *  shot is available
     */
    private void addNewData(GameStatus gs, int nextAvailableShot) {
        
        // get the list of available targets
        List<ScaleableGameObject> targetList = gs.getTargetList();
        
        // determine first new frame to calculate
        if (m_tableLength < nextAvailableShot) {
            m_tableLength = nextAvailableShot;
            if (m_tableLength > m_tableSize) {
                m_tableLength = m_tableSize;
            }
        }
        
        // all data starting from the first new frame is newly
        // calculated and thus valid data
        if (m_tableLength < m_tableDataStart) {
            m_tableDataStart = m_tableLength;
        } else {
            m_tableDataStart = nextAvailableShot;
            if (m_tableDataStart > m_tableSize) {
                m_tableDataStart = m_tableSize;
            }
        }
        
        // fill table with missing lines
        int calculationsDone = 0;
        while ((m_tableLength < m_tableSize) 
                && (calculationsDone < MAX_CALCULATIONS_PER_FRAME)) {
            
            // add the new moves for the current frame
            calculateCompleteFrame(gs, targetList, m_tableLength);
            
            // update number of processed cells
            calculationsDone += (m_tableLength * 2 + 1) * targetList.size();
            
            // new line was added to the table
            m_tableLength++;
        }
    }

    /**
     * Calculates all moves for a given relative frame in the move table.
     * 
     * @param gs the game status
     * @param targetList the list of targets, for which the moves are
     *  calculated
     * @param frameNo the relative frame number, for which the moves are
     *  calculated
     */
    private void calculateCompleteFrame(GameStatus gs,
            List<ScaleableGameObject> targetList, int frameNo) {
        // consider each shot angle that is reachable from the
        // current position within the given number of frames
        for (int angle = -frameNo; angle <= frameNo; angle++) {

            // get queue for this frame and angle
            TreeSet<Move> queue = getMoveQueue(gs, frameNo, angle);
            
            // remove any former garbage
            queue.clear();

            // calculate moves for each available target
            for (ScaleableGameObject o : targetList) {

                // do not calculate anything for untracked objects,
                // since these objects do not have a valid direction
                // vector
                if (o.getId() != GameObject.UNTRACKED) {
                    
                    // if move is possible, add it to the move table
                    Move m = calculateMove(gs, frameNo, angle, o);
                    if (m != null) {
                        queue.add(m);
                    }
                }
            }
        }
    }
    
    /**
     * Calculates, if a the given game object can be hit after the given 
     * number of frames with the given angle. If so, a move object is created, 
     * which also holds the information about the frame number when the target
     * would be destroyed.
     * 
     * @param frameNo the frame number when to shoot, relative to the frame 
     *  number of the current game status; i.e. a value of zero denotes the
     *  frame number given by <code>gs.getFrameNo()</code>
     * @param angle the angle to use for shooting, relative to the current 
     *  angle of the ship
     * @param o the game object that should be hit
     * @param collisionFrameNo the frame number, when the given game 
     *  object will collide with the ship, or zero, if the game object is 
     *  not on collision course with the ship
     * @return a move object that describes how to hit the given target under 
     *  the given circumstances, or <code>null</code>, if the target cannot 
     *  be hit under these premises
     */
    private Move calculateMove(GameStatus gs, int frameNo, int angle, 
            ScaleableGameObject o) {
        // initialize return value
        Move move = null;
        
        // get angle relative to the current angle of the ship
        Ship ship = gs.getShip();
        int currentAngle = ship.getShotAngle();
        int relativeAngle = (currentAngle + AngleByteRange.ANGLE_BYTE_STEP_SIZE
                * angle) & 0xff;

        // get shot vector and ship position for the given angle
        Point shotVector = ShotVector.TABLE[relativeAngle];
        
        // determine minimum distance between the shot and the target
    
        // calculate difference between shot and target direction
        double bx = o.getDirection().getX() - shotVector.getX();
        double by = o.getDirection().getY() - shotVector.getY();
        
        // calculate time (frame number), when objects have minimum distance
        Point objectLocation = o.relativeProject(frameNo);
        double t = -((objectLocation.getX() * bx + objectLocation.getY() * by) 
                / (bx * bx + by * by));

        // make sure that the object is not behind the ship
        if (t > 0) {

            // calculate distance for the nearest lower discrete frame
            double minDist;
            double lowerT = Math.floor(t);
            double lowerH1 = objectLocation.getX() + lowerT * bx;
            double lowerH2 = objectLocation.getY() + lowerT * by;
            double lowerDist = lowerH1 * lowerH1 + lowerH2 * lowerH2;
            
            // check, if bullet crosses the target in lower discrete frame
            if (lowerDist < o.getSquareRadius()) {
                t = lowerT;
                minDist = lowerDist; 
            } else {
                
                // calculate distance for the nearest upper discrete frame
                double upperT = Math.ceil(t);
                double upperH1 = objectLocation.getX() + upperT * bx;
                double upperH2 = objectLocation.getY() + upperT * by;
                double upperDist = upperH1 * upperH1 + upperH2 * upperH2;
                
                // check, if bullet crosses the target in upper discrete frame
                if (upperDist < o.getSquareRadius()) {
                    t = upperT;
                    minDist = upperDist; 
                } else {
                    
                    // shot cannot hit 
                    return move;
                }
            }

            // correct t by the radius of the object
            Point d = shotVector.delta(o.getDirection());
            double sqrLen = d.getX() * d.getX() + d.getY() * d.getY();
            double sqrRadius = o.getSquareRadius() 
                + Bullet.BULLET_SQUARE_RADIUS - minDist;
            double correction = Math.sqrt(sqrRadius / sqrLen);
            t -= correction;
            
            // check, if target is in range
            if ((t > 0) && (t < Bullet.BULLET_LIFE_TIME)) {            

                // get shot & impact frame numbers
                int shotFrameNo = gs.getFrameNo() + frameNo;
                int impactFrameNo = shotFrameNo + (int)Math.round(t);

                // determine collision frame number
                Integer collisionFrameNo = m_collisionFrames.get(o.getId());
                if (collisionFrameNo == null) {
                    collisionFrameNo = calculateCollisionFrame(gs, o);
                    m_collisionFrames.put(o.getId(), collisionFrameNo);
                }
                
                // do not consider collision, if move cannot prevent it
                if (collisionFrameNo < impactFrameNo) {
                    collisionFrameNo = Move.NO_COLLISION;
                }
                
                // create move object
                move = new Move(shotFrameNo, impactFrameNo, relativeAngle, 
                        o.getId(), o.getSquareRadius(), (int)minDist, 
                        collisionFrameNo);
            }
        }
        
        return move;
    }

    /**
     * Calculates the frame number, when the given object will collide with
     * the ship from the given game status. 
     * 
     * @param gs the game status
     * @param o the game object, for which the collision frame should be 
     *  calculated
     * @return the collision frame number or {@link Move#NO_COLLISION}, if
     *  the object will not collide with the ship for the next
     *  {@link #MAX_COLLISION_LOOK_AHEAD} frames.
     */
    private int calculateCollisionFrame(GameStatus gs, ScaleableGameObject o) {
        
        // start with the assumption that the object will not collide
        int collisionFrameNo = Move.NO_COLLISION;
        
        // get the moving direction of the ship
        Ship ship = gs.getShip();
        Point shipDirection = ship.getDirection();

        // calculate difference between ship and target direction
        Point objectDirection = o.getDirection();
        Point objectLocation = o.getRelativeLocation();
        double bx = objectDirection.getX() - shipDirection.getX();
        double by = objectDirection.getY() - shipDirection.getY();
        double bSquareLen = bx * bx + by * by;

        // calculate phase
        double xPhase = Math.abs(Screen.WIDTH / (double)objectDirection.getX());
        double yPhase = Math.abs(Screen.HEIGHT / (double)objectDirection.getY());
        double phase = (xPhase < yPhase) ? xPhase : yPhase;

        // determine how many straight lines have to be examined,
        // before the required number of frames has been reached
        int lines = (int)Math.ceil(MAX_COLLISION_LOOK_AHEAD / phase) + 1; 

        // examine all required straight lines
        for (int line = 0; line < lines; line++) {
            
            // calculate point that is located on the current straight line
            double xMovement = line * phase * objectDirection.getX();
            double yMovement = line * phase * objectDirection.getY();
            double px = Point.normalize(objectLocation.getX() + xMovement, 
                    Screen.WIDTH);
            double py = Point.normalize(objectLocation.getY() + yMovement, 
                    Screen.HEIGHT);
            
            // calculate time (number of frames), when objects have minimum 
            // distance
            double t = -((px * bx + py * by) / bSquareLen);

            // calculate the actual minimum distance
            double h1 = px + t * bx;
            double h2 = py + t * by;
            double minDist = h1 * h1 + h2 * h2;

            // object will hit the ship, if the minimum distance is less
            // than the sum of the ship's and the object's radius
            if (minDist < o.getSquareRadius() + ship.getSquareRadius()) {
                
                // correct t by the number of frames that are needed to
                // reach the current straight line
                t += line * phase;
                
                // if t is negative, the object is moving away from the ship
                if (t > 0) {
                    
                    // correct t by the radius of the objects
                    double correction = 0;
                    Point d = shipDirection.delta(objectDirection);
                    double sqrLen = d.getX() * d.getX() + d.getY() * d.getY();
                    double sqrRadius = o.getSquareRadius()
                            + ship.getSquareRadius() - minDist;
                    correction = Math.sqrt(sqrRadius / sqrLen);
                    
                    // determine collision frame number
                    collisionFrameNo = gs.getFrameNo()
                            + (int)Math.ceil(t - correction);
                    
                    // hit colliding targets a few frames before the
                    // actual impact
                    collisionFrameNo -= COLLISION_SAFETY_FRAMES;
                    
                    // collision found -> no further calculation necessary
                    break;
                }
            }            
        }
        
        // return the collision frame
        return collisionFrameNo;
    }
    
    /**
     * Returns the priority queue with the moves that are possible when shooting
     * on the given relative frame with the given relative angle.
     * 
     * @param frameNo the frame number, relative to the current game status
     * @param angle the shot angle, relative to the current shot angle of 
     *  the ship
     * @return the priority queue for the given parameters
     */
    public final TreeSet<Move> getMoveQueue(GameStatus gs, 
            int frameNo, int angle) {
        // determine the relative frame number
        int relativeFrameNo = (m_tableStart + frameNo) % m_tableSize;

        // determine the relative angle
        Ship ship = gs.getShip();
        int currentAngle = ship.getShotAngle();
        int relativeAngle = (currentAngle + AngleByteRange.ANGLE_BYTE_STEP_SIZE 
                * angle) & 0xff;

        TreeSet<Move> queue = m_moveTable[relativeFrameNo][relativeAngle];
        return queue;
    }

    /**
     * Returns the number in the move table that currently contain
     * valid moves. 
     * 
     * @return the tableLength
     */
    public final int getTableLength() {
        return m_tableLength;
    }
}
