/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.state;

import java.util.List;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.GameObject;

/**
 * @author Henning Faber
 */
class ApproximateMatchStrategy implements IMatchStrategy {

    /**
     * Flag that determines, if the strategy should consider the difference
     * between the actual and the expected moving direction of a potential
     * match, even if the actual moving direction is null. 
     */
    private final boolean m_matchAngleForNullDirections;
    
    /**
     * Creates an approximate match strategy with the given parameters.
     * 
     * @param matchAngleForNullDirections if <code>true</code>, the strategy
     *  will consider the difference between the actual and the expected
     *  direction vector, even if the actual direction vector is null; if
     *  <code>false</code>, it will not
     */
    public ApproximateMatchStrategy(boolean matchAngleForNullDirections) {
        super();
        m_matchAngleForNullDirections = matchAngleForNullDirections; 
    } 
    
    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.state.IMatchStrategy#satisfiesPrecondition(de.hfaber.asteroids.asteroids.game.objects.GameObject, de.hfaber.asteroids.asteroids.game.objects.GameObject, int)
     */
    public boolean satisfiesPrecondition(GameObject current, GameObject prev,
            int frameGap) {
        
        // maximum possible move distance
        int allowedMaxDist = current.getMaxFrameDist();

        // determine actual distance between the two objects
        Point location = current.getLocation();
        int dist = location.distance(prev.getLocation());
        
        // precondition is satisfied, if the actual distance is less than
        // or equal to the allowed distance
        return dist <= allowedMaxDist * frameGap * frameGap;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.state.IMatchStrategy#selectBestMatch(java.util.List, de.hfaber.asteroids.asteroids.game.objects.GameObject, int)
     */
    public GameObject selectBestMatch(GameObject current, 
            List<? extends GameObject> matchList, int frameGap) {

        // initialize best match
        GameObject bestMatch = null; 
        int bestQuality = Integer.MAX_VALUE;
        
        // evaluate each game object from the match list
        for (GameObject match : matchList) {
            
            // get the number of pixels that the object has
            // moved from the position of the potential match to 
            // the actual position (i.e. the direction vector
            // of the last movement)
            Point matchLocation = match.getLocation();
            Point delta = matchLocation.delta(current.getLocation());
            
            // get the expected direction vector from the potential
            // match (i.e. the vector that has been tracked when
            // the match was created one frame ago)
            Point expectedDirection = match.getDirection();
            
            // If the object already has a non-null direction vector,
            // determine the surface area that is spanned by this
            // actual direction vector and the expected one.
            // A smaller value denotes a better match, because
            // then the angle between the vectors is also small.
            int quality = Integer.MAX_VALUE - 1; 
            if (m_matchAngleForNullDirections
                    || (expectedDirection.getY() != 0)) {
                quality = expectedDirection.surfaceArea(delta);
                quality = Math.abs(quality);
            }
            
            // if quality of this match is the better than the
            // best so far, remember it as the new best match
            if (quality < bestQuality) {
                bestMatch = match;
                bestQuality = quality;
            }
        }
        
        // return the determined best match
        return bestMatch;
    }

}
