/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.gui;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.UIManager;

import com.jgoodies.forms.builder.PanelBuilder;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author Henning Faber
 */
public class TelemetryFrame extends JFrame {

    /**
     * Serial version UID for serialization API.
     */
    private static final long serialVersionUID = -2237989586617415694L;

    // set look and feel, if frame is used
    static {
        try {
            UIManager.setLookAndFeel(
                    "com.jgoodies.looks.plastic.PlasticLookAndFeel");
        } catch (Exception e) {
            // ignore
        }    
    }
    
    private JLabel m_frameRateLbl;
    private JLabel m_freeMemLbl;
    private JLabel m_scoreLbl;
    private JLabel m_scorePerFiveMinLbl;
    private JLabel m_lifesLbl;
    private JLabel m_roundLbl;
    private JLabel m_gameDurationLbl;
    
    private RoundDurationTableModel m_roundDurationModel;
    private JTable m_roundDurationTbl;


    /**
     * Creates a telemetry frame, initializes the components and
     * adds them to the content pane.
     */
    public TelemetryFrame() {
        super();
        initComponents();
        getContentPane().add(createPanel());
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setTitle("Telemetry");
        setSize(275, 505);
        setLocation(355, 0);
    }

    /**
     * Initializes the panel components.
     */
    private void initComponents() {
        m_frameRateLbl = new JLabel();
        m_freeMemLbl = new JLabel();
        m_scoreLbl = new JLabel();
        m_scorePerFiveMinLbl = new JLabel();
        m_lifesLbl = new JLabel();
        m_roundLbl = new JLabel();
        m_gameDurationLbl = new JLabel();
        
        m_roundDurationModel = new RoundDurationTableModel();
        m_roundDurationTbl = new JTable(m_roundDurationModel);
    }
    
    /**
     * Adds the components to the panel.
     */
    private JComponent createPanel() {
        // create layout
        FormLayout layout = new FormLayout(
            // columns
                "fill:min:grow",
            // rows
                "pref, 15px, pref, 5px, fill:min:grow");
        
        // create panel builder
        PanelBuilder builder = new PanelBuilder(layout);
        builder.setDefaultDialogBorder();

        // game info
        builder.add(createGameInfoPanel());
        builder.nextLine(2);
        
        // round duration section
        builder.addSeparator("Round durations");
        builder.nextLine(2);
        
        // round duration table
        builder.add(createScrollPane(m_roundDurationTbl));
        builder.nextLine(2);

        // return the panel
        return builder.getPanel();
    }

    /**
     * Creates a sub panel that displays various game options.
     * 
     * @return the option sub panel
     */
    private Component createGameInfoPanel() {
        // create layout
        FormLayout layout = new FormLayout(
            // columns
                "min, 5px, fill:min:grow, 10px, "
                    + "min, 5px, min",
            // rows
                "pref, 5px, " 
                    + "pref, 5px, "
                    + "pref, 5px, "
                    + "pref, 5px, "
                    + "pref, 5px, "
                    + "pref");
        
        // create panel builder
        PanelBuilder builder = new PanelBuilder(layout);

        // game info section
        builder.addSeparator("Game information");
        builder.nextLine(2);
        
        // score
        builder.add(new JLabel("Score:"));
        builder.nextColumn(2);
        builder.add(m_scoreLbl);
        builder.nextColumn(2);

        // frame rate
        builder.add(new JLabel("Framerate:"));
        builder.nextColumn(2);
        builder.add(m_frameRateLbl);
        builder.nextLine(2);
        
        // score per 5 minutes
        builder.add(new JLabel("Score/5min:"));
        builder.nextColumn(2);
        builder.add(m_scorePerFiveMinLbl);
        builder.nextColumn(2);

        // free memory
        builder.add(new JLabel("Freemem:"));
        builder.nextColumn(2);
        builder.add(m_freeMemLbl);
        builder.nextLine(2);

        // lifes
        builder.add(new JLabel("Lifes:"));
        builder.nextColumn(2);
        builder.add(m_lifesLbl);
        builder.nextLine(2);
        
        // rounds
        builder.add(new JLabel("Round:"));
        builder.nextColumn(2);
        builder.add(m_roundLbl);
        builder.nextLine(2);
        
        // playing time
        builder.add(new JLabel("Time elapsed:"));
        builder.nextColumn(2);
        builder.add(m_gameDurationLbl);
        
        // return the option sub panel
        return builder.getPanel();
    }

    /**
     * Puts the given component into a scroll pane inside a minimum
     * filling panel.
     * 
     * @param c the component
     * @return panel with scroll pane and component
     */
    private Component createScrollPane(Component c) {
        FormLayout layout = new FormLayout("fill:min:grow", "fill:min:grow");
        PanelBuilder builder = new PanelBuilder(layout);
        JScrollPane scrollPane = new JScrollPane(c);
        builder.add(scrollPane);
        return builder.getPanel();
    }

    /**
     * @return the frameRateLbl
     */
    public final JLabel getFrameRateLbl() {
        return m_frameRateLbl;
    }

    /**
     * @return the freeMemLbl
     */
    public final JLabel getFreeMemLbl() {
        return m_freeMemLbl;
    }

    /**
     * @return the playingTimeLbl
     */
    public final JLabel getGameDurationLbl() {
        return m_gameDurationLbl;
    }

    /**
     * @return the lifesLbl
     */
    public final JLabel getLifesLbl() {
        return m_lifesLbl;
    }

    /**
     * @return the roundLbl
     */
    public final JLabel getRoundLbl() {
        return m_roundLbl;
    }

    /**
     * @return the scoreLbl
     */
    public final JLabel getScoreLbl() {
        return m_scoreLbl;
    }

    /**
     * @return the scorePerFiveMinLbl
     */
    public final JLabel getScorePerFiveMinLbl() {
        return m_scorePerFiveMinLbl;
    }

    /**
     * @return the roundDurationModel
     */
    public final RoundDurationTableModel getRoundDurationModel() {
        return m_roundDurationModel;
    }
}
