/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.simplebot;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.GameObject;

/**
 * @author Henning Faber
 */
public final class DistCalculator {

    /**
     * Calculates the square of the minimum distance that two game objects 
     * will have from each other and the number of frames until this minimum 
     * distance is reached.
     * 
     * @param o1 the first game object
     * @param o2 the second game object
     * @return A MinDist object, holding the caluclated square minimum 
     *  distance and the frame number, when this minimum distance will 
     *  occur. If the frame number is negative, the objects are moving 
     *  from each other and have already had their minimum distance in 
     *  the past. If the frame number is NaN, the objects are moving 
     *  parallel.
     */
    public static MinDist calculateMinimumDistance(GameObject o1, 
            GameObject o2) {
        return calculateMinimumDistance(o1.getLocation(), o1.getDirection(), 
                o2.getLocation(), o2.getDirection());
    }

    /**
     * Calculates the square of the minimum distance that two points will 
     * have from each other and the number of frames until this minimum 
     * distance is reached. Both points are described by their location 
     * (l1 and l2), their moving direction (d1 and d2) and their moving 
     * speed, which is determined by the length of their direction vector.
     * 
     * @param l1 the location of the first point at frame 0 (now)
     * @param d1 the moving direction of the first point
     * @param l2 the location of the second point at frame 0 (now)
     * @param d2 the moving direction of the second point
     * @return A MinDist object, holding the caluclated square minimum 
     *  distance and the frame number, when this minimum distance will 
     *  occur. If the frame number is negative, the objects are moving 
     *  from each other and have already had their minimum distance in 
     *  the past. If the frame number is NaN, the objects are moving 
     *  parallel.
     */
    public static MinDist calculateMinimumDistance(Point l1, Point d1, 
            Point l2, Point d2) {
        
        // calculate difference between object locations 
        Point a = l2.delta(l1);
    
        // calculate difference between object directions 
        double bx = d1.getX() - d2.getX();
        double by = d1.getY() - d2.getY();
        
        // calculate time (frame number), when objects have minimum distance
        double t = -((a.getX() * bx + a.getY() * by) / (bx * bx + by * by));
        
        // calculate the actual minimum distance
        double h1 = a.getX() + t * bx;
        double h2 = a.getY() + t * by;
        double minDist = h1 * h1 + h2 * h2;
        
        // return the result
        MinDist result = new MinDist(t, minDist);
        return result;
    }

    /**
     * Calculates the square of the minimum distance that two points will 
     * have from each other and the number of frames until this minimum 
     * distance is reached. Both points are described by their location 
     * (l1 and l2), their moving direction (d1 and d2) and their moving 
     * speed, which is determined by the length of their direction vector.
     * Here, the first point is supposed to be at the origin. 
     * 
     * @param d1 the moving direction of the first point
     * @param l2 the location of the second point at frame 0 (now)
     * @param d2 the moving direction of the second point
     * @return A MinDist object, holding the caluclated square minimum 
     *  distance and the frame number, when this minimum distance will 
     *  occur. If the frame number is negative, the objects are moving 
     *  from each other and have already had their minimum distance in 
     *  the past. If the frame number is NaN, the objects are moving 
     *  parallel.
     */
    public static MinDist calculateMinimumDistance(Point d1, 
            Point l2, Point d2) {
        // calculate difference between object directions 
        double bx = d2.getX() - d1.getX();
        double by = d2.getY() - d1.getY();
        
        // calculate time (frame number), when objects have minimum distance
        double t = -((l2.getX() * bx + l2.getY() * by) / (bx * bx + by * by));
        
        // calculate the actual minimum distance
        double h1 = l2.getX() + t * bx;
        double h2 = l2.getY() + t * by;
        double minDist = h1 * h1 + h2 * h2;
        
        // return the result
        MinDist result = new MinDist(t, minDist);
        return result;
    }
}
