/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.sittingduck;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.AngleByteRange;


/**
 * A move consists of the point of time when firing a shot with a certain
 * angle will inevitably hit a certain target. The point of time is identified 
 * by the frame number, the angle is given by the angle byte and the target 
 * is identified by its game object id.
 * 
 * @author Henning Faber
 */
public class Move implements Comparable<Move> {

    /**
     * Value of the collision frame, if the target is not on collision
     * course with the ship.
     */
    public static final int NO_COLLISION = Integer.MAX_VALUE;
    
    /**
     * The frame number, when the shot has to be fired.
     */
    private final int m_shotFrameNo;
    
    /**
     * The frame number, when the shot will impact on the target and
     * destroy it.
     */
    private final int m_impactFrameNo;

    /**
     * The angle with which the shot has to be fired. 
     */
    private final int m_angle;
    
    /**
     * The id of the target that will be hit.
     */
    private final int m_targetId;
    
    /**
     * The size of the target.
     */
    private final int m_targetSize;
    
    /**
     * The minimum square distance between the shot and the target. 
     */
    private final int m_shotDistance;

    /**
     * The frame number, when the target will collide with the ship or
     * Integer.MAX_VALUE, if the target is not on collision course with 
     * the ship. 
     */
    private final int m_collisionFrameNo;
    
    /**
     * Creates a move with the given parameters.
     * 
     * @param shotFrameNo the shot frame number
     * @param impactFrameNo the impact frame number
     * @param angle the angle byte
     * @param targetId the id of the target
     * @param targetSize the size of the target
     * @param collisionFrameNo the collision frame number
     */
    public Move(int shotFrameNo, int impactFrameNo, int angle, int targetId,
            int targetSize, int shotDistance, int collisionFrameNo) {
        super();
        m_shotFrameNo = shotFrameNo;
        m_impactFrameNo = impactFrameNo;
        m_angle = angle;
        m_targetId = targetId;
        m_targetSize = targetSize;
        m_shotDistance = shotDistance;
        m_collisionFrameNo = collisionFrameNo;
    }

    /**
     * Determines if this move is still a valid move after the game has
     * advanced forward to the given move.
     * 
     * @param move the parent move
     * @return <code>true</code>, if this move is reachable from the given
     *  move, <code>false</code> if not
     */
    public final boolean isReachableFrom(Move move) {
        return isReachableFrom(move.getShotFrameNo(), move.getAngle());
    }
    
    /**
     * Determines if this move is still a valid move after the game has
     * advanced forward to the given frame and the ship has rotated to
     * the given angle.
     * 
     * @param frameNo the frame number
     * @param angle the rotation angle of the ship
     * @return <code>true</code>, if this move is reachable from the game
     *  state, <code>false</code> if not
     */
    public final boolean isReachableFrom(int frameNo, int angle) {
        return isReachableFrom(frameNo, angle, m_shotFrameNo, m_angle);
    }
    
    /**
     * Determines if a move that shoots a target at the given frame number
     * and shot angle is still a valid move after the game has
     * advanced forward to the given frame and the ship has rotated to
     * the given angle.
     * 
     * @param parentFrameNo the shot frame number of the 'move' from where
     *  the child 'move' should be tested
     * @param parentAngle the ship's shot angle of the 'move' from where
     *  the child 'move' should be tested 
     * @param childFrameNo the shot frame number of the 'move' that should be
     *  tested for reachability 
     * @param childAngle the ship's shot angle of the 'move' that should be
     *  tested for reachability
     * @return <code>true</code>, if the child 'move' is reachable from the 
     *  given parent 'move', <code>false</code> if not
     */
    public static final boolean isReachableFrom(int parentFrameNo, 
            int parentAngle, int childFrameNo, int childAngle) {
        // calculate number of frames between parent move and child move
        int frameDist = childFrameNo - parentFrameNo;
        
        // calculate how many steps are needed to rotate from the given
        // angle to this angle
        int thisAngle = childAngle
                - (childAngle % AngleByteRange.ANGLE_BYTE_STEP_SIZE)
                * AngleByteRange.ANGLE_BYTE_VALUE_COUNT;
        int otherAngle = parentAngle
                - (parentAngle % AngleByteRange.ANGLE_BYTE_STEP_SIZE)
                * AngleByteRange.ANGLE_BYTE_VALUE_COUNT;
        int stepDist = (thisAngle - otherAngle)
                / AngleByteRange.ANGLE_BYTE_STEP_SIZE;
        stepDist = Point.normalize(stepDist,
                AngleByteRange.ANGLE_BYTE_VALUE_COUNT);
        stepDist = Math.abs(stepDist);
        
        // this move is reachable, if the number of needed steps for the 
        // rotation is less than or equal to the number of available frames
        boolean isReachable = frameDist >= stepDist; 
        return isReachable;
    }
    
    /**
     * @return the shotFrameNo
     */
    public final int getShotFrameNo() {
        return m_shotFrameNo;
    }

    /**
     * @return the impactFrameNo
     */
    public final int getImpactFrameNo() {
        return m_impactFrameNo;
    }

    /**
     * @return the angle
     */
    public final int getAngle() {
        return m_angle;
    }

    /**
     * @return the targetId
     */
    public final int getTargetId() {
        return m_targetId;
    }

    /**
     * @return the targetSize
     */
    public final int getTargetSize() {
        return m_targetSize;
    }

    /**
     * @return the shotDistance
     */
    public final int getShotDistance() {
        return m_shotDistance;
    }

    /**
     * @return the collisionFrame
     */
    public final int getCollisionFrameNo() {
        return m_collisionFrameNo;
    }

    /* (non-Javadoc)
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    public int compareTo(Move o) {
        // shoot as soon as possible
        int result = m_shotFrameNo - o.m_shotFrameNo;
        if (result == 0) {

            // prefer targets that are destroyed early
            result = m_impactFrameNo - o.m_impactFrameNo;
            if (result == 0) {
                
                // use target id to avoid equal moves 
                result = o.m_targetId - m_targetId;
            }
        }
        
        return result;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        int hash = m_angle;
        hash += m_shotFrameNo * AngleByteRange.ANGLE_BYTE_VALUE_COUNT;
        hash += m_targetId * AngleByteRange.ANGLE_BYTE_VALUE_COUNT * 1501;
        return hash;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Move) {
            Move other = (Move)obj;
            return (m_shotFrameNo == other.m_shotFrameNo)
                    && (m_angle == other.m_angle)
                    && (m_targetId == other.m_targetId);
        } else {
            return super.equals(obj);
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[angle=");
        sb.append(m_angle);
        sb.append(" shotFrameNo=");
        sb.append(m_shotFrameNo);
        sb.append(" impactFrameNo=");
        sb.append(m_impactFrameNo);
        sb.append(" targetId=");
        sb.append(m_targetId);
        sb.append("]");
        return sb.toString();
    }
}
