/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.client;

/**
 * Stores the commands that the bot sends to mame server. Provides 
 * an interface to change the command states and to retrieve the 
 * <em>key byte</em>, which can be sent to the server. 
 * 
 * @author Henning Faber
 */
public class Commands {

    // constants for the key bits within the key byte
    private static final int KEY_NOCOMMANDS = 0x00;
    private static final int KEY_HYPERSPACE = 0x01; 
    private static final int KEY_FIRE = 0x02; 
    private static final int KEY_THRUST = 0x04;
    private static final int KEY_RIGHT = 0x08; 
    private static final int KEY_LEFT = 0x10; 
    private static final int KEY_START = 0x20; 

    /**
     * The key byte that holds the pressed keys as a combination of the
     * KEY_XXX constants.
     */
    private int m_keys;
    
    /**
     * Creates a commands object that has no keys pressed initially.
     */
    public Commands() {
        super();
        clear();
    }
    
    /**
     * Releases all keys. 
     */
    public final void clear() {
        m_keys = KEY_NOCOMMANDS;
    }
    
    /**
     * Presses or releases the <em>hyperspace</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressHyperspace(boolean b) {
        setKey(KEY_HYPERSPACE, b);
    }

    /**
     * Presses or releases the <em>fire</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressFire(boolean b) {
        setKey(KEY_FIRE, b);
    }

    /**
     * Presses or releases the <em>thrust</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressThrust(boolean b) {
        setKey(KEY_THRUST, b);
    }
    
    /**
     * Presses or releases the <em>left</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressLeft(boolean b) {
        setKey(KEY_LEFT, b);
    }
    
    /**
     * Presses or releases the <em>right</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressRight(boolean b) {
        setKey(KEY_RIGHT, b);
    }

    /**
     * Presses or releases the <em>start</em> key.
     * 
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    public final void pressStart(boolean b) {
        setKey(KEY_START, b);
    }

    /**
     * Returns the state of the <em>hyperspace</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isHyperspacePressed() {
        return getKey(KEY_HYPERSPACE);
    }

    /**
     * Returns the state of the <em>fire</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isFirePressed() {
        return getKey(KEY_FIRE);
    }

    /**
     * Returns the state of the <em>thrust</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isThrustPressed() {
        return getKey(KEY_THRUST);
    }
    
    /**
     * Returns the state of the <em>left</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isLeftPressed() {
        return getKey(KEY_LEFT);
    }
    
    /**
     * Returns the state of the <em>right</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isRightPressed() {
        return getKey(KEY_RIGHT);
    }

    /**
     * Returns the state of the <em>start</em> key. 
     * 
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    public final boolean isStartPressed() {
        return getKey(KEY_START);
    }
    
    /**
     * @return the keys
     */
    public final byte getCommands() {
        return (byte)m_keys;
    }
    
    /**
     * Determines if the given key is pressed.
     * 
     * @param key the key, given as one of the <code>KEY_XXX</code> constants
     * @return <code>true</code>, if the key is pressed, <code>false</code>,
     *  if not
     */
    private boolean getKey(int key) {
        return (m_keys & key) == key;
    }

    /**
     * Presses or releases the given key. 
     * 
     * @param key the key to press or release, given as on eof the 
     *  <code>KEY_XXX</code> constants
     * @param b <code>true</code> to press the key, <code>false</code> to
     *  release it
     */
    private void setKey(int key, boolean b) {
        if (b) {
            m_keys |= key;
        } else {
            m_keys &= ~key;
        }
    }
}
