/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.game.field.Screen;


/**
 * An asteroid.
 * 
 * @author Henning Faber
 */
public class Asteroid extends ScaleableGameObject implements
        ITypeableGameObject {

    /**
     * The LOGGER instance. 
     */
    private static final Logger LOGGER = Logger.getLogger(Asteroid.class);
    
    /**
     * The scale factor for a large asteroid. This may be used in
     * connection with {{@link #getScaleFactor()}.
     */
    private static final int LARGE_ASTEROID = 0;

    /**
     * The scale factor for a middle size asteroid. This may be 
     * used in connection with {{@link #getScaleFactor()}.
     */
    private static final int MIDDLE_ASTEROID = 15;

    /**
     * The scale factor for a small asteroid. This may be used in 
     * connection with {{@link #getScaleFactor()}.
     */
    private static final int SMALL_ASTEROID = 14;

    /**
     * The maximum square distance that an asteroid is expected to move
     * from one frame to another.
     */
    private static final int MAX_ASTEROID_FRAME_DIST = 
        6 * Screen.INTERNAL_PRECISION * 6 * Screen.INTERNAL_PRECISION;

    /**
     * The square radius of a large asteroid measured in sub pixels.
     */
    public static final int LARGE_ASTEROID_SQUARE_RADIUS = 35 * 35
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    /**
     * The square radius of a middle size asteroid measured in sub pixels.
     */
    public static final int MIDDLE_ASTEROID_SQUARE_RADIUS = 16 * 16
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    /**
     * The square distance of a small asteroid measured in sub pixels.
     */
    public static final int SMALL_ASTEROID_SQUARE_RADIUS = 8 * 8 
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    // asteroid types
    public static final int TYPE_1 = 1;
    public static final int TYPE_2 = 2;
    public static final int TYPE_3 = 3;
    public static final int TYPE_4 = 4;
    
    /**
     * The type of the asteroid.
     */
    private int m_type;

    /**
     * Creates an asteroid of the given type with the given location and
     * size.
     * 
     * @param x the x coordinate of the asteroid
     * @param y the y coordinate of the asteroid
     * @param size the size of the asteroid
     * @param type the type of teh asteroid
     */
    public Asteroid(int x, int y, int size, int type) {
        super(x, y, size);
        m_type = type;
    }
    
    /**
     * Creates an asteroid that represents what the given asteroid is 
     * expected to be like in the given number of frames.
     * 
     * @param toProject the asteroid to project
     * @param frameCount the number of frames to project
     */
    public Asteroid(Asteroid toProject, int frameCount) {
        super(toProject, frameCount);
        m_type = toProject.m_type;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getPrecision()
     */
    @Override
    protected final int getPrecision() {
        return Screen.INTERNAL_PRECISION;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getMaxFrameDist()
     */
    @Override
    public final int getMaxFrameDist() {
        return MAX_ASTEROID_FRAME_DIST;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.gameobjects.ScaleableGameObject#getDistCorrection()
     */
    @Override
    public final int getSquareRadius() {
        int sf = getScaleFactor();
        switch (sf) {
            case LARGE_ASTEROID:
                return LARGE_ASTEROID_SQUARE_RADIUS; 
            case MIDDLE_ASTEROID:
                return MIDDLE_ASTEROID_SQUARE_RADIUS;
            case SMALL_ASTEROID:
                return SMALL_ASTEROID_SQUARE_RADIUS;
            default: // should not happen!
                LOGGER.warn("Unable to determine correct asteroid radius for "
                        + "asteroid with scale factor " + sf);
                return 0;
        }
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#trackCustomProperties(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    @Override
    protected void trackCustomProperties(GameObject prev, int frameGap) {
        // not used
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.ITypeableGameObject#getType()
     */
    public final int getType() {
        return m_type;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.gameobjects.ScaleableGameObject#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[id=");
        sb.append(getId());
        sb.append(" x=");
        sb.append(getRelativeLocation().getX());
        sb.append(" y=");
        sb.append(getRelativeLocation().getY());
        sb.append(" size=");
        sb.append(getScaleFactor());
        sb.append(" type=");
        sb.append(getType());
        sb.append("]");
        return sb.toString();
    }
}
