/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

package de.hfaber.asteroids.game.objects;

/**
 * An explosion, which results from a destroyed asteroid. An explosion
 * is never changing its location and has thus always a direction vector 
 * of (0,0). It defines a <em>time to life</em> (ttl), which is the
 * remaining number of frames, before it will disappear.
 * 
 * @author Henning Faber
 */
public class Explosion extends GameObject {

    /**
     * The number of frames for a full explosion animation.
     */
    public static final int EXPLOSION_LIFE_TIME = 37;
    
    /**
     * The remaining number of frames, before the explosion will disappear.
     */
    private int m_ttl;
    
    /**
     * Creates an explosion.
     * 
     * @param x the x co-ordinate of the explosion
     * @param y the y co-ordinate of the explosion
     */
    public Explosion(int x, int y) {
        super(x, y);
        m_ttl = EXPLOSION_LIFE_TIME;
    }

    /**
     * Creates an explosion that represents what the given explosion is 
     * expected to be like in the given number of frames. Since the 
     * explosion is never chaning its location, the 
     * 
     * @param toProject
     * @param frameCount
     */
    public Explosion(GameObject toProject, int frameCount) {
        super(toProject, frameCount);
        m_ttl -= frameCount;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#trackCustomProperties(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    @Override
    protected final void trackCustomProperties(GameObject prev, int frameGap) {
        Explosion prevExplosion = (Explosion)prev;
        m_ttl = prevExplosion.m_ttl - frameGap;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#getMaxFrameDist()
     */
    @Override
    public final int getMaxFrameDist() {
        // not used
        return 0;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#getPrecision()
     */
    @Override
    protected final int getPrecision() {
        // not used
        return 1;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#getSqaureRadius()
     */
    @Override
    public final int getSquareRadius() {
        // not used
        return 0;
    }

    /**
     * @return the ttl
     */
    public final int getTtl() {
        return m_ttl;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[id=");
        sb.append(getId());
        sb.append(" x=");
        sb.append(getRelativeLocation().getX());
        sb.append(" y=");
        sb.append(getRelativeLocation().getY());
        sb.append(" ttl=");
        sb.append(m_ttl);
        sb.append("]");
        return sb.toString();
    }
}
