/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.game.field.Screen;


/**
 * A saucer.
 * 
 * @author Henning Faber
 */
public class Saucer extends ScaleableGameObject {

    /**
     * The LOGGER instance. 
     */
    private static final Logger LOGGER = Logger.getLogger(Saucer.class);
    
    /**
     * The scale factor for a large saucer. This may be used in
     * connection with {{@link #getScaleFactor()}.
     */
    public static final int LARGE_SAUCER = 15;

    /**
     * The scale factor for a small saucer. This may be used in
     * connection with {{@link #getScaleFactor()}.
     */
    public static final int SMALL_SAUCER = 14;
    
    /**
     * The maximum square distance that a saucer is expected to move 
     * from one frame to another.
     */
    private static final int MAX_SAUCER_FRAME_DIST = 2 
        * Screen.INTERNAL_PRECISION * 2 * Screen.INTERNAL_PRECISION;

    /**
     * The square radius of a large saucer measured in sub pixels.
     */
    private static final int LARGE_SAUCER_SQUARE_RADIUS = 13 * 8
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    /**
     * The square radius of a small saucer measured in sub pixels.
     */
    private static final int SMALL_SAUCER_SQUARE_RADUIS = 7 * 4
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    /**
     * Creates a saucer with the given location and size.
     * 
     * @param x
     *            the x coordinate of the saucer
     * @param y
     *            the y coordinate of the saucer
     * @param size
     *            the size of the saucer
     */
    public Saucer(int x, int y, int size) {
        super(x, y, size);
    }
    
    /**
     * Creates a saucer that represents what the given saucer is 
     * expected to be like in the given number of frames.
     * 
     * @param toProject the saucer object to project
     * @param frameCount the number of frames to project
     */
    public Saucer(Saucer toProject, int frameCount) {
        super(toProject, frameCount);
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getPrecision()
     */
    @Override
    protected final int getPrecision() {
        return 1;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getMaxFrameDist()
     */
    @Override
    public final int getMaxFrameDist() {
        // NOTE: Although this method returns the correct value for the
        //       maximum saucer frame distance, it is not used by the
        //       actual matching algorithm, since there is a maximum of one
        //       saucer per game status and thus always a one-to-one match.  
        return MAX_SAUCER_FRAME_DIST;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.gameobjects.ScaleableGameObject#getDistCorrection()
     */
    @Override
    public final int getSquareRadius() {
        int sf = getScaleFactor();
        switch (sf) {   
            case LARGE_SAUCER:
                return LARGE_SAUCER_SQUARE_RADIUS;
            case SMALL_SAUCER:
                return SMALL_SAUCER_SQUARE_RADUIS;
            default: // should not happen!
                LOGGER.warn("Unable to determine correct saucer radius for "
                        + "saucer with scale factor " + sf);
                return 0;
        }
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#trackCustomProperties(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    @Override
    protected void trackCustomProperties(GameObject prev, int frameGap) {
        // not used
    }
}
