/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import de.hfaber.asteroids.game.field.Point;

/**
 * Look up table for shot vectors of a <em>stationary</em> ship.
 * This is used to improve performance and avoid re-calculating
 * the shot vector several times during runtime.
 * 
 * @author Henning Faber
 */
public final class ShotVector {

    public static final Point[] TABLE = {
        new Point(63, 0),
        new Point(63, 1),
        new Point(63, 3),
        new Point(63, 4),
        new Point(63, 6),
        new Point(63, 8),
        new Point(63, 9),
        new Point(62, 11),
        new Point(62, 12),
        new Point(62, 14),
        new Point(61, 15),
        new Point(61, 17),
        new Point(61, 18),
        new Point(60, 20),
        new Point(60, 21),
        new Point(59, 23),
        new Point(58, 24),
        new Point(58, 25),
        new Point(57, 27),
        new Point(56, 28),
        new Point(56, 30),
        new Point(55, 31),
        new Point(54, 32),
        new Point(53, 34),
        new Point(53, 35),
        new Point(52, 36),
        new Point(51, 38),
        new Point(50, 39),
        new Point(49, 40),
        new Point(48, 41),
        new Point(47, 42),
        new Point(46, 44),
        new Point(45, 45),
        new Point(44, 46),
        new Point(42, 47),
        new Point(41, 48),
        new Point(40, 49),
        new Point(39, 50),
        new Point(38, 51),
        new Point(36, 52),
        new Point(35, 53),
        new Point(34, 53),
        new Point(32, 54),
        new Point(31, 55),
        new Point(30, 56),
        new Point(28, 56),
        new Point(27, 57),
        new Point(25, 58),
        new Point(24, 58),
        new Point(23, 59),
        new Point(21, 60),
        new Point(20, 60),
        new Point(18, 61),
        new Point(17, 61),
        new Point(15, 61),
        new Point(14, 62),
        new Point(12, 62),
        new Point(11, 62),
        new Point(9, 63),
        new Point(8, 63),
        new Point(6, 63),
        new Point(4, 63),
        new Point(3, 63),
        new Point(1, 63),
        new Point(0, 63),
        new Point(-1, 63),
        new Point(-3, 63),
        new Point(-4, 63),
        new Point(-6, 63),
        new Point(-8, 63),
        new Point(-9, 63),
        new Point(-11, 62),
        new Point(-12, 62),
        new Point(-14, 62),
        new Point(-15, 61),
        new Point(-17, 61),
        new Point(-18, 61),
        new Point(-20, 60),
        new Point(-21, 60),
        new Point(-23, 59),
        new Point(-24, 58),
        new Point(-25, 58),
        new Point(-27, 57),
        new Point(-28, 56),
        new Point(-30, 56),
        new Point(-31, 55),
        new Point(-32, 54),
        new Point(-34, 53),
        new Point(-35, 53),
        new Point(-36, 52),
        new Point(-38, 51),
        new Point(-39, 50),
        new Point(-40, 49),
        new Point(-41, 48),
        new Point(-42, 47),
        new Point(-44, 46),
        new Point(-45, 45),
        new Point(-46, 44),
        new Point(-47, 42),
        new Point(-48, 41),
        new Point(-49, 40),
        new Point(-50, 39),
        new Point(-51, 38),
        new Point(-52, 36),
        new Point(-53, 35),
        new Point(-53, 34),
        new Point(-54, 32),
        new Point(-55, 31),
        new Point(-56, 30),
        new Point(-56, 28),
        new Point(-57, 27),
        new Point(-58, 25),
        new Point(-58, 24),
        new Point(-59, 23),
        new Point(-60, 21),
        new Point(-60, 20),
        new Point(-61, 18),
        new Point(-61, 17),
        new Point(-61, 15),
        new Point(-62, 14),
        new Point(-62, 12),
        new Point(-62, 11),
        new Point(-63, 9),
        new Point(-63, 8),
        new Point(-63, 6),
        new Point(-63, 4),
        new Point(-63, 3),
        new Point(-63, 1),
        new Point(-63, 0),
        new Point(-63, -1),
        new Point(-63, -3),
        new Point(-63, -4),
        new Point(-63, -6),
        new Point(-63, -8),
        new Point(-63, -9),
        new Point(-62, -11),
        new Point(-62, -12),
        new Point(-62, -14),
        new Point(-61, -15),
        new Point(-61, -17),
        new Point(-61, -18),
        new Point(-60, -20),
        new Point(-60, -21),
        new Point(-59, -23),
        new Point(-58, -24),
        new Point(-58, -25),
        new Point(-57, -27),
        new Point(-56, -28),
        new Point(-56, -30),
        new Point(-55, -31),
        new Point(-54, -32),
        new Point(-53, -34),
        new Point(-53, -35),
        new Point(-52, -36),
        new Point(-51, -38),
        new Point(-50, -39),
        new Point(-49, -40),
        new Point(-48, -41),
        new Point(-47, -42),
        new Point(-46, -44),
        new Point(-45, -45),
        new Point(-44, -46),
        new Point(-42, -47),
        new Point(-41, -48),
        new Point(-40, -49),
        new Point(-39, -50),
        new Point(-38, -51),
        new Point(-36, -52),
        new Point(-35, -53),
        new Point(-34, -53),
        new Point(-32, -54),
        new Point(-31, -55),
        new Point(-30, -56),
        new Point(-28, -56),
        new Point(-27, -57),
        new Point(-25, -58),
        new Point(-24, -58),
        new Point(-23, -59),
        new Point(-21, -60),
        new Point(-20, -60),
        new Point(-18, -61),
        new Point(-17, -61),
        new Point(-15, -61),
        new Point(-14, -62),
        new Point(-12, -62),
        new Point(-11, -62),
        new Point(-9, -63),
        new Point(-8, -63),
        new Point(-6, -63),
        new Point(-4, -63),
        new Point(-3, -63),
        new Point(-1, -63),
        new Point(0, -63),
        new Point(1, -63),
        new Point(3, -63),
        new Point(4, -63),
        new Point(6, -63),
        new Point(8, -63),
        new Point(9, -63),
        new Point(11, -62),
        new Point(12, -62),
        new Point(14, -62),
        new Point(15, -61),
        new Point(17, -61),
        new Point(18, -61),
        new Point(20, -60),
        new Point(21, -60),
        new Point(23, -59),
        new Point(24, -58),
        new Point(25, -58),
        new Point(27, -57),
        new Point(28, -56),
        new Point(30, -56),
        new Point(31, -55),
        new Point(32, -54),
        new Point(34, -53),
        new Point(35, -53),
        new Point(36, -52),
        new Point(38, -51),
        new Point(39, -50),
        new Point(40, -49),
        new Point(41, -48),
        new Point(42, -47),
        new Point(44, -46),
        new Point(45, -45),
        new Point(46, -44),
        new Point(47, -42),
        new Point(48, -41),
        new Point(49, -40),
        new Point(50, -39),
        new Point(51, -38),
        new Point(52, -36),
        new Point(53, -35),
        new Point(53, -34),
        new Point(54, -32),
        new Point(55, -31),
        new Point(56, -30),
        new Point(56, -28),
        new Point(57, -27),
        new Point(58, -25),
        new Point(58, -24),
        new Point(59, -23),
        new Point(60, -21),
        new Point(60, -20),
        new Point(61, -18),
        new Point(61, -17),
        new Point(61, -15),
        new Point(62, -14),
        new Point(62, -12),
        new Point(62, -11),
        new Point(63, -9),
        new Point(63, -8),
        new Point(63, -6),
        new Point(63, -4),
        new Point(63, -3),
        new Point(63, -1)
    };

    /**
     * Returns the shot vector that applies for the given angle byte.
     * 
     * @param angleByte the angle byte for which the shot vector
     *  should be calculated
     * @return the shot vector
     */
    public static final Point getShotVector(int angleByte) {
        double angle = Math.PI * angleByte / 128;
        int astsin = (int)Math.round(127 * Math.sin(angle));
        int astcos = (int)Math.round(127 * Math.cos(angle));
        int fx = (int)(Math.floor(astcos / 2));
        int fy = (int)(Math.floor(astsin / 2));
        
        // return the shot direction vector
        Point shotDirection = new Point(fx, fy);
        return shotDirection;
    }
    
    /**
     * Main method that outputs the code block for the shot vector table.
     * 
     * @param args argument list
     */
    public static void main(String[] args) {
        System.out.println("    public static final Point[] TABLE = {");
        for (int angle = 0; angle < 256; angle++) {
            Point a = getShotVector(angle);
            System.out.print("        new Point(");
            System.out.print(a.getX());
            System.out.print(", ");
            System.out.print(a.getY());
            System.out.print(")");
            System.out.println(angle == 255 ? "" : ",");
        }
        System.out.println("    };");
    }
}
