﻿/*
 * Asteroids Player Client 
 * 2008 ct-Wettbewerb
 * 
 * by Stefc (stefan.boether@gmail.com) */

using System;
using System.Linq;
using System.Collections.Generic;
using System.Collections.ObjectModel;


namespace stefc.asteroids
{
    
    class GameFrame
    {
        private List<SpaceObject> asteroids;
        private List<Shot> shots;
        private int velocity;
        
        public GameFrame(IGameState gameState)
        {
            asteroids = new List<SpaceObject>();
            shots = new List<Shot>();
            velocity = 0;

            Ufo = null;
            Score = 0;
            Ships = 0;

            GameState = gameState;
        }

        
        public int Score
        {
            get; set;
        }

        public int Velocity
        {
            get { return velocity; }
        }

        public IGameState GameState
        {
            get;
            private set;
        }

        public int Ships
        {
            get;
            set;
        }

        public Ufo Ufo
        {
            get;
            set;
        }

        public Ship Ship
        {
            get;
            set;
        }

        public bool IsOutOfSync(int winkelByte)
        {
            if (Ship == null)
                return true;
            return GameUtils.IsOutOfSync(winkelByte, new Point((int)Ship.Dx, (int)Ship.Dy));
        }

        public int Resync(int winkelByte)
        {
            if (Ship == null)
                return -1;
            Point pt = new Point((int)Ship.Dx, (int)Ship.Dy);
            return GameUtils.Resync(winkelByte, pt);
        }

        public void Add(Asteroid asteroid)
        {
            asteroids.Add(asteroid);
        }

        private void Delete(Asteroid asteroid)
        {
            asteroids.Remove(asteroid);
        }

        public void Add(Shot shot)
        {
            shots.Add(shot);
        }

        /// <summary>
        /// Koordinatensystem normalisieren
        /// Das Raumschiff befindet sich im Mittelpunkt an (0,0)
        /// </summary>
        public void Normalize()
        {
            foreach (Asteroid asteroid in asteroids)
                Normalize(asteroid);

            foreach (Shot shot in shots)
                Normalize(shot);

            if (Ufo != null)
                Normalize(Ufo);

            asteroids.Sort();
        }

        /// <summary>
        /// Koordinate relativ zum Raumschiff normalisieren
        /// </summary>
        /// <param name="spaceObject"></param>
        private void Normalize(SpaceObject spaceObject)
        {
            int dx = spaceObject.X - Ship.X; ;
            while (dx < -(Const.WIDTH/2)) dx += Const.WIDTH; // dx normalisieren auf -512 ... 511
            while (dx >= (Const.WIDTH/2)) dx -= Const.WIDTH;

            int dy = spaceObject.Y - Ship.Y;
            while (dy < -(Const.HEIGHT/2)) dy += Const.HEIGHT;  // dy normalisieren auf -384 ... 383
            while (dy >= (Const.HEIGHT/2)) dy -= Const.HEIGHT;

            spaceObject.X = dx;
            spaceObject.Y = dy;
        }

        private IList<SpaceObject> GetDangerous()
        {
            List<SpaceObject> result = new List<SpaceObject>();

            if (Ufo != null)
                result.Add(Ufo);

            foreach (Asteroid asteroid in asteroids)
            {

                Vector a = asteroid.Position;
                Vector b = asteroid.Velocity;

                // if (a.Length < (Const.SHOT_DISTANCE-100)) // ist er überhaupt im Schussradius ?
                {

                    // Console.Write(String.Format("({0},{1})-({2:F2},{3:F2}) ", a.X, a.Y, asteroid.Dx, asteroid.Dy));

                    double angle = MathUtils.GetHitAngle(a, Ship.GetRadius(), asteroid.GetRadius());

                    double aAngle = MathUtils.NormalizeAngle(a.Angle);
                    double bAngle = MathUtils.NormalizeAngle(b.Angle - 180);

                    double delta = MathUtils.AngleDelta(bAngle, aAngle);

                    if (Math.Abs(delta) <= angle)
                        if (b.Length > 0)
                        {
                            result.Add(asteroid);
                            // Console.Write("*");
                        }
                }
            }
            // Console.WriteLine();

            return result;
        }



        public Target SearchNearest(bool searchAll)
        {
            IList<SpaceObject> list = new List<SpaceObject>(asteroids);
            if (Ufo != null)
                list.Add(Ufo);
            return SearchNearest(list, searchAll);
        }

        public Target SearchDangerous()
        {
            return SearchNearest(GetDangerous(),true);
        }

        public IEnumerable<SpaceObject> Asteroids
        {
            get { return asteroids; }
        }

        private Target SearchNearest(IEnumerable<SpaceObject> objects, bool searchAll)
        {
            int srcAngle = GameState.WinkelByte;
            int minDelta = 256;
            int result = -1;
            bool resultDangerous = false;
            SpaceObject resultObject = null;

            foreach (SpaceObject spaceObject in objects)
            {
                if (!GameState.IsTargetBlocked(spaceObject))
                {
                    Vector target = GameUtils.GetTarget(spaceObject);
                    if (target != null)
                    {
                        int destAngle = GameUtils.GetNearestAngle(srcAngle,
                            GameUtils.GetBestAngle(target,spaceObject));
                        
                        if (destAngle != -1)
                        {
                            int delta = GameUtils.AngleByteDelta(srcAngle, destAngle);
                            bool isDangerous = true;
                            if (spaceObject is Asteroid)
                            {
                                // Objekte die weit weg sind werden später anvisiert
                                delta = delta + ((int)target.Length / spaceObject.GetRadius());
                                // delta = delta * (int)spaceObject.Velocity.Length;
                                isDangerous = GameUtils.IsDangerous(spaceObject, Ship);
                                if (isDangerous)
                                    delta = delta / 20;
                            }
                            else
                            {
                                delta = delta / 30;  // Ufo's bekommen eine höhere Piorität 
                            }

                            
                            //Console.Write(String.Format("-{0}[{1}] {2}[{3}] {4} ", 
                            //    srcAngle,  GameUtils.ByteToAngle(srcAngle), 
                            //    destAngle, GameUtils.ByteToAngle(destAngle), 
                            //    delta));

                            //if (((target.Length < (Const.SHOT_DISTANCE) || searchAll)) && (delta < minDelta))
                            if (delta < minDelta)
                            {
                                minDelta = delta;
                                result = destAngle;
                                resultObject = spaceObject;
                                resultDangerous = spaceObject is Ufo;
                            }
                            
                        }
                    }
                }
            }

            if (result == -1)
                return null;
            return new Target(result, resultObject, resultDangerous);
        }

        internal bool HasAsteroids
        {
            get { return asteroids.Count > 0; }
        }

        internal bool HasOnlyFewAsteroids
        {
            get { return asteroids.Count < 3; }
        }

        internal bool HasBigAsteroids
        {
            get
            {
                foreach (Asteroid asteroid in asteroids)
                    if (asteroid.Size == 0)
                        return true; 

                return false; 

            }
        }

        
        internal bool HyperSpaceJump
        {
            get
            {
                // kommt mir ein Asteroid zu nah? 
                if (HasAsteroids)
                {
                    double min = asteroids.Min<SpaceObject>((asteroid) =>
                    {
                        return asteroid.NextPosition.Length - (asteroid.GetRadius()*1.2);
                    });

                    if (min < Ship.GetRadius())
                    {
                        return true;
                    }
                }

                // rückt mir ein Ufo zu nah auf die Pelle?
                if (Ufo != null)
                    if (Ufo.NextPosition.Length - Ufo.GetRadius() < Ship.GetRadius())
                    {
                        return true;
                    }

                return false;
            }
        }

        /// <summary>
        /// Befindet sich ein Asteroid in der Blickrichtung des Raumschiffs ?
        /// </summary>
        internal bool HasAsteroidInView
        {
            get
            {
                // Blickrichtung 
                Vector source = Vector.CreateCartesian(Ship.Dx, Ship.Dy);

                foreach (Asteroid asteroid in asteroids)
                {
                    Vector target = GameUtils.GetTarget(asteroid);
                    if (target != null)
                    {
                        // if (target.Length < Const.SHOT_DISTANCE)
                        {
                            double delta = MathUtils.AngleDelta(target.Angle, source.Angle);
                            if (Math.Abs(delta) < 10)
                                return true;
                        }
                    }
                }
                return false;
            }
        }

        /// <summary>
        /// Bewegungsvektoren der Ufos und Asteroiden ermitteln
        /// Differenz zwischen zwei Frames
        /// </summary>
        /// <param name="prevGame">Vorheriger Frame</param>
        internal void MotionDetection(GameFrame prevGame, int latency)
        {
            // Bewegung des Schiffs ermitteln
            if (Ship != null && prevGame.Ship != null)
            {
                Point delta = Ship.GetDelta(prevGame.Ship);
                velocity = delta.X * delta.X + delta.Y * delta.Y;
            }
            else
                velocity = 0;

            // Bewegung des Ufos ermitteln
            if (Ufo != null)
                if (prevGame.Ufo != null)
                {
                    Ufo.SetPrevious(prevGame.Ufo, latency);
                }
                else
                    Ufo.Id = SpaceObject.NextId;

            foreach (Asteroid asteroid in asteroids)
            {
                int minDist = int.MaxValue;
                Asteroid match = null;
                foreach (Asteroid prevAsteroid in prevGame.GetMatched(asteroid))
                {
                    Point delta = asteroid.GetDelta(prevAsteroid);
                    int dist = delta.X * delta.X + delta.Y * delta.Y;
                    if (dist < minDist)
                    {
                        minDist = dist;
                        match = prevAsteroid;
                    }
                }
                if (match != null)
                {
                    prevGame.Delete(match);
                    asteroid.SetPrevious(match,latency);
                }
                else
                    asteroid.Id = SpaceObject.NextId;
            }
        }

        internal void GenerateIds()
        {
            foreach (Asteroid asteroid in asteroids)
                asteroid.Id = SpaceObject.NextId;

            if (Ufo != null)
                Ufo.Id = SpaceObject.NextId;
        }

        private IEnumerable<SpaceObject> GetMatched(SpaceObject other)
        {
            IEnumerable<SpaceObject> result = from asteroid in asteroids where asteroid.CompareTo(other) == 0 select asteroid;
            return result; 
        }

        
        
    }
}
