﻿/*
 * Asteroids Player Client 
 * 2008 ct-Wettbewerb
 * 
 * by Stefc (stefan.boether@gmail.com) */

using System;
using System.IO;


namespace stefc.asteroids
{
    public class Operands
    {
        private int dx;
        private int dy;
        private int sf;
        private int vz;

        private Operands(int dx, int dy, int sf, int vz)
        {
            this.dx = dx;
            this.dy = dy;
            this.sf = sf;
            this.vz = vz;
        }

        public int Dx
        {
            get { return dx; }
        }

        public int Dy
        {
            get { return dy; }
        }
        
        public int Sf
        {
            get { return sf; }
        }
        
        public int Vz
        {
            get { return vz; }
        }

        public static Operands Create()
        {
            return new Operands(0, 0, 0, 0);
        }

        public static Operands CreateLABS(UInt16 word1, UInt16 word2)
        {
            return new Operands((UInt16)(word2 & 0x3FF), (UInt16)(word1 & 0x3FF),0,(UInt16)(word2 >> 12));
        }

        public static Operands CreateSVEC(UInt16 word1)
        {
            Int16 dy = (Int16)(word1 & 0x300);
            if ((word1 & 0x400) != 0)
                dy = (Int16)(-dy);
            Int16 dx = (Int16)((word1 & 0x3) << 8);
            if ((word1 & 0x4) != 0)
                dx = (Int16)(-dx);
            int sf = ((word1 & 8) >> 2) | ((word1 & 0x800) >> 11);
            sf += 2;
            int vz = (byte)((word1 & 0xF0) >> 4);

            return new Operands(dx, dy, sf, vz);
        }

        public static Operands CreateVCTR(UInt16 word1, UInt16 word2)
        {
            Int16 dy = (Int16)(word1 & 0x3FF);
            if ((word1 & 0x400) != 0)
                dy = (Int16)(-dy);
            Int16 dx = (Int16)(word2 & 0x3FF);
            if ((word2 & 0x400) != 0)
                dx = (Int16)(-dx);
            int sf = (word1 >> 12);
            int vz = (word2 >> 12);

            return new Operands(dx, dy, sf, vz);
        }


        public bool IsScore
        {
            get
            {
                return (dx == 100) && (dy == 876) && (vz == 1);
            }
        }


    }

           
    /// <summary>
    /// Parse the vector Operation
    /// </summary>
    class VectorOperation
    {
        private UInt16 word1;
        private UInt16 word2;
        private byte op;
        private Operands operands;

        public VectorOperation(BinaryReader reader)
        {
            word1 = reader.ReadUInt16();
            op = (byte)(word1 >> 12);
                
            word2 = 0;
            if ((OpCode == OpCodes.LABS) || (OpCode == OpCodes.VCTR))
                word2 = reader.ReadUInt16();

            switch (OpCode)
            {
                case OpCodes.LABS:
                    operands = Operands.CreateLABS(word1, word2);
                    break;
                case OpCodes.VCTR:
                    operands = Operands.CreateVCTR(word1, word2);
                    break;
                case OpCodes.SVEC:
                    operands = Operands.CreateSVEC(word1);
                    break;

                default:
                    break;
            }

        }

        public override string ToString()
        {
            if ((OpCode == OpCodes.LABS) || (OpCode == OpCodes.VCTR))
                return String.Format("{0:X4} {1:X4}  {2:} {3:10} {4}", word1, word2, Operator, OperandsToString, Comment);
            else
                return String.Format("{0:X4}       {1} {2:10} {3}", word1, Operator, OperandsToString, Comment);
        }

        public Operands Operands
        {
            get { return operands; }
        }

        public OpCodes OpCode
        {
            get 
            { 
                if (op <= (byte)OpCodes.VCTR)
                    return OpCodes.VCTR;
                else
                    return (OpCodes)op;
            }
        }

        private string Operator
        {
            get
            {
                switch (OpCode)
                {
                    case OpCodes.VCTR: 
                        return "VCTR";
                    case OpCodes.LABS:
                        return "LABS";
                    case OpCodes.HALT:
                        return "HALT";
                    case OpCodes.JSRL:
                        return "JSRL";
                    case OpCodes.RTSL:
                        return "RTSL";
                    case OpCodes.JMPL:
                        return "JMPL";
                    case OpCodes.SVEC:
                        return "SVEC";

                    default:
                        return "????";
                }
            }
        }

        private string OperandsToString
        {
            get 
            {
                if (OpCode == OpCodes.LABS)
                {
                    return String.Format("({0}, {1}), s{2}", operands.Dx,operands.Dy,operands.Vz);
                }

                if (OpCode == OpCodes.SVEC || OpCode == OpCodes.VCTR)
                {
                    return String.Format("({0}, {1}), s{2}, z{3}", operands.Dx, operands.Dy, operands.Sf, operands.Vz);
                }

                if ((OpCode == OpCodes.JMPL) || (OpCode == OpCodes.JSRL))
                    return String.Format("${0:X3}", (UInt16)(word1 & 0xFFF));
                
                return String.Empty; 
            } 
        }

        
    

        public char ToChar(UInt16 addr)
        {
            switch (addr)
            {
                case 0xB2C:
                    return ' ';
                case 0xADD:
                    return '0';
                case 0xB2E:
                    return '1';
                case 0xB32:
                    return '2';
                case 0xB3A:
                    return '3';
                case 0xB41:
                    return '4';
                case 0xB48:
                    return '5';
                case 0xB4F:
                    return '6';
                case 0xB56:
                    return '7';
                case 0xB5B:
                    return '8';
                case 0xB63:
                    return '9';

                case 0xA78:
                    return 'A';
                case 0xA80:
                    return 'B';
                case 0xA8D:
                    return 'C';
                case 0xA93:
                    return 'D';
                case 0xA9B:
                    return 'E';
                case 0xAAA:
                    return 'G';
                case 0xAB3:
                    return 'H';
                case 0xACD:
                    return 'L';
                case 0xABA:
                    return 'I';
                case 0xAD8:
                    return 'N';
                case 0xAE3:
                    return 'P';
                case 0xAC7:
                    return 'K';
                case 0xAF3:
                    return 'R';
                case 0xAFB:
                    return 'S';
                case 0xB02:
                    return 'T';
                case 0xB08:
                    return 'U';
                case 0xB13:
                    return 'W';
                case 0xB1F:
                    return 'Y';
                case 0xB26:
                    return 'Z';

                default:
                    return '?';
            }
        }

        public bool IsDigit
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return (addr == 0xADD) || (addr == 0xB2C) || ((addr >= 0xB2E) && (addr <= 0xB63));
            }
        }

        public bool IsCharacter
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return ((addr >= 0xA78) && (addr <= 0xB26));
            }
        }

        public char Digit
        {
            get
            {
                return ToChar((UInt16)(word1 & 0xFFF));
            }
        }

        public bool IsLive
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return (addr == 0xA6D);
            }
        }

        public bool IsUfo
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return (addr == 0x929);
            }
        }

        public bool IsCopyright
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return (addr == 0x852);
            }
        }


        public bool IsAsteroid
        {
            get 
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                return (addr == 0x8f3 || addr == 0x8ff || addr == 0x90d || addr == 0x91a);        
            }
        }

        public bool IsShot
        {
            get
            {
                return (operands.Dx == 0 && operands.Dy == 0 && operands.Vz == 15);
            }
        }

        public bool IsShip
        {
            get
            {
                return (op == 6 && operands.Vz == 12 && operands.Dx != 0 && operands.Dy != 0);
            }
        }
        public int AsteroidType
        {
            get
            {
                UInt16 addr = (UInt16)(word1 & 0xFFF);
                if (addr == 0x8f3) return 1;
                if (addr == 0x8ff) return 2;
                if (addr == 0x90d) return 3;
                if (addr == 0x91a) return 4;
                return 0;
            }

        }

        private string Comment
        {
            get {

                if ((OpCode == OpCodes.LABS) || (OpCode == OpCodes.JMPL))
                    return String.Empty;

                if (OpCode == OpCodes.JSRL)
                {
                    UInt16 addr = (UInt16)(word1 & 0xFFF);
                    switch (addr)
                    {
                        case 0x8f3:
				            return "; Metorid Typ 1";
			            case 0x8ff:
                            return "; Metorid Typ 2";
            			case 0x90d:
                            return "; Metorid Typ 3";
				        case 0x91a:
                            return "; Metorid Typ 4";
			            case 0x929:
                            return "; UFO";
                        case 0xA6D: 
                            return "; Schiff";
                        case 0x852:
                            return "; Copyright";
                        case 0xB2C:
                            return String.Empty;

                        case 0x880:
                        case 0x896:
                        case 0x8B5:
                        case 0x8D0: 
                            return "; Explosion";
                            
                        default:
                            return "; " + ToChar(addr);
                    }
                }

                return String.Empty; 
            }

        }
            
    }
}
