﻿using System;
using System.Windows;
using System.Collections.Generic;
using System.Text;

namespace ct.Asteroid
{
    // Max speed
    // ------------------------------------------
    //   Ufo      = (2,2)
    //   Ship     = (6,6)
    //   Shot     = (8,8) + speed of Ship or Ufo
    //   Asteroid = (4,4)

    // Max size
    // ------------------------------------------
    //   Ufo      = (20,14), (40,25) 
    //   Ship     = (18,26)
    //   Asteroid = (16,16), (32,32), (64,64)


    public abstract class TOrb
    {
        public enum TOrbType {otAsteroid, otShip, otUfo, otShot};
        
        public static Vector CNullVector = new Vector(0, 0);
        
        protected int OrbSize { get; set; }         // 0=large, 15=medium, 14=small 
        
        public const int COrbSizeLarge  =  0;
        public const int COrbSizeMedium = 15;
        public const int COrbSizeSmall  = 14;
        
        public static int MaxSize = 0;
        
        private bool fVisible = false;               // object is visible on screen
        public bool Visible                               
        { get { return fVisible; }
          set { 
            if ( fVisible == value ) return;           
            fVisible = value;
            if ( fVisible == false ) 
            {
              InitPositions();
              Speed = new Vector { X = 0, Y = 0 };
              LifeTime = 0;
            }
          } 
        }

        protected const int vtNS = 0;
        protected const int vtWE = 1;
        protected const int vtDiag = 2;
        protected Vector[] fVirtualPositions = new Vector[vtDiag+1];

        public TOrbType OrbType { get; protected set; }
        public Vector Position { get; set; }              // coordinates of midpoint
        public Vector VirtualPosition { get; set; }       // virtual coordinates of nearest midpoint
        public double X { get {return Position.X;} }
        public double Y { get {return Position.Y;} } 
        public Vector Speed { get; set; }      
        public bool NeedUpdate { get; set; }
        public ulong LifeTime { get; set; }               // FrameCounter
        public double DeltaDirection { get; set; }
        public double LastDistanceTo { get; set; }

        private void InitPositions()
        {
            Position = new Vector { X = 0, Y = 0 };
            VirtualPosition = CNullVector;
            fVirtualPositions[vtNS] = new Vector { X = 0, Y = 0 };
            fVirtualPositions[vtWE] = new Vector { X = 0, Y = 0 };
            fVirtualPositions[vtDiag] = new Vector { X = 0, Y = 0 };
        }

        private void Init()
        {
            fVisible = false;
            OrbSize = COrbSizeMedium;
            InitPositions();
            Speed = new Vector { X = 0, Y = 0 };
            NeedUpdate = false;
        }
        
        public TOrb()
        {
          this.Clear();
          this.LifeTime = 0;
          this.DeltaDirection = 0.0;
          this.LastDistanceTo = 0.0;
        }
        
        public virtual void Clear()
        {
          this.Init();
        }

        public override string ToString()
        {
          return String.Format ("Pos:{0}; Size:{1}; Speed:{2}({3:F}); Life:{4,2}; Visible:{5}; Delta:{6:F}; Dist:{7:F}", 
                                Position.ToString(), Size(), Speed.ToString(), Speed.Length, LifeTime.ToString(), Visible.ToString(), DeltaDirection, LastDistanceTo);
        }

        protected void SetVirtualPositions()
        {
            if (this.Position.X >= 0)
            {
                if (this.Position.Y >= 0)
                {
                    fVirtualPositions[vtNS].X = this.Position.X;
                    fVirtualPositions[vtNS].Y = this.Position.Y - 768;
                    fVirtualPositions[vtWE].X = this.Position.X - 1024;
                    fVirtualPositions[vtWE].Y = this.Position.Y;
                    fVirtualPositions[vtDiag].X = this.Position.X - 1024;
                    fVirtualPositions[vtDiag].Y = this.Position.Y - 768;
                }
                else  // Y < 0
                {
                    fVirtualPositions[vtNS].X = this.Position.X;
                    fVirtualPositions[vtNS].Y = this.Position.Y + 768;
                    fVirtualPositions[vtWE].X = this.Position.X - 1024;
                    fVirtualPositions[vtWE].Y = this.Position.Y;
                    fVirtualPositions[vtDiag].X = this.Position.X - 1024;
                    fVirtualPositions[vtDiag].Y = this.Position.Y + 768;
                }
            }
            else  // X < 0
            {
                if (this.Position.Y >= 0)
                {
                    fVirtualPositions[vtNS].X = this.Position.X;
                    fVirtualPositions[vtNS].Y = this.Position.Y - 768;
                    fVirtualPositions[vtWE].X = this.Position.X + 1024;
                    fVirtualPositions[vtWE].Y = this.Position.Y;
                    fVirtualPositions[vtDiag].X = this.Position.X + 1024;
                    fVirtualPositions[vtDiag].Y = this.Position.Y - 768;
                }
                else  // Y < 0
                {
                    fVirtualPositions[vtNS].X = this.Position.X;
                    fVirtualPositions[vtNS].Y = this.Position.Y + 768;
                    fVirtualPositions[vtWE].X = this.Position.X + 1024;
                    fVirtualPositions[vtWE].Y = this.Position.Y;
                    fVirtualPositions[vtDiag].X = this.Position.X + 1024;
                    fVirtualPositions[vtDiag].Y = this.Position.Y + 768;
                }
            }
        }

        protected void CalcVirtualPosition(TOrb aOrb)
        {
          // Calc the vector of the nearest virtual position...
          SortedList<double, Vector> sl = new SortedList<double, Vector>();
          sl.Add ((this.Position - (aOrb.Position + aOrb.Speed)).Length, aOrb.Position);
          try {
              sl.Add((this.Position - (aOrb.fVirtualPositions[vtNS] + aOrb.Speed)).Length, aOrb.fVirtualPositions[vtNS]);
          } catch { };
          try {
              sl.Add((this.Position - (aOrb.fVirtualPositions[vtWE] + aOrb.Speed)).Length, aOrb.fVirtualPositions[vtWE]);
          }
          catch { };
          try {
              sl.Add((this.Position - (aOrb.fVirtualPositions[vtDiag] + aOrb.Speed)).Length, aOrb.fVirtualPositions[vtDiag]);
          } catch { };

          aOrb.VirtualPosition = new Vector(sl.Values[0].X, sl.Values[0].Y);
        }
        
        public void Normalize(ref Vector aPos)
        {
            aPos.X = aPos.X - 512;
            aPos.Y = aPos.Y - 384;
            //while (aPos.X < -512) aPos.X += 1024;          // dx normalisieren auf -512 ... 511
            //while (aPos.X > 511) aPos.X -= 1024;
            //while (aPos.Y < -384) aPos.Y += 768;           // dy normalisieren auf -384 ... 383
            //while (aPos.Y > 383) aPos.Y -= 768;
        }

        public virtual void Set(Vector aPos)
        {
          // calculate speed with new and previous value...
          Vector posNew = new Vector(aPos.X, aPos.Y);
          this.Normalize(ref posNew);

          if (this.Position.Equals(posNew)) return;
          
          if ( this.Position.Equals(CNullVector) )
              this.Position = posNew;
          
          // Speed = aPos - Position;
          this.Speed = posNew - this.Position;
          
          // set new position of orb...
          this.Position = posNew;

          // Update the virtual positions for later use...
          this.SetVirtualPositions();
        }

        public virtual int Size()
        {
          return MaxSize;
        }
        
        public double DistanceTo(TOrb aOrb, ref double vs)
        {
          // Calc the vector of the nearest virtual position...
          CalcVirtualPosition(aOrb);

          double result = (this.Position - aOrb.VirtualPosition).Length;

          // Calc delta direction of aOrb in reference to itself...
          // negative = in my direction, positive = in other direction, zero = parallel
          vs = result - (this.Position - (aOrb.VirtualPosition + aOrb.Speed)).Length;
          aOrb.DeltaDirection = vs;

          aOrb.LastDistanceTo = Math.Max(0, result - (this.Size()/2 + aOrb.Size()/2));
          return aOrb.LastDistanceTo;
        }
    }

    public class TAsteroid : TOrb
    {
        private int Type { get; set; }               // 1 ... 4, äußere Form

        public static new int MaxSize = 64;  // 64 * 64
        
        private void Init()
        {
          this.Type = 0;
        }
        
        public TAsteroid() : base()
        {
          this.OrbType = TOrb.TOrbType.otAsteroid;
          this.Init();
          this.Visible = true;
        }

        public TAsteroid(Vector aPos, int aOrbSize, int aType) : base()
        {
          this.Set(aPos, aOrbSize, aType);
          this.Visible = true;
        }
        
        public override void Clear()
        {
          base.Clear();
          this.Init();
        }
        public override string ToString()
        {
          return base.ToString() + String.Format (", Type:{0}", Type);
        }
        
        public void Set(Vector aPos, int aOrbSize, int aType)
        {
 	      this.Set(aPos);
          this.OrbSize = aOrbSize;
          this.Type = aType;
          this.NeedUpdate = false;
        }

        public bool Match(Vector aPos, int aOrbSize, int aType)
        {
          Vector delta = new Vector(aPos.X, aPos.Y);
          this.Normalize(ref delta);

          return ( (aType == Type) && 
                   (aOrbSize == OrbSize) &&
                   ((delta - (this.Position + this.Speed)).Length < 8));
        }
        
        public override int Size()
        {
          switch (OrbSize)
          {	// Abstand um den ungefähren Radius des Asteroiden korrigieren
            case COrbSizeLarge:  return MaxSize;     // large Asteroid   64 * 64           
            case COrbSizeMedium: return 32;          // medium Asteroid  32 * 32            
            case COrbSizeSmall:  return 16;          // small Asteroid   16 * 16         
            default:             return base.Size();              
          }                                 
        }
    }

    class TUfo : TOrb
    {
        public static new int MaxSize = 40;  // 40 * 25
         
        public TUfo() : base()
        {
          this.OrbType = TOrb.TOrbType.otUfo;
        } 
         
        public void Set(Vector aPos, int aOrbSize, bool aVisible)
        {
   	      this.Set(aPos);
          this.OrbSize = aOrbSize;
          this.Visible = aVisible;
          this.NeedUpdate = false;
        }

        public bool Match(Vector aPos, int aOrbSize)
        {
          Vector delta = new Vector(aPos.X - Position.X, aPos.Y - Position.Y);
          this.Normalize(ref delta);

          return ((aOrbSize == OrbSize) &&
                 ((delta - (this.Position + this.Speed)).Length < 6));
        }
        
        public override int Size()
        {
          switch (OrbSize)
          {	// Abstand um den ungefähren Radius des UFOs korrigieren
            case COrbSizeMedium:  return MaxSize;      // medium UFO  40 * 25            
            case COrbSizeSmall:   return 20;           // small UFO   20 * 14
            default:              return base.Size();
          }
        }
    }

    class TShip : TOrb
    {
        public static new int MaxSize = 26;    // 26 * 18
        
        public Vector Direction { get; set; }        // view in direction
        
        private void Init()
        {
          Direction = new Vector { X = 0, Y = 0 };
        }
                
        public TShip() : base()
        {
          this.OrbType = TOrb.TOrbType.otShip;
          this.Init();
        }
        
        public override void Clear()
        {
          base.Clear();
          this.Init();
        }
        
        public override string ToString()
        {
          return base.ToString() + String.Format ("; Direction:{0}", Direction.ToString());
        }

        public void Set(Vector aPos, int aOrbSize, bool aVisible, Vector aDirection)
        {
          this.Set(aPos);
          this.OrbSize = aOrbSize;
          this.Visible = aVisible;
          this.Direction = aDirection; 
          this.NeedUpdate = false;
        }

        public override int Size()
        {
          return MaxSize;
        }
    }

    class TShot : TOrb
    {
        public static new int MaxSize = 2;

        public TOrb Source { get; protected set; }
              
        public TShot(Vector aPos, TOrb aSource) : base()
        {
          this.Source = aSource;
          this.OrbType = TOrb.TOrbType.otShot;
          this.Set(aPos);
          this.Visible = true;
        }

        public TShot(Vector aPos, TOrb aFirstSource, TOrb aSecondSource) : base()
        {
            Vector delta = new Vector(aPos.X, aPos.Y);
            this.Normalize(ref delta);

            // Which orb is the sourcer of the shot...
            if ((delta - aFirstSource.Position).Length <= (delta - aSecondSource.Position).Length)
                this.Source = aFirstSource;
            else
                this.Source = aSecondSource;

            this.OrbType = TOrb.TOrbType.otShot;
            this.Set(aPos);
            this.Visible = true;
        }
              
        public bool Match(Vector aPos)
        {
          Vector delta = new Vector(aPos.X, aPos.Y);
          this.Normalize(ref delta);

          //TShot tmpShot = new TShot(delta, this.Source);
          //tmpShot.SetVirtualPositions();
          //tmpShot.CalcVirtualPosition(this);
          
          //return ((tmpShot.VirtualPosition - this.Position).Length < 12);
          return ((delta - (this.Position + this.Speed)).Length < 12);
        }
              
        public override string ToString()
        {
          return base.ToString() + String.Format ("; Source:{0}", Enum.GetName(typeof(TOrbType), Source.OrbType));
        }
              
        public override int Size()
        {
          return MaxSize;
        }
    }
}
