/* asteroid.cpp : Spielt Asteroids gegen MAME

Strategie:
- Das Schiff wird nach links oder rechts in der Mitte bewegt, da man dort UFOs 
  und neue Asteroiden gut abschiessen kann
- Auf grosse Asteroiden werden unter Umstnden Salven bis zu vier Schsse 
  gefeuert, mittlere Asteroiden bis zu drei (komplett weg)
- Die Auswahl des nchsten Ziels erfolgt anhand Anzahl Drehungen, Anzahl 
  Warteframes und Anzahl Frames bis zum Treffer (davon aber nur ein Viertel, 
  wie experimentell als optimal ermittelt wurde). 
- 10 Sekunden vor Schluss erhlt ein UFO hchste Prioritt (1000 Punkte!)
- Ein UFO hat immer etwas hhere Prioritt, denn wenn es schnell abgeschossen 
  wird, kann es seltener das eigene Schiff abschiessen und gelegentlich dauert 
  das Abschiessen der letzte Asteroiden so lange, das noch ein UFO kommt

Technik:
- Es wird eine zustzliche Latenz benutzt, um Problemen damit aus dem Weg zu gehen
- Der Hauptthread wartet auf einen Frame, schickt dann die Tasten aus der letzten 
  Berechnung und startet eine neu Berechnung durch andere Threads
- Um mehr Frames simulieren zu knnen, arbeiten soviel Threads wie Cores vorhanden
- Die mglichen Schsse werden einer beim Start erzeugten Tabelle entnommen. 
  Darin stehen fr alle Punkte von 0 <= x <= 1023 und 0 <= y <= 767 mit dem Schiff 
  bei x=512 und y=384 Listen von mglichen Schssen, die dort treffen knnen.
- Fr jeden Radius eines Objekts gibt es eine Liste von x,y-Positionen innerhalb 
  des Objekts, bei denen es getroffen werden kann
- Das Winkelbyte (g_anShotAngleIndex) gibt den Schusswinkel an, der verwendet wird, 
  wenn ein Schuss angewiesen wird. Der Schuss taucht erst 3 Frames spter auf, hat 
  aber dann den gewnschten Winkel, da die Drehungen ebenfalls verzgert sind.
- Das Winkelbyte wird anhand der Blickrichtung synchronisiert. Wenn der Winkel der 
  Blickrichtung zu stark (7 Grad) vom Winkelbyte abweicht, wird entsprechend 
  korrigiert. Als Blickrichtung wird der Wert von zwei Frames vorher verwendet, 
  da die aktuelle Blickrichtung mit dem Winkelbyte vor zwei Frames korrelliert.
  Die Winkelkorrektur spricht aber nicht mehr an, seit whrend des Hypersprungs kein 
  Turn mehr gemacht wird.

Dank an:
- c't fr die Idee :-)
- Harald Bgeholz fr das Sample
- AnDann fr die Winkeltabelle (sehr wichtig!)

*/

#include "stdafx.h"
#include "afxmt.h"
#include "process.h"
#include "math.h"
#include "asteroid.h"
#include "mmsystem.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//-----------------------------------------------------------------------------
#define STAT            0
#define LATENCY         2
#define MAX_LATENCY     4

#define PI              3.141592654

#if _DEBUG
#define RECORD          0 // 0 = none, >=1 = record
#define PLAYBACK        1 // 0 = no, 1 = playback
#define NUM_THREADS     1  // should be equal to number of cores
#else
#define RECORD          0 // 0 = none, >=1 = record
#define PLAYBACK        0 // 0 = no, 1 = playback
#define NUM_THREADS     2  // should be equal to number of cores
#endif

#define STORE_TABLE     0 // 0 = no, 1 = yes
#define STORE_CIRCLE    0 // 0 = no, 1 = yes

#define REC_FILE_NAME   "c:\\tmp\\ast.txt"
#define TABLE_FILE_NAME "c:\\tmp\\table.txt"
#define CIRCLE_FILE_NAME "c:\\tmp\\circle.txt"

#define SEND_PORT       1979
#define REC_PORT        0

#define KEY_HYPERSPACE  1
#define KEY_FIRE        2
#define KEY_THRUST      4
#define KEY_RIGHT       8
#define KEY_LEFT        0x10

#define BIG_AST_RADIUS  32
#define MED_AST_RADIUS  16
#define SML_AST_RADIUS  8
#define BIG_UFO_RADIUS  16
#define SML_UFO_RADIUS  8

#define INVALID         0x7FFFFFFF
#define MIN_TRACK_DIST  6
#define MAX_TRACK_DIST  14
#define SHIP_RADIUS     15
#define SHOT_SPEED      7.94
#define SHOT_RANGE      68 // frames
#define MAX_HIT_FRAMES  120
#define TURNS_CIRCLE    85
#define TURNS_SIM       (TURNS_CIRCLE+SHOT_RANGE)
#define HIT_RANGE       150
#define MIN_HIT_FRAMES  (LATENCY+1)
#define MAX_OWN_SHOTS   4
#define NO_HIT_MALUS    HIT_RANGE
#define MIN_SHOT_DIST   22
#define MAX_FRAME_CNT   18000
#define FIRE_CNT        (LATENCY+1)

#define CLOSE_TO_BORDER     100
#define TOO_CLOSE_TO_BORDER 50

#define MIN_X           0
#define MAX_X           1024
#define MIN_Y           128
#define MAX_Y           896
#define SIZE_X          1024
#define SIZE_Y          768
#define MIDDLE_X        (SIZE_X/2)
#define MIDDLE_Y        (SIZE_Y/2)
#define INIT_SHOT_POS   2.4

#define SHIP_INDEX      MAX_SHOTS
#define SAUCER_INDEX    (MAX_SHOTS+1)
#define MAX_OBJECTS     (MAX_ASTEROIDS + MAX_SHOTS + 2)

#define DEFAULT_X       (MIN_X + SIZE_X * 7 / 8)
#define DEFAULT_Y       (MIN_Y + SIZE_Y / 2)
#define DEFAULT_AREA_X  30
#define DEFAULT_AREA_Y  200
#define DEFAULT_MIN_X   (DEFAULT_X - DEFAULT_AREA_X)
#define DEFAULT_MAX_X   (DEFAULT_X + DEFAULT_AREA_X)
#define DEFAULT_MIN_Y   (DEFAULT_Y - DEFAULT_AREA_Y)
#define DEFAULT_MAX_Y   (DEFAULT_Y + DEFAULT_AREA_Y)


//-----------------------------------------------------------------------------
static const int MAX_ASTEROIDS = 27;
static const int MAX_SHOTS = 6;


//-----------------------------------------------------------------------------
class AngleAndFrames
{
public:
  BYTE nAngle, nFrames;
};

class ObjInfo
{
public:
  void set (int x, int y, int nRadius);

  int    x, y;
  int    nSpeedFrames;
  double nSpeedX, nSpeedY, nSimX, nSimY;
  int    nRadius;
  int    nShotCnt;
  int    nPrio;
  int    nHits;
  int    nSimCnt;
  int    nMultiShot;
  bool   bPrev, bScanned, bMyShot;
  int    nFramesShoot, nTurns, nWaits, nFrames;
  ObjInfo* pPrev;
  ObjInfo* pNext;
};

class Asteroid
{
public:
  ObjInfo info;
  int type; // 1 ... 4, uere Form

  void set(int x, int y, int type, int sf);
};

class Shot
{
public:
  ObjInfo info;

  void set(int x, int y);
};

#pragma pack(1)
struct FramePacket
{
  char vectorram[1024];
  BYTE frameno;  // wird bei jedem Frame inkrementiert
  BYTE ping;     // Der Server schickt das letzte empfangene ping-Byte zurck
};

class KeysPacket
{
private:
  char signature[6];
public:
  BYTE keys;
  BYTE ping;     // wird vom Server bei nchster Gelegenheit zurckgeschickt. Fr Latenzmessung.

  KeysPacket(void);
  void clear(void);         // alle Tasten loslassen
  void hyperspace(bool b);  // Hyperspace drcken (true) oder loslassen (false)
  void fire(bool b);        // Feuerknopf drcken (true) oder loslassen (false)
  void thrust(bool b);      // Beschleunigen ...
  void right(bool b);       // rechts drehen ...
  void left(bool b);        // links drehen
};
#pragma pack()

class SimInfo
{
public:
  SimInfo ();

  void StoreBestShot (ObjInfo* pPos, int nThread);

  int  nFired;
  int  anMyShotsHit[MAX_OWN_SHOTS];
  ObjInfo* pSelPos;
  ObjInfo* pShotPos;
  int nFrameCnt;

  bool abTermThread[NUM_THREADS];
  CSemaphore resultSem;
  int  nTurn;
  int  nWaits, nFrames, nFramesShoot;
  bool bMoveToTarget;

  int nLevel;
};

class GameStatus
{
public:
  bool ship_present;  // Schiff sichtbar
  ObjInfo ship;
  int ship_dx;        // Blickrichtung des Schiffes
  int ship_dy;
  bool saucer_present;// UFO sichtbar
  ObjInfo saucer;
  int nAsteroids; // Anzahl Asteroiden
  Asteroid asteroids[MAX_ASTEROIDS];
  int  nShots;     // Anzahl Schsse
  Shot shots[MAX_SHOTS];
  int  nMyShots;
  int  nNextShot;
  bool bShipWillBeHit;
  SimInfo* pSim;

  CSemaphore  objectSem;
  ObjInfo*    pObjectList;
  GameStatus* pPrevGame;

  void clear             (void);
  void InterpretScreen   (FramePacket &packet);
  bool TrackMatch        (ObjInfo* pPrevPos, ObjInfo* pCurrPos);
  void CheckTrackVariant (int nPrevCount, 
                          int nCurrCount,
                          ObjInfo* (*fGetObject) (GameStatus*, int),
                          int nPrevIndex, int nCurrIndex, 
                          int nSkipped, bool abInserted[], 
                          bool abRemoved[], bool bInserted, 
                          bool bRemoved, int* pnMinSkipped, 
                          bool abMaxInserted[], bool abMaxRemoved[]);
  void SetSpeed          (ObjInfo* pPrevPos, ObjInfo* pNewPos);
  void GetSpeedOfObjects (int nPrevCount, int nCount, 
                          ObjInfo* (*fGetObject) (GameStatus*, int));
  void GetSpeed          ();
  int  ShootableTargets  ();
  void CorrectAngle      ();
  void SetOwnShots       ();
  void GetShotAvailable  ();
  void GetObjectList     ();
  void GetBestShot       (int nThread);
  bool GetBestShot       (int nThread,
                          int x, int y, int nRadius, int nNextShot, 
                          int nFramesPassed, 
                          int* pnTurns, int* pnWaits, int* pnFrames);
  bool ShipSpeedHigh     (int dx, int dy);
  bool MoveToTarget      ();
  int  ShipAtBorder      ();
  bool DefaultPosition   (int& nDefaultOffsetX);
  bool IsShipDirection   (int dx, int dy);
  int  AngleDifference   (int dx, int dy);
  bool TurnToDefault     (int& nDefaultOffsetX, int& dy);
  bool MoveToDefault     ();
  void Save              (const char* sFilename);
  void Load              (FILE* pFile);
};

class Player
{
public:
  void Run             (void);

private:
  void ReceivePacket   (FramePacket &packet);
  void SendPacket      (KeysPacket &packet);
};


/*-------------------------------------------------------------------------*/
CWinApp theApp;

using namespace std;

static char     g_sIpAddress[4*4];
static CSocket* g_pSocket = NULL;

static const double g_anShotAngles[256] = 
{
  0.0, 3.6, 8.2, 12.7, 16.5, 21.2, 25.2, 29.5, 33.5, 37.8, 41.7, 46.2, 50.8,
  55.3, 59.4, 63.4, 67.5, 71.6, 76.1, 79.9, 84.5, 89.1, 92.8, 97.2, 101.8,
  105.5, 110.2, 114.1, 118.1, 122.8, 126.7, 131.1, 135.0, 139.4, 143.3, 147.9, 
  151.9, 156.6, 160.7, 164.5, 169.2, 172.8, 177.3, 181.9, 185.5, 189.9, 194.5, 
  198.2, 203.0, 206.9, 211.0, 215.4, 220.0, 223.8, 227.5, 232.2, 235.9, 240.3, 
  245.1, 248.8, 252.7, 257.3, 261.1, 265.6, 270.0, 273.6, 278.2, 282.7, 286.5, 
  291.2, 295.0, 299.0, 303.5, 307.8, 311.7, 316.2, 320.0, 324.5, 328.6, 332.7, 
  336.6, 341.6, 345.3, 349.9, 354.5, 358.1, 2.8, 7.2, 11.0, 15.5, 19.3, 23.4, 
  28.1, 32.8, 36.7, 40.6, 45.0, 49.4, 53.3, 57.2, 61.9, 66.6, 70.7, 74.5, 79.0, 
  82.8, 87.2, 91.9, 95.5, 100.1, 104.7, 108.4, 113.4, 117.3, 121.4, 125.6, 130.0, 
  133.8, 138.3, 142.2, 146.5, 151.0, 155.0, 158.8, 163.6, 167.3, 171.8, 176.6, 
  180.0, 184.4, 188.9, 192.7, 197.3, 201.2, 205.0, 209.7, 214.2, 217.8, 222.5, 
  226.2, 230.0, 234.6, 239.0, 243.1, 247.0, 251.8, 255.5, 260.1, 264.5, 268.1, 
  272.7, 277.2, 280.8, 285.5, 289.3, 293.4, 298.1, 302.2, 306.7, 310.6, 315.0, 
  318.9, 323.3, 327.2, 331.9, 335.9, 339.8, 344.5, 348.2, 352.8, 357.2, 0.9, 
  5.5, 10.1, 13.9, 18.4, 22.5, 26.6, 30.6, 34.7, 39.2, 43.8, 48.3, 52.2, 56.5, 
  60.5, 64.7, 68.8, 73.5, 77.3, 81.8, 86.4, 90.0, 94.5, 98.9, 102.7, 107.3, 
  111.2, 115.3, 120.2, 124.2, 127.8, 132.5, 136.1, 140.8, 145.3, 149.8, 153.9, 
  157.9, 161.8, 166.3, 170.1, 174.3, 179.1, 182.9, 187.2, 191.5, 195.6, 200.3, 
  204.1, 208.1, 212.1, 216.6, 221.0, 225.0, 228.9, 233.3, 237.8, 241.9, 245.9, 
  249.8, 254.5, 258.4, 262.8, 267.3, 270.9, 275.5, 279.9, 283.7, 288.2, 292.1, 
  296.1, 300.2, 304.7, 309.2, 313.8, 317.5, 322.2, 325.8, 329.8, 334.7, 338.8, 
  342.7, 347.2, 351.1, 355.5
};

static BYTE g_nShotAngleIndex = 0;
static BYTE g_anPrevAngles[LATENCY];
static int  g_anPrevFrames[LATENCY];
static int  g_nLatencyPos;

static AngleAndFrames* g_aapShotTable[SIZE_X][SIZE_Y];
static int g_anCircleSml[3*(SML_AST_RADIUS+1)*(SML_AST_RADIUS+1)];
static int g_anCircleMed[3*(MED_AST_RADIUS+1)*(MED_AST_RADIUS+1)];
static int g_anCircleBig[3*(BIG_AST_RADIUS+1)*(BIG_AST_RADIUS+1)];

static HANDLE      g_ahSimEvent[NUM_THREADS];
static GameStatus* g_pSimGame;

static CSemaphore  g_debugSem;


/*-------------------------------------------------------------------------*/
static void ReleaseSocket ()
{
  delete g_pSocket;
  g_pSocket = NULL;
}


/*-------------------------------------------------------------------------*/
static bool CreateSocket ()
{
  if (g_pSocket == NULL)
  {
    g_pSocket = new CSocket ();

    if (!g_pSocket-> Create (REC_PORT, SOCK_DGRAM))
    {
      ReleaseSocket ();
      return false;
    }

    g_pSocket-> AsyncSelect (FD_READ);
  }

  return true;
}


/*-------------------------------------------------------------------------*/
static int round (double f)
{
  if (f >= 0)
    return (int) (f + 0.5);

  return (int) (f - 0.5);
}


/*-------------------------------------------------------------------------*/
SimInfo::SimInfo ()
{
  int i;

  nFired  = 0;
  pSelPos = NULL;

  for (i = 0; i < MAX_OWN_SHOTS; i++)
  {
    anMyShotsHit[i] = 0;
  }

  for (i = 0; i < NUM_THREADS; i++)
  {
    abTermThread[i] = false;
  }

  nTurn = INVALID;
}


/*-------------------------------------------------------------------------*/
KeysPacket::KeysPacket (void)
{
  signature[0] = 'c';
  signature[1] = 't';
  signature[2] = 'm';
  signature[3] = 'a';
  signature[4] = 'm';
  signature[5] = 'e';
  keys = '@';
  ping = 0;
}


/*-------------------------------------------------------------------------*/
void KeysPacket::clear (void)
{
  keys = '@';
}


/*-------------------------------------------------------------------------*/
void KeysPacket::hyperspace (bool b)
{
  if (b)
    keys |= KEY_HYPERSPACE;
  else
    keys &= ~KEY_HYPERSPACE;
}


/*-------------------------------------------------------------------------*/
void KeysPacket::fire (bool b)
{
  if (b)
    keys |= KEY_FIRE;
  else
    keys &= ~KEY_FIRE;
}


/*-------------------------------------------------------------------------*/
void KeysPacket::thrust (bool b)
{
  if (b)
    keys |= KEY_THRUST;
  else
    keys &= ~KEY_THRUST;
}


/*-------------------------------------------------------------------------*/
void KeysPacket::left (bool b)
{
  if (b)
  {
    keys |= KEY_LEFT;
    right(false);
  }
  else
    keys &= ~KEY_LEFT;
}


/*-------------------------------------------------------------------------*/
void KeysPacket::right (bool b)
{
  if (b)
  {
    keys |= KEY_RIGHT;
    left(false);
  }
  else
    keys &= ~KEY_RIGHT;
}


/*-------------------------------------------------------------------------*/
void ObjInfo::set (int x, int y, int nRadius)
{
  this->x            = x;
  this->y            = y;
  this->nSpeedX      = 0;
  this->nSpeedY      = 0;
  this->nSpeedFrames = 1;
  this->nRadius      = nRadius;
  this->nShotCnt     = 0;
  this->nMultiShot   = 0;
  this->pPrev        = NULL;
  this->bPrev        = false;
}


/*-------------------------------------------------------------------------*/
void Asteroid::set (int x, int y, int type, int sf)
{
  int nRadius;

  switch (sf)
  {
  case 0:  // groer Asteroid
    nRadius = BIG_AST_RADIUS;
    break;
  case 15: // mittlerer Asteroid
    nRadius = MED_AST_RADIUS;
    break;
  case 14: // kleiner Asteroid
    nRadius = SML_AST_RADIUS;
    break;
  }

  info.set (x, y, nRadius);

  this->type = type;
}


/*-------------------------------------------------------------------------*/
void Shot::set (int x, int y)
{
  info.set (x, y, 1);
  info.bMyShot = false;
}

#if RECORD >= 2

/*-------------------------------------------------------------------------*/
void GameStatus::Save (const char* sFilename)
{
  int   i;
  FILE* pFile;
  ObjInfo* pPos;

  if (pSim->nFrameCnt >= 0)
  {
    g_debugSem.Lock ();

    pFile = fopen (sFilename, "at");

    if (pFile != NULL)
    {
      fprintf (pFile, "Frame %i Saucer %i %i/%i %.3f %.3f %i %i Angle %i Ship %i/%i Dir %i/%i Asteroids %i Shots %i Sel %i/%i %i %i %i %i fire %i\n", 
        pSim->nFrameCnt, (int) saucer_present, saucer.x, saucer.y, saucer.nSpeedX, saucer.nSpeedY,
        saucer.nRadius, saucer.nShotCnt, g_nShotAngleIndex, ship.x, ship.y, ship_dx, ship_dy, nAsteroids, nShots, 
        pSim->pSelPos != NULL ? pSim->pSelPos->x : 0,
        pSim->pSelPos != NULL ? pSim->pSelPos->y : 0,
        pSim->nFramesShoot, pSim->nTurn, pSim->nWaits, pSim->nFrames, pSim->nFired);

      for (i = 0; i < nAsteroids; i++)
      {
        pPos = &asteroids[i].info;

        fprintf (pFile, "%i %i %i %i %.3f %.3f %i %02X\n", 
            i, pPos->x, pPos->y, pPos->nRadius, pPos->nSpeedX, pPos->nSpeedY, pPos->nShotCnt, pPos->nMultiShot);
      }

      fprintf (pFile, "Shots\n");

      for (i = 0; i < nShots; i++)
      {
        pPos = &shots[i].info;

        fprintf (pFile, "%i %i %i %.3f %.3f %i\n", 
            i, pPos->x, pPos->y, pPos->nSpeedX, pPos->nSpeedY, (int) pPos->bMyShot);
      }

      fprintf (pFile, "\n");

      fclose (pFile);
    }

    g_debugSem.Unlock ();
  }
}

#endif
#if PLAYBACK

/*-------------------------------------------------------------------------*/
void GameStatus::Load (FILE* pFile)
{
  char  sLine[256];
  int   i = 0, nState = 1, nSaucerPresent, nAngle, nMyShot;
  float nSpeedX, nSpeedY;
  ObjInfo* pPos;

  nAsteroids = 0;
  nShots     = 0;

  while (fgets (sLine, sizeof (sLine), pFile) != NULL)
  {
    // empty line marks end of record
    if (sLine[0] == '\n')
      break;

    if (strncmp (sLine, "Shots", 5) == 0)
    {
      nState = 2;
      nShots = 0;
    }
    else if (strncmp (sLine, "Frame", 5) == 0)
    {
      sscanf (sLine, "Frame %i Saucer %i %i/%i %f %f %i %i Angle %i Ship %i/%i Dir %i/%i", 
        &i, &nSaucerPresent, &saucer.x, &saucer.y, &nSpeedX, &nSpeedY, &saucer.nRadius, &saucer.nShotCnt, 
        &nAngle, &ship.x, &ship.y, &ship_dx, &ship_dy);
      saucer_present = nSaucerPresent != 0;
      saucer.nSpeedX = nSpeedX;
      saucer.nSpeedY = nSpeedY;
      ship.nSpeedX   = 0;
      ship.nSpeedY   = 0;
      g_nShotAngleIndex = (BYTE) nAngle;
      nState = 1;
      nAsteroids = 0;
    }
    else if (nState == 1)
    {
      pPos = &asteroids[nAsteroids++].info;
      sscanf (sLine, "%i %i %i %i %f %f %i", &i, &pPos->x, &pPos->y, &pPos->nRadius,
        &nSpeedX, &nSpeedY, &pPos->nShotCnt);
      pPos->nSpeedX = nSpeedX;
      pPos->nSpeedY = nSpeedY;
      pPos->nSpeedFrames = 1;
    }
    else
    {
      pPos = &shots[nShots++].info;
      sscanf (sLine, "%i %i %i %f %f %i", &i, &pPos->x, &pPos->y, &nSpeedX, &nSpeedY, &nMyShot);
      pPos->nSpeedX = nSpeedX;
      pPos->nSpeedY = nSpeedY;
      pPos->bMyShot = nMyShot != 0;
      pPos->nRadius = 1;
      pPos->nSpeedFrames = 1;
    }
  }
}

#endif
#if RECORD >= 1

/*-------------------------------------------------------------------------*/
int Debug (const char* format, ...)
{
  char debugPrintfBuffer [256];

  va_list argPtr;
  int     len;
  FILE*   pFile;

  va_start (argPtr, format);
  len = vsprintf (debugPrintfBuffer, format, argPtr);
  va_end (argPtr);

  g_debugSem.Lock ();

  pFile = fopen (REC_FILE_NAME, "at");

  if (pFile != NULL)
  {
    fprintf (pFile, debugPrintfBuffer);
    fclose  (pFile);
  }

  g_debugSem.Unlock ();

  return len;
}

#else

#define Debug printf

#endif

/*-------------------------------------------------------------------------*/
void GameStatus::clear (void)
{
  ship_present   = false;
  saucer_present = false;
  nAsteroids     = 0;
  nShots         = 0;
  bShipWillBeHit = false;
}


/*-------------------------------------------------------------------------*/
static int XAxis (int x)
{
  while (x < MIN_X) x += SIZE_X; // x normalisieren auf MIN_X..MAX_X-1
  while (x >= MAX_X) x -= SIZE_X;

  return x;
}


/*-------------------------------------------------------------------------*/
static int YAxis (int y)
{
  while (y < MIN_Y) y += SIZE_Y; // y normalisieren auf MIN_Y..MAY_Y-1
  while (y >= MAX_Y) y -= SIZE_Y;

  return y;
}


/*-------------------------------------------------------------------------*/
static double XAxisD (double x)
{
  while (x < MIN_X) x += SIZE_X; // x normalisieren auf MIN_X..MAX_X-1
  while (x >= MAX_X) x -= SIZE_X;

  return x;
}


/*-------------------------------------------------------------------------*/
static double YAxisD (double y)
{
  while (y < MIN_Y) y += SIZE_Y; // y normalisieren auf MIN_Y..MAY_Y-1
  while (y >= MAX_Y) y -= SIZE_Y;

  return y;
}


//-----------------------------------------------------------------------------
static int DistanceSquared (int x1, int y1, int x2, int y2)
{
  int nDiffX = abs (x1-x2);
  int nDiffY = abs (y1-y2);

  // all objects are reached in two directions
  if (nDiffX > SIZE_X / 2)
    nDiffX = SIZE_X - nDiffX;

  if (nDiffY > SIZE_Y / 2)
    nDiffY = SIZE_Y - nDiffY;

  return nDiffX * nDiffX + nDiffY * nDiffY;
}


//-----------------------------------------------------------------------------
#define Distance(x1, y1, x2, y2)    sqrt (DistanceSquared (x1, y1, x2, y2))


//-----------------------------------------------------------------------------
void GameStatus::SetSpeed (ObjInfo* pPrevPos, ObjInfo* pNewPos)
{
  int nDiffX = pNewPos->x - pPrevPos->x;
  int nDiffY = pNewPos->y - pPrevPos->y;

  if (nDiffX >= SIZE_X / 2)
    nDiffX -= SIZE_X;

  if (nDiffX <= -SIZE_X / 2)
    nDiffX += SIZE_X;

  if (nDiffY >= SIZE_Y / 2)
    nDiffY -= SIZE_Y;

  if (nDiffY <= -SIZE_Y / 2)
    nDiffY += SIZE_Y;

  pNewPos->nSpeedFrames = pPrevPos->nSpeedFrames;

  pNewPos->nSpeedX = pPrevPos->nSpeedX 
    + (nDiffX - pPrevPos->nSpeedX) / pPrevPos->nSpeedFrames;
  pNewPos->nSpeedY = pPrevPos->nSpeedY 
    + (nDiffY - pPrevPos->nSpeedY) / pPrevPos->nSpeedFrames;

  // check for tracking error
  if ((pPrevPos->nSpeedX != 0 || pPrevPos->nSpeedY != 0)
    && (fabs (fabs (pNewPos->nSpeedX) - fabs (pPrevPos->nSpeedX)) >= 2
      || fabs (fabs (pNewPos->nSpeedY) - fabs (pPrevPos->nSpeedY)) >= 2))
  {
    /*
    printf ("tracking error: obj prev %i/%i sp %.3f/%.3f new %i/%i sp %.3f/%.3f\n",
      pPrevPos->x, pPrevPos->y, pPrevPos->nSpeedX, pPrevPos->nSpeedY,
      pNewPos->x, pNewPos->y, pNewPos->nSpeedX, pNewPos->nSpeedY);
    */
    pNewPos->nSpeedX = pNewPos->nSpeedY = 0;
    pNewPos->nSpeedFrames = 1;
  }
  else
  {
    if (pNewPos->nSpeedFrames < 100)
      pNewPos->nSpeedFrames++;

    if (pNewPos->nRadius == 1 && fabs (pNewPos->nSpeedX) <= 2 && fabs (pNewPos->nSpeedY) <= 2)
    {
      printf ("shot tracking error: obj %i/%i sp %.3f/%.3f\n",
        pNewPos->x, pNewPos->y, pNewPos->nSpeedX, pNewPos->nSpeedY);

      pNewPos->nSpeedX = pNewPos->nSpeedY = 0;
      pNewPos->nSpeedFrames = 1;
    }
  }

  if (*(((DWORD*) &pNewPos->nSpeedX) + 0) == 0 
    && (*(((DWORD*) &pNewPos->nSpeedX) + 1) == 0x7FF00000 || *(((DWORD*) &pNewPos->nSpeedX) + 1) == 0xFFF80000))
  {
    printf ("tracking error: obj %i/%i sp %.3f/%.3f\n",
      pNewPos->x, pNewPos->y, pNewPos->nSpeedX, pNewPos->nSpeedY);
  }

  pNewPos->nShotCnt   = pPrevPos->nShotCnt > 0 ? pPrevPos->nShotCnt - 1 : pPrevPos->nShotCnt;
  pNewPos->nMultiShot = (pPrevPos->nMultiShot & 0x0F) != 0 ? pPrevPos->nMultiShot - 1 : 0;
  pNewPos->bMyShot    = pPrevPos->bMyShot;
  pNewPos->pPrev      = pPrevPos;
  pPrevPos->bPrev     = true;

  if (pNewPos->nRadius == 1 && pNewPos->nSpeedX * pNewPos->nSpeedX + pNewPos->nSpeedY * pNewPos->nSpeedY > 70)
  {
    if (pNewPos->nSpeedFrames == 2 
      && DistanceSquared (pPrevPos->x, pPrevPos->y, ship.x, ship.y) < (SHIP_RADIUS + 12) * (SHIP_RADIUS + 12))
    {
      //printf ("correct shot %i/%i sp %.3f/%.3f\n",
      //  pNewPos->x, pNewPos->y, pNewPos->nSpeedX, pNewPos->nSpeedY);

      if (pNewPos->nSpeedX > 0)
        pNewPos->nSpeedX -= 1;
      else
        pNewPos->nSpeedX += 1;

      if (pNewPos->nSpeedY > 0)
        pNewPos->nSpeedY -= 1;
      else
        pNewPos->nSpeedY += 1;
    }
  }

  /*
  printf ("obj at %i/%i prev %i/%i frames %i speed %f/%f prev %f/%f\n", 
    pNewPos->x, pNewPos->y, pPrevPos->x, pPrevPos->y, 
    pNewPos->nSpeedFrames, pNewPos->nSpeedX, pNewPos->nSpeedY,
    pPrevPos->nSpeedX, pPrevPos->nSpeedY);
    */
}


//-----------------------------------------------------------------------------
static ObjInfo* GetAsteroidObject (GameStatus* pGame, int nIndex)
{
  return &pGame->asteroids[nIndex].info;
}


//-----------------------------------------------------------------------------
static ObjInfo* GetShotObject (GameStatus* pGame, int nIndex)
{
  return &pGame->shots[nIndex].info;
}


//-----------------------------------------------------------------------------
bool GameStatus::TrackMatch (ObjInfo* pPrevPos, ObjInfo* pCurrPos)
{
  int nDistance;

  /*
  If distance is within MIN_TRACK_DIST (shots: MIN_TRACK_DIST < dist < 
  MAX_TRACK_DIST), check speed. If previous speed is zero or previous position 
  plus speed is almost new position, we found a match.
  To reduce errors with shots, a shot within MIN_SHOT_DIST from the ship 
  is not accepted as new position of a previous shot.
  */
            
  if (pPrevPos->nRadius != pCurrPos->nRadius)
    return false;

  nDistance = DistanceSquared (pCurrPos->x, pCurrPos->y, pPrevPos->x, pPrevPos->y);

  if (pCurrPos->nRadius == 1 
      && nDistance >= MIN_TRACK_DIST*MIN_TRACK_DIST
      && nDistance <= MAX_TRACK_DIST*MAX_TRACK_DIST
    || pCurrPos->nRadius != 1
      && nDistance <= MIN_TRACK_DIST*MIN_TRACK_DIST)
  {
    if (pPrevPos->nSpeedX == 0 && pPrevPos->nSpeedY == 0)  // speed can not be checked
      return true;

    nDistance = DistanceSquared (pCurrPos->x, pCurrPos->y, 
      pPrevPos->x + round (pPrevPos->nSpeedX), pPrevPos->y + round (pPrevPos->nSpeedY));

    return nDistance <= 5;  // max 1 pixel difference in one and two in other direction
  }

  return false;
}


//-----------------------------------------------------------------------------
void GameStatus::CheckTrackVariant (int nPrevCount, int nCurrCount,
                                    ObjInfo* (*fGetObject) (GameStatus*, int),
                                    int nPrevIndex, int nCurrIndex, 
                                    int nSkipped, bool abInserted[], 
                                    bool abRemoved[], bool bInserted, 
                                    bool bRemoved, int* pnMinSkipped, 
                                    bool abMinInserted[], bool abMinRemoved[])
{
  ObjInfo* pPrevPos;
  ObjInfo* pCurrPos;
  int      i;

  if (bInserted)
  {
    abInserted[nCurrIndex++] = true;
    nSkipped++;
  }

  if (bRemoved)
  {
    abRemoved [nPrevIndex++] = true;
  }

  while (nPrevIndex < nPrevCount && nCurrIndex < nCurrCount)
  {
    pPrevPos = (*fGetObject) (pPrevGame, nPrevIndex);
    pCurrPos = (*fGetObject) (this, nCurrIndex);

    if (TrackMatch (pPrevPos, pCurrPos))
    {
      abInserted[nCurrIndex++] = false;
      abRemoved [nPrevIndex++] = false;
    }
    else
    {
      if (nSkipped >= 4 || nSkipped >= *pnMinSkipped) // this reduces time consumption
        return;

      CheckTrackVariant (nPrevCount, nCurrCount, fGetObject, 
        nPrevIndex, nCurrIndex, nSkipped, abInserted, abRemoved, true, false, 
        pnMinSkipped, abMinInserted, abMinRemoved);

      if (nSkipped >= *pnMinSkipped) // this reduces time consumption
        return;

      CheckTrackVariant (nPrevCount, nCurrCount, fGetObject, 
        nPrevIndex, nCurrIndex, nSkipped, abInserted, abRemoved, false, true, 
        pnMinSkipped, abMinInserted, abMinRemoved);

      // inserted and removed tested in following loops
      abInserted[nCurrIndex++] = true;
      abRemoved [nPrevIndex++] = true;
      nSkipped++;

      if (nSkipped >= *pnMinSkipped) // this reduces time consumption
        return;
    }
  }

  nSkipped += nCurrCount - nCurrIndex;

  if (nSkipped < *pnMinSkipped)
  {
    *pnMinSkipped = nSkipped;

    for (i = 0; i < nPrevIndex; i++)
    {
      abMinRemoved[i] = abRemoved[i];
    }

    while (nPrevIndex < nPrevCount)
    {
      abMinRemoved[nPrevIndex++] = true;
    }

    for (i = 0; i < nCurrIndex; i++)
    {
      abMinInserted[i] = abInserted[i];
    }

    while (nCurrIndex < nCurrCount)
    {
      abMinInserted[nCurrIndex++] = true;
    }
  }
}


//-----------------------------------------------------------------------------
void GameStatus::GetSpeedOfObjects (int nPrevCount, int nCount, 
                                    ObjInfo* (*fGetObject) (GameStatus*, int))
{
  bool abInserted[MAX_ASTEROIDS], abRemoved[MAX_ASTEROIDS];
  bool abMinInserted[MAX_ASTEROIDS], abMinRemoved[MAX_ASTEROIDS];
  int  i, j, nMinSkipped;

  /*
  The order of existing objects remain the same; new objects are inserted 
  anywhere, and if an object vanishes, the following objects are shifted up.
  Therefore compare positions of previous objects (0 <= i < nPrevCount) 
  with current objects (0 <= j < nCount). If distance is within MIN_TRACK_DIST 
  (shots: MIN_TRACK_DIST < dist < MAX_TRACK_DIST), check speed. If previous 
  speed is zero or previous position plus speed is almost new position, we found 
  the object and increase i and j. If we did not find it, we repeat the check 
  with temporarily incremented j, then temporarily incremented i. This is 
  repeated until the first match is found.
  To reduce errors and complexity, multiple variants are tested for maximum 
  number of matches. Each variant consists of one int for number of matches 
  and a bool array with marks for inserted and a bool array with marks for 
  removed objects. Each time we find that objects at i and j do not match, 
  we create a variant which assumes an inserted object, another variant 
  assuming a removed object, and a third variant that assumes an inserted and 
  removed object at the same time. Then these variants are checked for matches. 
  In the end, the largest number of matches in the variants is used to 
  calculate the new speed of the objects.
  */

  for (i = 0; i < MAX_ASTEROIDS; i++)
  {
    abInserted[i] = false;
    abRemoved [i] = false;
  }

  nMinSkipped = MAX_ASTEROIDS*2;

  CheckTrackVariant (nPrevCount, nCount, fGetObject, 
    0, 0, 0, abInserted, abRemoved, false, false, 
    &nMinSkipped, abMinInserted, abMinRemoved);

  i = j = 0;

  while (i < nPrevCount && j < nCount)
  {
    if (abMinRemoved[i])
    {
      i++;
    }
    else if (abMinInserted[j])
    {
      j++;
    }
    else
    {
      SetSpeed ((*fGetObject) (pPrevGame, i), (*fGetObject) (this, j));
      i++;
      j++;
    }
  }
}


//-----------------------------------------------------------------------------
static void GetAsteroidSpeed (void* pv)
{
  bool* pbCompleted = (bool*) pv;

  g_pSimGame->GetSpeedOfObjects (
    g_pSimGame->pPrevGame->nAsteroids, 
    g_pSimGame->nAsteroids, GetAsteroidObject);

  *pbCompleted = true;
}


//-----------------------------------------------------------------------------
void GameStatus::GetSpeed ()
{
  // assume no frame loss!

  // the work can be done by two threads if one thread tracks the asteroids,
  // the other thread tracks the shots, saucer and ship

  // 1. get asteroids speed vector
#if NUM_THREADS > 1
  bool bThreadComplete = false;

  _beginthread (GetAsteroidSpeed, 10000, &bThreadComplete);

#else

  GetSpeedOfObjects (pPrevGame->nAsteroids, nAsteroids,
                     GetAsteroidObject);

#endif

  // 2. get shots speed vector
  GetSpeedOfObjects (pPrevGame->nShots, nShots,
                     GetShotObject);

  // 3. get saucer speed
  saucer.nSpeedX = 0;
  saucer.nSpeedY = 0;
  saucer.nShotCnt= 0;

  if (saucer_present && pPrevGame->saucer_present)
  {
    saucer.nSpeedX = saucer.x - pPrevGame->saucer.x;
    saucer.nSpeedY = saucer.y - pPrevGame->saucer.y;

    if (saucer.nSpeedY >= SIZE_Y / 2)
    {
      // can happen if position changed from bottom to top
      saucer.nSpeedY -= SIZE_Y;
    }
    else if (saucer.nSpeedY <= - SIZE_Y / 2)
    {
      // can happen if position changed from top to bottom
      saucer.nSpeedY += SIZE_Y;
    }

    if (abs (saucer.nSpeedX - pPrevGame->saucer.nSpeedX) < 2
      && abs (saucer.nSpeedY - pPrevGame->saucer.nSpeedY) < 2)
    {
      if (pPrevGame->saucer.nShotCnt >= 1)
        saucer.nShotCnt = pPrevGame->saucer.nShotCnt - 1;
    }
    // else complete other speed, shoot again
  }

  // 4. get ship speed
  if (ship_present && pPrevGame->ship_present)
  {
    SetSpeed (&pPrevGame->ship, &ship);
    ship.nSpeedFrames = 5;
  }

#if NUM_THREADS > 1
  while (!bThreadComplete)
  {
    Sleep (0);
  }
#endif
}


/*-------------------------------------------------------------------------*/
static void Normalize (int& dx, int& dy)
{
  while (dx < -512) dx += SIZE_X; // dx normalisieren auf -512 ... 511
  while (dx > 511) dx -= SIZE_X;
  while (dy < -384) dy += SIZE_Y;  // dy normalisieren auf -384 ... 383
  while (dy > 383) dy -= SIZE_Y;
}


//-----------------------------------------------------------------------------
static int GetIndex (int nPos, int nSize)
{
  while (nPos < 0)
    nPos += nSize;

  while (nPos >= nSize)
    nPos -= nSize;

  return nPos;
}


/*-------------------------------------------------------------------------*/
bool GameStatus::GetBestShot (int nThread,
                              int x, int y, int nRadius, int nNextShot, 
                              int nFramesPassed, 
                              int* pnTurns, int* pnWaits, int* pnFrames)
{
  int  i, j, x1, y1, nFrames, nTurns, nWaits, nAbsTurns;
  int  nMinTurns, nMinWaits, nMinFrames, nMinPrio;
  int* pnOffsets;
  AngleAndFrames* pShots;

  switch (nRadius)
  {
  case SML_AST_RADIUS: 
    pnOffsets = g_anCircleSml;
    break;

  case MED_AST_RADIUS: 
    pnOffsets = g_anCircleMed; 
    break;

  case BIG_AST_RADIUS: 
    pnOffsets = g_anCircleBig; 
    break;

  default: 
    pnOffsets = g_anCircleSml; 
    printf ("Illegal radius\n"); 
    break;
  }

  nMinPrio = INVALID;

  // check all positions in a circle depending on the radius
  for (j = 0; pnOffsets[j] != INVALID && !pSim->abTermThread[nThread]; )
  {
    x1 = x + pnOffsets[j++];
    y1 = y + pnOffsets[j++];

    pShots   = g_aapShotTable[GetIndex (x1, SIZE_X)][GetIndex (y1, SIZE_Y)];

    if (pShots != NULL)
    {
      for (i = 0; (nFrames = pShots[i].nFrames) != 0xFF && !pSim->abTermThread[nThread]; i++)
      {
        nTurns = (int) pShots[i].nAngle - (int) g_nShotAngleIndex;

        if (nTurns > 128)
        {
          nTurns = nTurns - 256;  // turn in other direction
        }
        else if (nTurns <= -128)
        {
          nTurns = nTurns + 256;  // turn in other direction
        }

        nAbsTurns = abs (nTurns);

        if (nNextShot > nAbsTurns + LATENCY)
        {
          // no shot immediately after the turns
          nWaits = nNextShot - nAbsTurns - LATENCY;
        }
        else
        {
          // shot possible after the turns
          nWaits = 0;
        }

        // the number of frames until hit is number of turns
        // plus number of frames until hit plus number of frames 
        // needed to wait for a shot to be available
        nFrames += nAbsTurns + nWaits;

        // check if the object is at position x,y after these frames passed
        if (nFramesPassed >= nFrames + LATENCY)
        {
          // the shot is early enough
          nWaits += nFramesPassed - nFrames - LATENCY;

          if (nFrames < nMinPrio)
          {
            nMinPrio   = nFrames;
            nMinTurns  = nTurns;
            nMinWaits  = nWaits;
            nMinFrames = pShots[i].nFrames;

            //if (nWaits == 0)
            {
/*
              if (nTurns == 0)
              {
                printf ("Hit at %i/%i after %i frames at %i/%i with angle %f\n",
                  x, y, nMinFrames, x1, y1, g_anShotAngles[g_nShotAngleIndex]);
              }
*/

              //goto found;
            }
          }
        }
        // else shot is too late
      }
    }
  }

  if (nMinPrio != INVALID)
  {
//found:
    *pnTurns  = nMinTurns;
    *pnWaits  = nMinWaits;
    *pnFrames = nMinFrames;

    return true;
  }

  return false;
}


/*-------------------------------------------------------------------------*/
void GameStatus::CorrectAngle ()
{
#if !PLAYBACK
  double nAngle;
  int    i, j;

  // get angle between (1,0) and pGame->ship_dx, pGame->ship_dy
  nAngle = acos (ship_dx / sqrt (ship_dx*ship_dx + ship_dy*ship_dy)) / 2 / PI * 360;

  if (ship_dy < 0)
    nAngle = 360-nAngle;

  //printf ("angle of ship is %f, shot is %f\n", nAngle, g_anShotAngles[g_nShotAngleIndex]);
  i = g_anPrevAngles[g_nLatencyPos];

  // check if correction necessary
  if (nAngle < 11 && g_anShotAngles[i] >= 350)
    nAngle += 360;
  else if (g_anShotAngles[i] < 11 && nAngle >= 350)
    nAngle -= 360;

  if (nAngle >= g_anShotAngles[i] + 7)
  {
    Debug ("frame %i angle of ship is %f, shot is %f\n", pSim->nFrameCnt, nAngle, g_anShotAngles[i]);

    while (nAngle >= g_anShotAngles[i] + 7)
    {
      for (j = 0; j < LATENCY; j++)
        g_anPrevAngles[j]++;

      g_nShotAngleIndex++;
      i++;
    }

    Debug ("angle increased to %f\n", g_anShotAngles[i]);
  }
  else if (nAngle <= g_anShotAngles[i] - 7)
  {
    Debug ("frame %i angle of ship is %f, shot is %f\n", pSim->nFrameCnt, nAngle, g_anShotAngles[i]);

    while (nAngle <= g_anShotAngles[i] - 7)
    {
      for (j = 0; j < LATENCY; j++)
        g_anPrevAngles[j]--;

      g_nShotAngleIndex--;
      i--;
    }

    Debug ("angle decreased to %f\n", g_anShotAngles[i]);
  }

#endif
}


/*-------------------------------------------------------------------------*/
void GameStatus::SetOwnShots ()
{
  int i, k;
  ObjInfo* pPos;

  // look for my own shots
  nMyShots = 0;

  for (i = 0; i < nShots; i++)  
  {
    pPos = &shots[i].info;

    if (pPos->nSpeedX == 0 && pPos->nSpeedY == 0) // not tracked
    {
      k = DistanceSquared (pPos->x, pPos->y, ship.x, ship.y);

      if (k <= MIN_SHOT_DIST*MIN_SHOT_DIST)
      {
        // first occurrance near ship -> my shot, passed through tracking
        pPos->bMyShot = true;
      }
    }

    if (i >= MAX_OWN_SHOTS && pPos->bMyShot)
    {
      // this must be a saucer shot (should never be executed)
      printf ("Saucer shot #%i was set to be my shot!!!! Frame %i\n", i, pSim->nFrameCnt);
      pPos->bMyShot = false;
    }

    if (pPos->bMyShot)
    {
      nMyShots++;
    }
  }
}


/*-------------------------------------------------------------------------*/
void GameStatus::GetShotAvailable ()
{
  int  nShotAvail, nShotsAvailable;
  int* pnMinShotAvail;
  int  i, nFree;

  nShotsAvailable = MAX_OWN_SHOTS - nMyShots - (int) (pSim->nFired != 0);

  nFree = 0;

  for (i = 0; i < MAX_OWN_SHOTS; i++)
  {
    if (pSim->anMyShotsHit[i] == 0)
    {
      nFree++;
    }
  }

  for (i = 0; i < MAX_OWN_SHOTS; i++)
  {
    if (pSim->anMyShotsHit[i] == 1)// && nMissed > 0)
    {
      if (nFree < nShotsAvailable)
      {
        // if the asteroid is missed because the shot hit something else, set value to 0
        nFree++;
      }
      else
      {
        // if the asteroid is missed because the shot misses, increase value again
        pSim->anMyShotsHit[i] = 5; // no better value available
      }

      //nMissed--;
    }
  }

  // calculate frames until shot available
  nShotAvail = INVALID;

  for (i = 0; i < MAX_OWN_SHOTS; i++)
  {
    // decrease number of frames until hit (one frame has passed since last check)
    if (pSim->anMyShotsHit[i] > 0)
      pSim->anMyShotsHit[i]--;

#if RECORD >= 3
    Debug ("anMyShots[%i] = %i\n", i, pSim->anMyShotsHit[i]);
#endif

    if (pSim->anMyShotsHit[i] < nShotAvail)
    {
      pnMinShotAvail = &pSim->anMyShotsHit[i];
      nShotAvail     = *pnMinShotAvail;
    }
  }

  if (nShotAvail > 0 && nShotsAvailable > 0)
  {
    // shot is available but not yet known
    *pnMinShotAvail = 0;
    nShotAvail      = 0;
  }

  nNextShot      = LATENCY + (int) (pSim->nFired == FIRE_CNT) + nShotAvail;
  pSim->pShotPos = NULL;
  pSim->bMoveToTarget = false;

  if (pSim->nFired > 0)
    pSim->nFired--;

  //if (nNextShot > LATENCY)
    //printf ("Next shot after %i frames\n", nNextShot);
}


/*-------------------------------------------------------------------------*/
void GameStatus::GetObjectList ()
{
  int      i, j;
  ObjInfo* pPos;

  // get objects; shots also simulated for hit checking
  pObjectList = NULL;

  // get object list according to priority
  for (j = -3; j < nAsteroids; j++)
  {
    if (j == -1)
    {
      pPos = &saucer;

      if (!saucer_present)
      {
        continue;
      }

      if (pPos->nShotCnt > 0 || pPos->nSpeedX == 0 && pPos->nSpeedY == 0)
        continue;
    }
    else if (j < 0)
    {
      // check last two shots
      i = nShots + j + 1;

      if (i < 0)
        continue; // no shot

      pPos = &shots[i].info;

      if (pPos->bMyShot || pPos->nSpeedX == 0 && pPos->nSpeedY == 0)
        continue;
    }
    else
    {
      pPos = &asteroids[j].info;

      // reduction of tracked objects:
      // don't track objects without speed or already shot at
      // if object is behind the ship and enough objects are in front, ignore it
      if (pPos->nShotCnt > 0 && pPos->nMultiShot == 0 
        || pPos->nSpeedX == 0 && pPos->nSpeedY == 0)
      {
        continue;
      }
    }

    pPos->nPrio = j;
    pPos->nHits = NO_HIT_MALUS;
    pPos->bScanned = false;
    pPos->nSimCnt  = 1;
    pPos->nSimX    = pPos->x - ship.x + MIDDLE_X;
    pPos->nSimY    = pPos->y - ship.y + MIDDLE_Y;

    // insert in front
    pPos->pNext = pObjectList;
    pObjectList = pPos;
  }
}


/*-------------------------------------------------------------------------*/
void SimInfo::StoreBestShot (ObjInfo* pPos, int nThread)
{
#if PLAYBACK || RECORD >= 3
  Debug ("Thread %i Obj at %i/%i min %i: %i %i %i %i\n", nThread, 
    pPos->x, pPos->y, nFramesShoot, pPos->nFramesShoot, pPos->nTurns, pPos->nWaits, pPos->nFrames);
#endif

  resultSem.Lock ();

  if (pPos->nFramesShoot < nFramesShoot)
  {
    nFramesShoot = pPos->nFramesShoot;
    nTurn        = pPos->nTurns;
    nWaits       = pPos->nWaits;
    nFrames      = pPos->nFrames;
    pSelPos      = pPos;
  }

  resultSem.Unlock ();
}


/*-------------------------------------------------------------------------*/
void GameStatus::GetBestShot (int nThread)
{
  double   nSpeedX, nSpeedY;
  int      i, k, nIndexX, nIndexY, nSimCnt, nRadius, nMaxShots, dx, dy;
  int      nFrames, nTurns, nWaits, nFramesShoot;
  int      nScannedObjs;
  bool     bSimulating;
  ObjInfo* pPos;

  /*
  Simulate flight of every object which has a speed for SHOT_RANGE frames.
  Estimate number of frames needed to shoot at it and search hitting shot 
  if this number of frames passed in simulation.
  The number of frames that will pass before the shot appears is 
    latency (the number of frames passed from command to execution)
    + 1 if we fired in previous frame (fire only every second frame)
    + number of frames until shot is available (if 4 are fired)
  If the object is already shot at, it is not tracked to save time
  */

  nScannedObjs = 0;
  nSimCnt      = nAsteroids <= 3 ? (6 - nAsteroids) * TURNS_SIM / 2 : TURNS_SIM;
  bSimulating  = true;
  pPos         = NULL;

  while (!pSim->abTermThread[nThread])
  {
    objectSem.Lock ();

    do
    {
      if (pPos != NULL)
        pPos = pPos->pNext;

      if (pPos == NULL)
      {
        // at end of list
        if (bSimulating)
        {
          // last pass simulated something, start new pass
          bSimulating = false;
          pPos        = pObjectList;
        }
        else
        {
          objectSem.Unlock ();

          // all objects completely simulated
          goto term;
        }
      }
    }
    while (pPos == NULL || pPos->bScanned);

    pPos->bScanned = true;

    objectSem.Unlock ();

    if (pPos->nSimCnt == 1)
      nScannedObjs++;

    nFramesShoot = INVALID;

    // coordinates are in screen positions, change to ship relative position
    nSpeedX = pPos->nSpeedX - ship.nSpeedX;
    nSpeedY = pPos->nSpeedY - ship.nSpeedY;

    for (i = 0; i < 10 && pPos->nSimCnt < nSimCnt; i++, pPos->nSimCnt++)
    {
      bSimulating = true;

      if (pSim->abTermThread[nThread])
      {
        //Debug ("Frame %i Terminate #%i during scanning %i of %i objs (scanned %i, pos %i, time %lu)\n", 
        //  pSim->nFrameCnt, nThread, nObjectPos2, nObjectCnt2, nScannedObjs, i,
        //  timeGetTime ());
        break;  // no tracking of collisions!
      }

      pPos->nSimX += nSpeedX;
      pPos->nSimY += nSpeedY;

      nIndexX = GetIndex (round (pPos->nSimX), SIZE_X);
      nIndexY = GetIndex (round (pPos->nSimY), SIZE_Y);

      if (pPos->nPrio >= -1 && pPos->nPrio < MAX_ASTEROIDS)
      {
        if (GetBestShot (nThread, nIndexX, nIndexY, pPos->nRadius, nNextShot, pPos->nSimCnt,
                         &nTurns, &nWaits, &nFrames))
        {
          // we found a possible hit
          if (nTurns == 0 && pSim->nFired != FIRE_CNT && nWaits == 0)
          {
            pSim->nFired = FIRE_CNT;
            nNextShot   += 2;
            pSim->pShotPos = pPos;

            if (nAsteroids == 1 && !saucer_present && pPos->nRadius == SML_AST_RADIUS)
              pPos->nShotCnt = nFrames / 4; // faster repetition
            else
              pPos->nShotCnt = nFrames + (LATENCY+1); // a few more frames needed, else double shot

            if (pPos != &saucer && pPos->nRadius >= MED_AST_RADIUS)
            {
              // multiple shots possible
              pPos->nMultiShot += 0x10; // one shot added 
              pPos->nMultiShot |= 0x02; // for 3 frames

              // if the asteroid moves in same direction as shot, it is more 
              // likely that the shots will hit
              nMaxShots = pPos->nRadius / MED_AST_RADIUS + 2; // 3 for medium, 4 for large size

              dx = pPos->nSpeedX * 1000;
              dy = pPos->nSpeedY * 1000;
              Normalize (dx, dy);

              nMaxShots -= abs (ship_dx * dy - ship_dy * dx) / 200000;

              if (pPos->nMultiShot >> 4 >= nMaxShots)
              {
                // maximum number of shots
                pPos->nMultiShot = 0;
              }
            }

            // store number of frames until hit
            for (k = 0; k < MAX_OWN_SHOTS; k++)
            {
              // decrease number of frames until hit (one frame has passed since last check)
              if (pSim->anMyShotsHit[k] == 0)
              {
                pSim->anMyShotsHit[k] = nFrames;
        //        printf ("My shot #%i needs %u to hit\n", i, nFrames);
                break;
              }
            }

            // don't look further
            pPos->nSimCnt = nSimCnt;
            break;
          }
          else
          {
            nFramesShoot = abs (nTurns) + nWaits + nFrames / 4 + pPos->nHits;

            if (nAsteroids >= MAX_ASTEROIDS-2 && pPos->nRadius != 8)
            {
              // only one new asteroid will appear, this costs points
              nFramesShoot += pPos->nRadius;
            }

            if (pPos->nSpeedFrames < 3 && pPos->nRadius == 8)
            {
              // bad speed malus
              nFramesShoot += 5 - pPos->nSpeedFrames;
            }

            if (pPos->nPrio == -1)
            {
              if (pSim->nFrameCnt >= MAX_FRAME_CNT-600)
              {
                // saucer priority
                //nFramesShoot -= 10;
                // saucer on last level can increase the score by 1000
                nFramesShoot /= 2;
              }
              else
              {
                // saucer priority higher
                nFramesShoot -= 20;
                // saucer priority lower
                //nFramesShoot *= nFramesShoot / 2;
              }
            }
/*
            if (pPos->nShotCnt < 0)
            {
              // multiple shot priority
              //nFramesShoot -= 10;
            }

            if (pSim->pSelPos != NULL && pPos->pPrev == pSim->pSelPos)
            {
              // previously selected priority
              //nFramesShoot -= 10;

              //printf ("prev sel %i %i/%i %i/%i %i %i %i %i\n",
              //  nNextShot, pPos->x, pPos->y, nIndexX, nIndexY, nFramesShoot, nTurns, nWaits, nFrames);
            }

            if (pPos->nMultiShot != 0)
            {
              // multishot priority depends on distance: nearer = lower priority
              nFramesShoot += SHOT_RANGE / 2 - nFrames;
            }
*/
            pPos->nFramesShoot = nFramesShoot;
            pPos->nFrames      = nFrames     ;
            pPos->nTurns       = nTurns      ;
            pPos->nWaits       = nWaits      ;
            pPos->nFrames      = nFrames     ;

            pSim->StoreBestShot (pPos, nThread);
          }

          if (pPos->nHits != NO_HIT_MALUS)
            break;

          // don't look for shot anymore (pPos->nPrio < MAX_ASTEROIDS), but still for hit
          pPos->nPrio = MAX_ASTEROIDS;
        }
      }

      // check for collision with ship
      nIndexX -= MIDDLE_X;
      nIndexY -= MIDDLE_Y;
      nRadius  = SHIP_RADIUS + pPos->nRadius;

      if (nIndexX * nIndexX + nIndexY * nIndexY < nRadius * nRadius)
      {
        if (pPos->nSimCnt <= MIN_HIT_FRAMES && pPos->nSimCnt >= 2)
        {
          Debug ("Object at %i/%i speed %.3f/%.3f radius %i hits ship at %i/%i in %i frames\n", 
            pPos->x, pPos->y, pPos->nSpeedX, pPos->nSpeedY,
            pPos->nRadius, ship.x, ship.y, pPos->nSimCnt);

          // collision can not be prevented
          bShipWillBeHit = true;
        }
        else if (pPos->nHits == NO_HIT_MALUS)
        {
          pPos->nHits = pPos->nSimCnt;

          // this changes priority
          if (pPos->nPrio == MAX_ASTEROIDS)
          {
            // a shot has already been found
            //Debug ("Reducing prio of %i/%i by %i\n", pPos->x, pPos->y, NO_HIT_MALUS - pPos->nSimCnt);
            pPos->nFramesShoot -= NO_HIT_MALUS - pPos->nSimCnt;

            pSim->StoreBestShot (pPos, nThread);
          }
        }

        // don't look further
        pPos->nSimCnt = nSimCnt;
        break;
      }
    }

    pPos->bScanned = false;
  }

term:;
  pSim->resultSem.Lock ();

  if (pSim->nTurn != INVALID && ShootableTargets () <= 3)
  {
    if (abs (pSim->nTurn) > TURNS_CIRCLE / 2 + 5)
    {
      pSim->bMoveToTarget = true;

      if (pSim->nTurn > 0)
        pSim->nTurn -= TURNS_CIRCLE;
      else
        pSim->nTurn += TURNS_CIRCLE;
    }
    else if (pSim->nWaits >= 10)
    {
      pSim->bMoveToTarget = true;
    }
  }

  pSim->resultSem.Unlock ();

  //printf ("fire %i turn %i wait %i frames %i angle %.3f\n",
  //  pSim->nFired, nMinTurns, nMinWaits, nMinFrames, 
  //  g_anShotAngles[(g_nShotAngleIndex + nMinTurns) & 0xFF]);
#if RECORD >= 3
  Debug ("Frame %i thread #%i scanned %i objs (term %i time %lu)\n", 
    pSim->nFrameCnt, nThread,
    nScannedObjs, (int) pSim->abTermThread[nThread], timeGetTime ());
#endif
}


/*-------------------------------------------------------------------------*/
bool GameStatus::ShipSpeedHigh (int dx, int dy)
{
  // maximum speed depends on distance from x, y (nearer = slower)
  // if thrust decreases the speed (i.e. x/y is opposite direction of 
  // ship.nSpeedX / ship.nSpeedY), then return false
  // absolute maximum to fire correctly is sqrt (0.5)
  int    nDist = dx * dx + dy * dy;
  double nSpeedX = ship.nSpeedX;
  double nSpeedY = ship.nSpeedY;
  double nSpeed = nSpeedX * nSpeedX + nSpeedY * nSpeedY;
  double nMax;

  // check for thrust reducing speed
  nSpeedX += (double) dx / 10000;
  nSpeedY += (double) dy / 10000;

  if (nSpeedX * nSpeedX + nSpeedY * nSpeedY < nSpeed)
  {
    return false;
  }

  if (nDist >= 500*500)
    nMax = 1;
  else if (nDist >= 400*400)
    nMax = 0.4;
  else if (nDist >= 300*300)
    nMax = 0.2;
  else if (nDist >= 200*200)
    nMax = 0.1;
  else if (nDist >= 100*100)
    nMax = 0.05;
  else
    nMax = 0.01;

  return nSpeed >= nMax;
}


/*-------------------------------------------------------------------------*/
int GameStatus::ShootableTargets ()
{
  int i, nShootable;

  if (saucer_present && saucer.nShotCnt == 0)
    nShootable = 1;
  else
    nShootable = 0;

  for (i = 0; i < nAsteroids; i++)
  {
    if (asteroids[i].info.nShotCnt == 0)
      nShootable++;
  }

  if (nShootable == 0)
    return MAX_ASTEROIDS;

  return nShootable;
}


/*-------------------------------------------------------------------------*/
bool GameStatus::MoveToTarget ()
{
  if (pSim->bMoveToTarget)
  {
    return !ShipSpeedHigh (ship_dx, ship_dy);
  }

  return false;
}

          
/*-------------------------------------------------------------------------*/
int GameStatus::ShipAtBorder ()
{
  int i, nSpeedX, nSpeedY;

  nSpeedX = (int) (ship.nSpeedX * TOO_CLOSE_TO_BORDER);
  nSpeedY = (int) (ship.nSpeedY * TOO_CLOSE_TO_BORDER);

  for (i = TOO_CLOSE_TO_BORDER; i <= CLOSE_TO_BORDER; i += TOO_CLOSE_TO_BORDER)
  {
    if (ship.x < MIN_X + i - (nSpeedX > 0 ? 0 : nSpeedX) 
      || ship.x > MAX_X - i - (nSpeedX < 0 ? 0 : nSpeedX)
      || ship.y < MIN_Y + i - (nSpeedY > 0 ? 0 : nSpeedY) 
      || ship.y > MAX_Y - i - (nSpeedY < 0 ? 0 : nSpeedY) )
    {
      return i;
    }
  }

  return 0;
}


/*-------------------------------------------------------------------------*/
bool GameStatus::DefaultPosition (int& nDefaultOffsetX)
{
  if (ship.x >= MIN_X + SIZE_X / 2)
    nDefaultOffsetX = 0;
  else
    nDefaultOffsetX = - SIZE_X * 6 / 8; // DEFAULT_X is at 7/8

  return ship.x >= DEFAULT_MIN_X + nDefaultOffsetX 
    && ship.x <= DEFAULT_MAX_X + nDefaultOffsetX
    && ship.y >= DEFAULT_MIN_Y && ship.y <= DEFAULT_MAX_Y;
}


/*-------------------------------------------------------------------------*/
int GameStatus::AngleDifference (int dx, int dy)
{
  double nAngle;
  int    nResult;

  nAngle = atan2 (dy, dx) / (2*PI) * 360;

  if (nAngle < 0)
    nAngle = 360 + nAngle;

  nResult = round (nAngle - g_anShotAngles[g_nShotAngleIndex]);

  if (nResult > 180)
    nResult -= 360;
  else if (nResult < -180)
    nResult += 360;

  return nResult;
}


/*-------------------------------------------------------------------------*/
bool GameStatus::IsShipDirection (int dx, int dy)
{
  int nDiffAngle;

  nDiffAngle = AngleDifference (dx, dy);

  return abs (nDiffAngle) < 20;
}


/*-------------------------------------------------------------------------*/
bool GameStatus::TurnToDefault (int& nDefaultOffsetX, int& dy)
{
  dy = (ship.y >= DEFAULT_MIN_Y && ship.y <= DEFAULT_MAX_Y) ? 0 : DEFAULT_Y - ship.y;

  return !DefaultPosition (nDefaultOffsetX) 
    && (ShipAtBorder () == TOO_CLOSE_TO_BORDER 
      || !ShipSpeedHigh (DEFAULT_X + nDefaultOffsetX - ship.x, dy));
}


/*-------------------------------------------------------------------------*/
bool GameStatus::MoveToDefault ()
{
  int nDefaultOffsetX, dy;

  return TurnToDefault (nDefaultOffsetX, dy) 
    && IsShipDirection (DEFAULT_X + nDefaultOffsetX - ship.x, dy);
}


/*-------------------------------------------------------------------------*/
void GameStatus::InterpretScreen (FramePacket &packet)
{
  unsigned short vector_ram[512];
  int dx, dy, sf, vx, vy, vz, vs;
  int v1x = 0;
  int v1y = 0;
  int shipdetect = 0;

  clear();

  /* Vektor-RAM in 16-Bit-Worte konvertieren. War in der ersten Version mal ein sportlicher
  Typecast: unsigned short *vector_ram = (unsigned short*)packet.vectorram;
  Das klappt aber nur auf Little-Endian-Systemen, daher besser portabel: */
  for (int i=0; i<512; ++i)
    vector_ram[i] = (unsigned char)packet.vectorram[2*i] | (unsigned char)packet.vectorram[2*i+1] << 8;

  if (vector_ram[0] != 0xe001 && vector_ram[0] != 0xe201)
    return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

  int pc = 1;
  while (pc < 512)
  {
    int op = vector_ram[pc] >> 12;
    switch (op)
    {
    case 0xa: // LABS
      vy = vector_ram[pc] & 0x3ff;
      vx = vector_ram[pc+1] & 0x3ff;
      vs = vector_ram[pc+1] >> 12;
      break;
    case 0xb: // HALT
      return;
    case 0xc: // JSRL
      switch (vector_ram[pc] & 0xfff)
      {
      case 0x8f3:
        asteroids[nAsteroids++].set(vx, vy, 1, vs);
        break;
      case 0x8ff:
        asteroids[nAsteroids++].set(vx, vy, 2, vs);
        break;
      case 0x90d:
        asteroids[nAsteroids++].set(vx, vy, 3, vs);
        break;
      case 0x91a:
        asteroids[nAsteroids++].set(vx, vy, 4, vs);
        break;
      case 0x929:
        saucer_present = true;
        saucer.x = vx;
        saucer.y = vy;

        switch (vs)
        {
        case 15: // groes UFO
          saucer.nRadius = BIG_UFO_RADIUS;
          break;
        case 14: // kleines UFO
          saucer.nRadius = SML_UFO_RADIUS;
          break;
        }
        break;
      }  
      break;
    case 0xd: // RTSL
      return;
    case 0xe: // JMPL
      /*
      pc = vector_ram[pc] & 0xfff;
      break;
      */
      return;
    case 0xf: // SVEC
      /*
      dy = vector_ram[pc] & 0x300;
      if ((vector_ram[pc] & 0x400) != 0)
        dy = -dy;
      dx = (vector_ram[pc] & 3) << 8;
      if ((vector_ram[pc] & 4) != 0)
        dx = -dx;
      sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
      vz = (vector_ram[pc] & 0xf0) >> 4;
      */
      break;
    default:
      dy = vector_ram[pc] & 0x3ff;
      if ((vector_ram[pc] & 0x400) != 0)
        dy = -dy;
      dx = vector_ram[pc+1] & 0x3ff;
      if ((vector_ram[pc+1] & 0x400) != 0)
        dx = -dx;
      sf = op;
      vz = vector_ram[pc+1] >> 12;
      if (dx == 0 && dy == 0 && vz == 15)
        shots[nShots++].set(vx, vy);
      if (op == 6 && vz == 12 && dx != 0 && dy != 0)
      {
        switch (shipdetect)
        {
        case 0:
          v1x = dx;
          v1y = dy;
          ++shipdetect;
          break;
        case 1:
          ship_present = true;
          ship.set (vx, vy, SHIP_RADIUS);
          ship_dx = v1x - dx;
          ship_dy = v1y - dy;
          ++shipdetect;
          break;
        }
      }
      else if (shipdetect == 1)
        shipdetect = 0;

      break;
    }
    if (op <= 0xa)
      ++pc;
    if (op != 0xe) // JMPL
      ++pc;
  }   

}


/*-------------------------------------------------------------------------*/
void Player::ReceivePacket (FramePacket &packet)
{
  int        nCnt;
  CString    sAddress;
  UINT       nPort;

  nCnt = g_pSocket->ReceiveFrom ((void*) &packet, sizeof (packet), sAddress, nPort);
}


/*-------------------------------------------------------------------------*/
void Player::SendPacket (KeysPacket &packet)
{
  int nCnt;

  nCnt = g_pSocket->SendTo ((void*) &packet, sizeof (packet), SEND_PORT, g_sIpAddress);
}


/*-------------------------------------------------------------------------*/
static void Simulate (void* pv)
{
  int nThread = (int) pv;

  for (;;)
  {
    WaitForSingleObject (g_ahSimEvent[nThread], INFINITE);

    g_pSimGame->pSim->abTermThread[nThread] = false;

    ResetEvent (g_ahSimEvent[nThread]);

    g_pSimGame->GetBestShot (nThread);
  }
}


/*-------------------------------------------------------------------------*/
void Player::Run (void)
{
  FramePacket frame;
  KeysPacket  keys;
  BYTE        anKeys[MAX_LATENCY];
  BYTE        anPing[MAX_LATENCY];
  ObjInfo*    apShotPos[MAX_LATENCY];
  int         nPingIndex, nDiffAngle, nDefaultOffsetX;
  BYTE        nPing, nDoubledKeys, nIgnoredKeys;
  GameStatus  game1, game2;
  SimInfo     sim;
  GameStatus* pGame;
  GameStatus* pPrevGame;
  BYTE        prevframe = 0;
  int         i, j, nTurn = 0, dx, dy;
  BOOL        bEmpty = TRUE;

  pGame     = &game1;
  pPrevGame = &game2;

  game2.clear();

  game1.pSim = &sim;
  game2.pSim = &sim;
  sim.nFrameCnt = -1;
  sim.nLevel    = 1;

  for (nPingIndex = 0; nPingIndex < MAX_LATENCY; nPingIndex++)
  {
    anKeys[nPingIndex] = '@';
    anPing[nPingIndex] = nPingIndex;
    apShotPos[nPingIndex] = NULL;
  }

  nPing      = MAX_LATENCY-1;
  nPingIndex = 0;

  for (i = 0; i < NUM_THREADS; i++)
  {
    g_ahSimEvent[i] = CreateEvent (NULL, TRUE, FALSE, NULL);
    _beginthread (Simulate, 10000, (void*) i);
  }

#if PLAYBACK
  FILE* pFile;
  bool  bTerm = false;

  pFile = fopen (REC_FILE_NAME, "rt");

  if (pFile != NULL)
  {
    //pPrevGame->Load (pFile);

    do
    {
      pGame->pPrevGame = pPrevGame;
      pGame->Load (pFile);
      //pGame->GetSpeed ();
      pGame->SetOwnShots      ();
      pGame->GetShotAvailable ();
      pGame->GetObjectList    ();

      pGame->GetBestShot (0);

      pPrevGame = pGame;

      if (pGame == &game1)
        pGame = &game2;
      else
        pGame = &game1;
    }
    while (!feof (pFile));

    fclose (pFile);
  }
#else

  keys.clear ();   // alle Tasten loslassen

#if LATENCY > 1
  // start communication
  SendPacket (keys);
#endif

  for (;;)
  {
    if (sim.nFrameCnt % 600 == 0)
    {
      // 10 seconds over
      printf ("\n%u seconds", sim.nFrameCnt / 60);
    }
    else if (sim.nFrameCnt % 60 == 0)
    {
      // 10 seconds over
      printf (".");
    }

    nPing++; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung
    keys.ping = nPing;
    anPing[nPingIndex] = nPing;
    anKeys[nPingIndex] = keys.keys;
    apShotPos[nPingIndex] = sim.pShotPos;

    if (++nPingIndex == MAX_LATENCY)
    {
      nPingIndex = 0;
    }

#if LATENCY > 1
    ReceivePacket (frame);
#else
#error "Not possible anymore"
    SendPacket    (keys);
    ReceivePacket (frame);
#endif

    // terminate simulation threads
    for (i = 0; i < NUM_THREADS; i++)
    {
      sim.abTermThread[i] = true;
    }

#if RECORD >= 2
    pPrevGame->Save (REC_FILE_NAME);
#endif

    // get current result of simulation and set keys accordingly
    sim.resultSem.Lock ();
    nTurn            = sim.nTurn;
    sim.nTurn        = INVALID;
    sim.nFramesShoot = INVALID;
    sim.resultSem.Unlock ();

    if (pGame->ship_present)
    {
      // sim.nFired reset in GetShotAvailable
      if (sim.nFired == FIRE_CNT)
      {
        keys.fire (true);
  //        printf ("Fire!\n");
      }

      if (pPrevGame->bShipWillBeHit)  // Flucht, wenn Kollision unausweichlich
      {
        Debug ("hyperspace\n");
        keys.hyperspace(true);
      }
      else if (nTurn != INVALID)
      {
        if (nTurn > 0)
        {
          keys.left(true);
          g_nShotAngleIndex++;
  //          printf ("turn left to angle index %i\n", (int) g_nShotAngleIndex);
        }
        else if (nTurn < 0)
        {
          keys.right(true);
          g_nShotAngleIndex--;
  //          printf ("turn right to angle index %i\n", (int) g_nShotAngleIndex);
        }
        else if (pPrevGame->MoveToTarget ())
        {
          // only one target, not turning to it, just waiting
          keys.thrust (true);
        }
      }

      if (pPrevGame->MoveToDefault ())
      {
        keys.thrust (true);
      }
    }
    else
    {
      pPrevGame->bShipWillBeHit = false;
      pGame->bShipWillBeHit = false;
    }

    SendPacket (keys);

    if (sim.nFrameCnt >= 0)
    {
      if (bEmpty && pGame->nAsteroids > 0)
      {
        bEmpty = FALSE;
        sim.nLevel++;
      }
      else if (!bEmpty && pGame->nAsteroids == 0)
      {
        printf ("\nLevel %i finished after %u seconds\n", sim.nLevel, sim.nFrameCnt / 60);
        bEmpty = TRUE;
      }
    }

    if (frame.frameno != ++prevframe)
    {
      if (sim.nFrameCnt >= 0)
      {
        if (sim.nFrameCnt >= MAX_FRAME_CNT)
        {
          // reinitialization
          sim.nFrameCnt = -1;
          printf ("Game over, restart necessary\n");
          exit (0);
        }
        else
        {
          printf("%d Frames verloren\n", frame.frameno - prevframe);
        }
      }

      prevframe = frame.frameno;           
    }

    if (frame.ping != (BYTE) (nPing-LATENCY+1))
    {
      //printf("Latenz %d, %d\n", (int) keys.ping, (int) frame.ping);

      prevframe = frame.frameno;           

      if (pGame->ship_present)
      {
        Debug ("Frame %i Latency %d, %d\n", sim.nFrameCnt, 
          (int) nPing, (int) frame.ping);

        /* 
        try to correct the angle index
        if not possible, it will be corrected using ship_dx,ship_dy
        therefore we assume that one keys packet has been evaluated twice
        and the next packet has been ignored
        check if correction necessary
        */
        i = nPingIndex;

        do
        {
          if (anPing[i] == frame.ping)
          {
            // this packet was last received by mame
            nDoubledKeys = anKeys[i];
            Debug ("Last received keys: %02X\n", (int) nDoubledKeys);

            if ((j = i + 1) == MAX_LATENCY)
            {
              j = 0;
            }

            nIgnoredKeys = anKeys[j];
            Debug ("Sent keys ignored: %02X\n", (int) nIgnoredKeys);

            if (nDoubledKeys != nIgnoredKeys)
            {
              // undo changes by ignored keys and do changes by doubled keys
              // hyperspace will be repeated anyway, thrust is not used
              if ((nDoubledKeys & KEY_FIRE) != 0)
              {
                sim.nFired = FIRE_CNT;
              }
              else if ((nIgnoredKeys & KEY_FIRE) != 0)
              {
                // shot not executed
                if (apShotPos[j] != NULL)
                {
                  /*
                  Debug ("shot cnt of %i/%i radius %i set from %i to zero\n",
                    apShotPos[j]->x, apShotPos[j]->y, apShotPos[j]->nRadius,
                    apShotPos[j]->nShotCnt);
                    */

                  apShotPos[j]->nShotCnt = 0;
                }

                sim.nFired = 0;
              }

              if ((nDoubledKeys & KEY_RIGHT) != 0 && (nIgnoredKeys & KEY_RIGHT) == 0 // undesired turn right
                || (nDoubledKeys & KEY_LEFT) == 0 && (nIgnoredKeys & KEY_LEFT) != 0) // no turn left
              {
                g_nShotAngleIndex--;

                for (j = 0; j < LATENCY; j++)
                {
                  g_anPrevAngles[j]--;
                }

                Debug ("Angle index decreased: %i\n", g_nShotAngleIndex);
              }

              if ((nDoubledKeys & KEY_RIGHT) == 0 && (nIgnoredKeys & KEY_RIGHT) != 0 // no turn right
                || (nDoubledKeys & KEY_LEFT) != 0 && (nIgnoredKeys & KEY_LEFT) == 0) // undesired turn left
              {
                g_nShotAngleIndex++;

                for (j = 0; j < LATENCY; j++)
                {
                  g_anPrevAngles[j]++;
                }

                Debug ("Angle index increased: %i\n", g_nShotAngleIndex);
              }
            }
            // else identical, no correction needed

            break;
          }

          if (++i == MAX_LATENCY)
          {
            i = 0;
          }
        }
        while (i != nPingIndex);
      }
    }

    g_pSimGame       = pGame;
    pGame->pPrevGame = pPrevGame;
    pGame->InterpretScreen (frame);
    pGame->GetSpeed ();

    keys.clear();   // alle Tasten loslassen

    if (sim.nFrameCnt >= 0)
      sim.nFrameCnt++;         // Zeit

    pGame->bShipWillBeHit = pPrevGame->bShipWillBeHit;

    if (pGame->ship_present && !pGame->bShipWillBeHit)
    {
      if (sim.nFrameCnt < 0)
        sim.nFrameCnt = 0;

      pGame->CorrectAngle     ();
      pGame->SetOwnShots      ();
      pGame->GetShotAvailable ();
      pGame->GetObjectList    ();

      // start threads
      for (i = 0; i < NUM_THREADS; i++)
      {
        SetEvent (g_ahSimEvent[i]);
      }

      if (!pGame->saucer_present && pGame->nAsteroids == 0)
      {
        // there is no target, turn to border because new level targets will appear there
        if (pGame->TurnToDefault (nDefaultOffsetX, dy)) 
        {
          dx = DEFAULT_X + nDefaultOffsetX - pGame->ship.x;
        }
        else
        {
          dx = nDefaultOffsetX == 0 ? 1 : -1;
          dy = 0;
        }

        nDiffAngle = pGame->AngleDifference (dx, dy);

        if (abs (nDiffAngle) > 2)
        {
          // turn to default position to move there
          sim.nTurn = nDiffAngle;
        }
      }
    }

    g_anPrevFrames[g_nLatencyPos  ] = sim.nFrameCnt;
    g_anPrevAngles[g_nLatencyPos++] = g_nShotAngleIndex;

    if (g_nLatencyPos == LATENCY)
      g_nLatencyPos = 0;

    pPrevGame = pGame;

    if (pGame == &game1)
      pGame = &game2;
    else
      pGame = &game1;
  }
#endif  // !PLAYBACK
}


//-----------------------------------------------------------------------------
static void GetShotTable ()
{
  int    nIndex, i, j, k, nCnt, nNewX, nNewY, nOldX, nOldY;
  double nShotAngle, x, y, nSpeedX, nSpeedY;
  AngleAndFrames** ppShotArray;
  AngleAndFrames*  pShots;
  AngleAndFrames*  pNewShots;
#if STAT
  int nHits = 0;
#endif

#if STORE_TABLE
  FILE* pFile;

  pFile = fopen (TABLE_FILE_NAME, "wt");
#endif
  /* 
  for every possible shoot angle, calculate the positions of a shot in that 
  direction from 1 to SHOT_RANGE frames. Store the shot with amount of frames 
  in a list of shots in a two dimensional array with indices x and y.
  The list is implemented as array of AngleAndFrames with 0xFF in nFrames as 
  termination mark. 
  Not all positions in the array get a shot, therefore the array contains 
  pointers to arrays of AngleAndFrames. The initial size of these arrays 
  is 4. It is increased by factor 2 if needed. 
  */

  for (nIndex = 0; nIndex < 256; nIndex++)
  {
#if STORE_TABLE
    fprintf (pFile, "Angle %.1f index %i:\n", g_anShotAngles[nIndex], nIndex);
#endif
    // store shots at the current position
    nShotAngle = g_anShotAngles[nIndex] / 360 * (2 * PI);
    nSpeedX    = cos (nShotAngle) * SHOT_SPEED;
    nSpeedY    = sin (nShotAngle) * SHOT_SPEED;
    x          = MIDDLE_X + INIT_SHOT_POS * nSpeedX;
    y          = MIDDLE_Y + INIT_SHOT_POS * nSpeedY;

    nOldX = MIDDLE_X;
    nOldY = MIDDLE_Y;

    for (i = LATENCY; i < SHOT_RANGE; i++)  // i may start at earlier index
    {
      for (nCnt = 0; nCnt < 10; nCnt++)
      {
        nNewX = GetIndex (round (x), SIZE_X);
        nNewY = GetIndex (round (y), SIZE_Y);

        if (nNewX != nOldX || nNewY != nOldY)
        {
          ppShotArray = &g_aapShotTable[nNewX][nNewY];
          pShots      = *ppShotArray;

          if (pShots == NULL)
          {
            // no array yet
            pShots = new AngleAndFrames[4];
            *ppShotArray = pShots;
            j = 0;
          }
          else
          {
            // search end of array
            for (j = 1; pShots[j].nFrames != 0xFF; j++)
            {
            }

            // check if array size must be increased
            if (j >= 3)
            {
              k = j+1;

              while ((k & 1) == 0)
                k >>= 1;

              if (k == 1)
              {
                // allocate new buffer and copy data
                pNewShots = new AngleAndFrames[(j+1) * 2];
                memcpy (pNewShots, pShots, j * sizeof (AngleAndFrames));
                delete pShots;
                pShots       = pNewShots;
                *ppShotArray = pShots;
              }
            }
          }

          pShots[j  ].nAngle  = nIndex;
          pShots[j  ].nFrames = i;
          pShots[j+1].nFrames = 0xFF;

#if STAT
          nHits++;
#endif
#if STORE_TABLE
          fprintf (pFile, "Hits %i/%i after %i frames\n", nNewX, nNewY, i);
#endif
        }

        nOldX = nNewX;
        nOldY = nNewY;

        x += nSpeedX / 10;
        y += nSpeedY / 10;
      }
    }
  }
#if STAT
  printf ("Stored %i hit points\n", nHits);
#endif

#if STORE_TABLE
  fprintf (pFile, "\nSorted by y than x:\n");

  for (nNewY = 0; nNewY < SIZE_Y; nNewY++)
  {
    for (nNewX = 0; nNewX < SIZE_X; nNewX++)
    {
      pShots = g_aapShotTable[nNewX][nNewY];

      if (pShots != NULL)
      {
        fprintf (pFile, "%i/%i:", nNewX, nNewY);

        for (nIndex = 0; pShots[nIndex].nFrames != 0xFF; nIndex++)
        {
          fprintf (pFile, " %i/%i", (int) pShots[nIndex].nAngle, (int) pShots[nIndex].nFrames);
        }

        fprintf (pFile, "\n");
      }
    }
  }

  fclose (pFile);
#endif
}


/*-------------------------------------------------------------------------*/
static void GetCircle (int* pnOffsets, int nRadius)
{
  int  x, y, nMove, nCnt, nAdd;
  bool bDirX;
        
#if STORE_CIRCLE
  FILE* pFile;

  pFile = fopen (CIRCLE_FILE_NAME, "at");

  fprintf (pFile, "Circle for radius %i\n", nRadius);
#endif

  // start at offset 0,0
  x = y = 0;
  nMove = 1;
  nCnt  = 0;
  nAdd  = 1;
  bDirX = TRUE;

  do
  {
    // store x,y if in radius distance
    if (x * x + y * y < nRadius * nRadius)
    {
      *pnOffsets++ = x;
      *pnOffsets++ = y;

#if STORE_CIRCLE
      fprintf (pFile, "%i/%i\n", x, y);
#endif
    }

    // modify x and y so that they move in a growing circle
    if (bDirX)
    {
      x += nAdd;
    }
    else
    {
      y += nAdd;
    }

    nCnt++;

    if (nCnt == nMove)
    {
      nCnt  = 0;
      bDirX = !bDirX;

      if (bDirX)
      {
        nMove++;
        nAdd = -nAdd;
      }
    }
  }   
  while (nMove <= nRadius);

  *pnOffsets = INVALID;

#if STORE_CIRCLE
  fclose (pFile);
#endif
}


/*-------------------------------------------------------------------------*/
int _tmain(int argc, TCHAR* argv[], TCHAR* envp[])
{
  Player player;
  int    nRetCode = 0;

  // MFC initialisieren, Ausgabe und Fehlermeldung bei Fehlern
  if (!AfxWinInit(::GetModuleHandle(NULL), NULL, ::GetCommandLine(), 0))
  {
    // ZU ERLEDIGEN: Fehlercode gem Ihren Anforderungen ndern
    cerr << _T("Fatal Error: MFC initialization failed") << endl;
    nRetCode = 1;
  }
  else if (argc != 2)
  {
    printf ("IP address needed as parameter\n");
  }
  else
  {
    strcpy (g_sIpAddress, argv[1]);

    AfxSocketInit (NULL);
  
    if (CreateSocket ())
    {
      GetShotTable ();
      GetCircle (g_anCircleSml, SML_AST_RADIUS);
      GetCircle (g_anCircleMed, MED_AST_RADIUS);
      GetCircle (g_anCircleBig, BIG_AST_RADIUS);

      player.Run ();
      ReleaseSocket ();
      nRetCode = 0;
    }
    else
    {
      printf ("CreateSocket failed\n");
      nRetCode = 2;
    }
  }

  return nRetCode;
}


