package core.asteroid;

import gui.AsteroidScreen;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.zip.CRC32;

import util.Pair;
import util.ShapeDirection;
import util.Statistics;
import util.Vector2D;
import core.KeyControl;
import core.opCodes.OpLocation;
import core.opCodes.OpVector;

public class Ship extends AsteroidObject {
	
	public static final int RANGE_ALL = 128;
	public static final int RANGE_360 = 43;
	
	public static final int ANGLE_COUNT = 256;
	
	
	public static Pair<Integer, Vector2D>[] shapeDirs;
	public static Hashtable<Integer, ShapeDirection> dmap;
	
	protected static CRC32 crc = new CRC32();
	
	
	static {
		try {
			ObjectInputStream in = new ObjectInputStream(new FileInputStream("shipAngles.dat"));
			shapeDirs = (Pair<Integer, Vector2D>[])in.readObject();
			in.close();
			
			/*float lastAng = 90;
			for(int i=0; i<256; i++) {
				float d = shapeDirs[i].b.getDeg() - lastAng;
				lastAng = shapeDirs[i].b.getDeg();
				System.out.printf("0x%08X %s delta=%.3f\n", shapeDirs[i].a, shapeDirs[i].b, d);
			}*/
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	public static void saveDirectionTable() {
		try {
			ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream("shipAngles.dat"));
			out.writeObject(Ship.shapeDirs);
			out.close();
		} catch(IOException e1) {
			e1.printStackTrace();
		}
	}
	
	
	/*
	 * aktuelle Schussrichtung - static, weil das neue Schiff nach
	 * dem Teleportieren/Sterben in die gleiche Richtung zeigt wie
	 * das alte...
	 */  
	private static int directionIndex = -1;
	private static int shapeBuf[] = new int[4];
	private static int shapeBufIdx = 0;
	

	public Ship(OpLocation location, ArrayList<OpVector> vectors) {
		super(location, vectors);
	}

	
	public boolean initDirection(KeyControl kc) {
		boolean init = false;
		
		if(shapeBufIdx == 0) {
			kc.pressKey(KeyControl.KEY_TURN_RIGHT, 0, 4);
		}
		
		int hash = calcShapeHash(vecs);
		if(hash == shapeBuf[shapeBuf.length-1]) {
			// Gengend Daten vorhanden um eindeutig die Richtung zu bestimmen
			for(int i=0; i<shapeDirs.length-shapeBuf.length; i++) {
				boolean match = true;
				for(int j=0; j<shapeBuf.length; j++) {
					if(shapeDirs[i+j].a != shapeBuf[j]) {
						match = false;
						break;
					}
				}
				if(match) {
					init = true;
					directionIndex = i+shapeBuf.length-1;
					//System.out.println("Richtung bestimmt: " + shapeDirs[directionIndex].b);
				}
			}
			
			// Temp Daten zur Richtungsbestimmung resetten
			shapeBufIdx = 0;
			for(int i=0; i<shapeBuf.length; i++) {
				shapeBuf[i] = 0;
			}
		} else {
			shapeBuf[shapeBufIdx++] = hash;
			if(shapeBufIdx >= shapeBuf.length) {
				for(int i=0; i<shapeBuf.length-1; i++) {
					shapeBuf[i] = shapeBuf[i+1];
				}
				shapeBufIdx = shapeBuf.length-1;
			}
		}
		
		return init;
	}
	
	
	public void checkDirection(KeyControl kc, int latency) {
		// aktuelle Richtung
		int hashS = calcShapeHash(vecs);
		
		// berechnete Richtung
		int ticks = kc.getPrevTurnTicks(latency);
		int idx = directionIndex - ticks;
		if(idx < 0) idx += 256;
		if(idx >= 256) idx -= 256;
		
		int hashI = shapeDirs[idx].a;
		if(hashI != hashS) {
			//int d = 0;
			for(int i=1; i<=128; i++) {
				//d = i;
				int idx1 = idx-i;
				int idx2 = idx+i;
				if(idx1 < 0) idx1 += 256;
				if(idx2 >= 256) idx2 -= 256;
				if(shapeDirs[idx1].a == hashS) {
					directionIndex = idx1 + ticks;
					break;
				}
				if(shapeDirs[idx2].a == hashS) {
					directionIndex = idx2 + ticks;
					break;
				}
			}
			if(directionIndex < 0) directionIndex += 256;
			if(directionIndex >= 256) directionIndex -= 256;
			
			int n = Math.round(Statistics.getFloat(Statistics.KEY_SYNC_LOST_CNT)) + 1;
			Statistics.putFloat(Statistics.KEY_SYNC_LOST_CNT, n);
		}
	}
	
	
	@Override
	public boolean isMatching(AsteroidObject obj, int frameDif) {
		// Es gibt immer nur ein Schiff...
		if(obj instanceof Ship) {
			return true;
		} else {
			return false;
		}
	}
	
	
	public void drawObject(Graphics2D g, AsteroidScreen screen, boolean drawInfo) {
		super.drawObject(g, screen, drawInfo);
		
		if(drawInfo) {
			Vector2D h = getCurrentShotDirection();
			if(h != null) {
				h.drawVector(g, loc.x, loc.y, 30);
				//g.drawString(String.format("phi=%.0f", h.getDeg()), loc.x+10, loc.y+30);
				//g.drawOval(loc.x-100, loc.y-100, 200, 200);
			}
			Vector2D v = getVelocity();
			AffineTransform t = g.getTransform();
			t.scale(1, -1);
			g.setTransform(t);
			g.drawString(String.format("v=(%.1f|%.1f) |v|=%.1f", v.x, v.y, v.getLength()), loc.x+20, -loc.y);
			t.scale(1, -1);
			g.setTransform(t);
		}
	}

	
	public String getObjectName() {
		return "Schiff";
	}


	public Color getDefaultColor() {
		return Color.YELLOW;
	}
	
	
	public boolean isDestroyable() {
		return true;
	}
	
	
	public boolean hasCollision() {
		return true;
	}
	
	
	protected static int calcShapeHash(ArrayList<OpVector> vecs) {
		byte[] b = new byte[4];
		
		crc.reset();
		for(int i=0; i<5 && i<vecs.size(); i++) {
			OpVector v = vecs.get(i);
			b[0] = (byte)((v.x & 0xFF000000)>>24);
			b[1] = (byte)((v.x & 0x00FF0000)>>16);
			b[2] = (byte)((v.x & 0x0000FF00)>>8);
			b[3] = (byte)((v.x & 0x000000FF)>>0);
			crc.update(b);
			b[0] = (byte)((v.y & 0xFF000000)>>24);
			b[1] = (byte)((v.y & 0x00FF0000)>>16);
			b[2] = (byte)((v.y & 0x0000FF00)>>8);
			b[3] = (byte)((v.y & 0x000000FF)>>0);
			crc.update(b);
			b[0] = (byte)((v.s & 0xFF000000)>>24);
			b[1] = (byte)((v.s & 0x00FF0000)>>16);
			b[2] = (byte)((v.s & 0x0000FF00)>>8);
			b[3] = (byte)((v.s & 0x000000FF)>>0);
			crc.update(b);
			b[0] = (byte)((v.z & 0xFF000000)>>24);
			b[1] = (byte)((v.z & 0x00FF0000)>>16);
			b[2] = (byte)((v.z & 0x0000FF00)>>8);
			b[3] = (byte)((v.z & 0x000000FF)>>0);
			crc.update(b);
		}
		return (int)crc.getValue();
	}
	
	
	public static Vector2D getCurrentShotDirection() {
		if(directionIndex == -1) {
			// aktuelle Richtung unbekannt...
			return null;
		} else {
			int idx = directionIndex;
			if(idx < 0) idx = 0;
			if(idx >= 256) idx -= 256;
			return shapeDirs[directionIndex].b;
		}
	}
	
	
	public static void tickLeft() {
		if(directionIndex != -1) {
			directionIndex--;
			if(directionIndex < 0) directionIndex += 256;
		}
	}
	
	
	public static void tickRight() {
		if(directionIndex != -1) {
			directionIndex++;
			if(directionIndex >= 256) directionIndex -= 256;
		}
	}
	
	
	public static int getNearestAngleIndex(Vector2D angle, int range) {
		float dif = 1000;
		int bestIdx = 0;
		
		for(int i=0; i<=range; i++) {
			int idx1 = directionIndex - i;
			int idx2 = directionIndex + i;
			if(idx1 < 0) idx1 += 256;	if(idx1 >= 256) idx1 -= 256;
			if(idx2 < 0) idx2 += 256;	if(idx2 >= 256) idx2 -= 256;

			float d = Math.abs(shapeDirs[idx1].b.calcDifAngle(angle));
			if(d < dif) {
				dif = d;
				bestIdx = idx1;
			}
			d = Math.abs(shapeDirs[idx2].b.calcDifAngle(angle));
			if(d < dif) {
				dif = d;
				bestIdx = idx2;
			}
		}
		return bestIdx;
	}
	
	
	public static Vector2D getNearestAngle(Vector2D angle, int range) {
		return shapeDirs[getNearestAngleIndex(angle, range)].b;
	}
	
	
	public static Vector2D getAngleByIndex(int index) {
		return shapeDirs[index].b;
	}
	
	
	public static int getTurnTicks(Vector2D targetAngle) {
		int idx = directionIndex;
		if(idx < 0) idx = 0;
		if(idx >= 256) idx -= 256;
		
		for(int i=0; i<256; i++) {
			if(targetAngle == shapeDirs[idx].b) {
				break;
			}
			idx++;
			if(idx >= shapeDirs.length) idx = 0;
		}
		
		int n = directionIndex - idx;
		if(n < -128) n += shapeDirs.length;
		if(n >= 128) n -= shapeDirs.length;
		return n;
	}
}
