package gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.AffineTransform;
import java.util.Vector;

import javax.swing.JPanel;
import javax.swing.Timer;

import util.IntAvgBuffer;
import util.MameFrameListener;
import util.Statistics;
import core.GameInfo;
import core.asteroid.AsteroidObject;
import core.asteroid.Decoration;
import core.asteroid.Shot;
import core.opCodes.OpLocation;

public class AsteroidScreen extends JPanel implements Runnable, ActionListener, MameFrameListener {

	public static final Color DARK_GREEN = new Color(0, 128, 0);
	
	private static final long serialVersionUID = 1L;
	
	
	private boolean renderInProgress = false;
	private boolean stopRequested = false;
	
	private GameInfo objects;
	
	private int lostFrames;
	private int frameCnt;
	private IntAvgBuffer fps;
	private boolean drawInfo = false;

	
	public AsteroidScreen() {
		lostFrames = 0;
		frameCnt = 0;
		fps = new IntAvgBuffer(50);

		setPreferredSize(new Dimension(640, 480));
		
		Thread paintThread = new Thread(this);
		paintThread.setPriority(Thread.MIN_PRIORITY);
		paintThread.start();
		
		Timer t = new Timer(100, this);
		t.start();
	}
	
	
	public void paint(Graphics graphics) {
		Graphics2D g = (Graphics2D)graphics;
		//g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		int inset = 8;

		// Background
		g.setColor(Color.BLACK);
		g.fillRect(0, 0, getWidth(), getHeight());
		g.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 20));
		
		// Border
		g.setColor(Color.GRAY);
		g.drawRect(0, 0, getWidth()-1, getHeight()-1);
		
		AffineTransform tf = new AffineTransform();
		tf.scale(getWidth()/(1024.0+inset*2), -getHeight()/(768.0+inset*2));
		tf.translate(inset, -768-inset);
		g.setTransform(tf);
		
		if(objects != null) {
			// Objekte
			Vector<AsteroidObject> objs = objects.getAllObjects();
			for(int i=0; i<objs.size(); i++) {
				AsteroidObject o = objs.get(i);
				boolean d = o instanceof Decoration ? false : drawInfo;
				o.drawObject(g, this, d);
				
				if(drawInfo && o.getHitTime() > 0 && !(o instanceof Shot)) {
					if(o.isHitPreliminary()) {
						g.setColor(Color.YELLOW);
					} else {
						g.setColor(Color.GREEN);
					}
					OpLocation loc = o.getLocation();
					int r = o.getHitRadius()+5;
					g.drawRect(loc.x-r, loc.y-r, r*2, r*2);
				}
			}
		}
	}
	
	
	public boolean isDrawInfo() {
		return drawInfo;
	}


	public void setDrawInfo(boolean drawInfo) {
		this.drawInfo = drawInfo;
	}


	private synchronized void waitForFrameData() {
		renderInProgress = false;
		try {
			wait();
		} catch(InterruptedException e) {
			e.printStackTrace();
		}
		renderInProgress = true;
	}
	
	
	public void run() {
		while(!stopRequested) {
			waitForFrameData();
			if(objects != null && objects.getGameTime() % 2 == 0 || fps.getAverage() < 50) {
				repaint();
			}
		}
	}
	
	
	public synchronized void requestStop() {
		stopRequested = true;
		notify();
	}


	public void mameFrameReceived(GameInfo objects) {
		this.lostFrames += objects.getLostPacketCount();
		frameCnt++;
		if(!renderInProgress) {
			this.objects = objects;
			// Ausgabe zeichnen
			synchronized(this) {
				notify();
			}
		}
	}


	public void actionPerformed(ActionEvent e) {
		fps.addElement(frameCnt*10);
		Statistics.putFloat(Statistics.KEY_FPS, fps.getAverage());
		frameCnt = 0;
	}
}
