package core;

import gui.AsteroidBotWnd;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;
import java.util.zip.GZIPInputStream;

import util.MameFrameListener;
import util.Statistics;

public class MameClient extends Thread {
	
	private static final int MAME_PORT = 1979;
	
	
	private AsteroidBotWnd wnd;
	
	private DatagramSocket socket;
	private InetAddress mameAddress;
	
	private KeyControl keyCtrl;

	private byte pingByte = 0;
	private int keyMask = 0;
	private long time = 0;
	
	private InputStream dumpStream;
	private Timer dumpTimer;
	
	private boolean stopRequested = false;
	
	private ArrayList<MameFrameListener> listeners;
	private ScreenInterpreter interpreter;
	private GameInfo objects;
	
	private int prevAsteroidCnt = 0;
	
	
	public MameClient(AsteroidBotWnd wnd) {
		this.wnd = wnd;
		
		listeners = new ArrayList<MameFrameListener>();
		objects = new GameInfo();
		
		// Vector ROM einlesen
		int[] vecRom = MemoryParser.loadVectorRom();
		interpreter = new ScreenInterpreter(vecRom);
	}
	
	
	public void connectToServer(String host, String playerName) {
		// Key Control Objekt erzeuegen
		keyCtrl = new KeyControl(this);
		
		try {
			mameAddress = InetAddress.getByName(host);
			socket = new DatagramSocket();
			sendKeys();
			// Bei Online Spielen Namen senden
			//if(host.equals("asteroids.heise.de"))
			sendName(playerName);
		} catch(Exception e) {
			e.printStackTrace();
		}
		
		// Thread starten
		start();
	}


	public void run() {
		byte[] buf = new byte[1026];
		DatagramPacket packet = new DatagramPacket(buf, buf.length);
		int prevIdxByte = 0;
		//long nanoTime = System.nanoTime();
		
		// Frames empfangen
		while(!stopRequested) {
			try {
				// Frame Daten empfangen
				/*double time = (System.nanoTime() - nanoTime) / 1e6;
				if(time > 15) {
					System.out.println("loop time: " + time + "ms");
				}*/
				socket.receive(packet);
				//nanoTime = System.nanoTime();

				// Knpfe senden
				sendKeys();
				keyCtrl.setKeys(1);
				
				// Empfangenes Paket verarbeiten
				String s = new String(buf, 0, 4);
				if(s.equals("busy")) {
					printBusyMsg(buf, packet.getLength());
					break;
				} else if(s.equals("game")) {
					//System.out.println("game over");
					wnd.gameOver();
					break;
				}
				
				// Latenzzeit ermitteln
				int ping = byteToInt(buf[1025]);
				int latency = pingByte - ping;
				if(latency < 0) latency += 256;
				Statistics.putFloat(Statistics.KEY_LATENCY, latency);
				
				int idxByte = byteToInt(buf[1024]);
				// prevIdxByte beim ersten Paket initialisieren
				if(time == 0) prevIdxByte = idxByte - 1;
				// Anzahl vergangener Frames (wenn > 1 gabs Paketverluste)
				int dif = idxByte - prevIdxByte;
				if(dif == -255) dif = 1;

				if(dif > 0) {
					// Paket ist gltig (dif < 0 bei falscher Paket Reihenfolge)
					prevIdxByte = idxByte;
					time += dif;
					int[] memData = MemoryParser.parseMemory(buf, 0, 1024);
					objects.newFrame(interpreter.interpretScreen(memData), interpreter.getGameMode(), dif, latency);
					setWave();
					callListeners();
					
					if(dif > 1) {
						float lost = Statistics.getFloat(Statistics.KEY_LOST_PACKETS);
						Statistics.putFloat(Statistics.KEY_LOST_PACKETS, lost + dif - 1);
					}
				}
			} catch(IOException e) {
				e.printStackTrace();
			}
		}
		socket.close();
	}
	
	
	public void readDumpFile(String dumpFile) {
		try {
			if(dumpFile.endsWith(".gz")) {
				dumpStream = new GZIPInputStream(new FileInputStream(dumpFile));
			} else {
				dumpStream = new FileInputStream(dumpFile);
			}
			
			byte[] header_v1 = {
					0x63, 0x74, 0x6D, 0x61, 0x6D, 0x65, 0x31, 0x0D, 0x0A, 0x02, 0x00
			};
			byte[] header_v2 = {
					0x63, 0x74, 0x6D, 0x61, 0x6D, 0x65, 0x32, 0x0D, 0x0A, 0x02, 0x00
			};
			byte[] header = new byte[11];
			dumpStream.read(header);
			
			for(int i=0; i<header.length; i++) {
				if(header[i] != header_v1[i] && header[i] != header_v2[i]) {
					// ungltige Datei
					return;
				}
			}
			
			if(header[6] == header_v1[6]) {
				// Version 1 Format
				dumpStream.skip(14);
			} else if(header[6] == header_v2[6]) {
				// Version 2 Format
				dumpStream.skip(46);
			}
			
			dumpTimer = new Timer();
			TimerTask task = new TimerTask() {
				public void run() {
					try {
						byte[] buf = new byte[1025];
						int idx = 0;
						boolean end = false;
						while(idx < 1025) {
							int len = dumpStream.read(buf, idx, 1025-idx);
							if(len == -1) {
								end = true;
								break;
							}
							idx += len;
						}
						if(end) {
							dumpStream.close();
							dumpTimer.cancel();
							return;
						}
						int[] memData = MemoryParser.parseMemory(buf, 0, 1024);
						objects.newFrame(interpreter.interpretScreen(memData), interpreter.getGameMode(), 1, 1);
						setWave();
						callListeners();
					} catch (IOException e) {
						e.printStackTrace();
						dumpTimer.cancel();
					}
				}
			};
			dumpTimer.scheduleAtFixedRate(task, 0, 17);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	private void printBusyMsg(byte[] buf, int len) {
		String s = new String(buf, 0, len-2);
		String num = s.substring(5);
		int secs = Integer.parseInt(num)/60;
		wnd.setServerBusyTime(secs);
	}


	private void setWave() {
		float wave = Statistics.getFloat(Statistics.KEY_WAVE);
		int asteroidCnt = objects.getAsteroidCount();
		if(getGameMode() == ScreenInterpreter.GAME_MODE_GAME && prevAsteroidCnt == 0 && asteroidCnt > 0)
			wave++;
		if(getGameMode() == ScreenInterpreter.GAME_MODE_GAME)
			prevAsteroidCnt = asteroidCnt;
		Statistics.putFloat(Statistics.KEY_WAVE, wave);
	}
	
	
	private void callListeners() {
		for(int i=0; i<listeners.size(); i++) {
			listeners.get(i).mameFrameReceived(objects);
		}
	}


	private void sendKeys() {
		byte[] buf = new byte[8];
		
		buf[0] = 'c';
		buf[1] = 't';
		buf[2] = 'm';
		buf[3] = 'a';
		buf[4] = 'm';
		buf[5] = 'e';
		buf[6] = (byte)keyMask;
		buf[7] = pingByte++;
		
		DatagramPacket packet = new DatagramPacket(buf, 8, mameAddress, MAME_PORT);
		try {
			socket.send(packet);
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	private void sendName(String playerName) {
		byte[] buf = new byte[38];
		for(int i=0; i<buf.length; i++) {
			buf[i] = 0;
		}
		
		String s = "ctname" + playerName;
		try {
			byte[] b = s.getBytes("UTF-8");
			for(int i=0; i<buf.length && i<b.length; i++) {
				buf[i] = b[i];
			}
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		
		DatagramPacket packet = new DatagramPacket(buf, 38, mameAddress, MAME_PORT);
		try {
			socket.send(packet);
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	public void setKeyMask(int keyMask) {
		this.keyMask = keyMask;
	}
	
	
	public KeyControl getKeyControl() {
		return keyCtrl;
	}
	
	
	public void startGame() {
		keyCtrl.pressKey(KeyControl.KEY_START, 0, 3);
	}
	
	
	public void dumpScreenMem() {
		interpreter.dumpScreenMem();
	}
	
	
	public int getGameMode() {
		return interpreter.getGameMode();
	}
	
	
	public long getTime() {
		return time;
	}
	
	
	public void requestStop() {
		stopRequested = true;
	}
	
	
	public boolean addAsteroidFrameListener(MameFrameListener listener) {
		return listeners.add(listener);
	}


	public boolean removeAsteroidFrameListener(MameFrameListener listener) {
		return listeners.remove(listener);
	}
	
	
	private int byteToInt(byte b) {
		return b<0 ? (int)b+256 : b;
	}
}
