package core;

import java.io.FileInputStream;
import java.io.IOException;

import core.opCodes.OpLocation;
import core.opCodes.OpVector;

public class MemoryParser {
	
	
	/**
	 * Extrahiert die X-, Y-Vektoren, Skalierung und Helligkeit aus dem VCTR Befehl.
	 * 
	 * @param w1 erstes Befehlswort
	 * @param w2 zweites Befehlswort
	 */
	public static OpVector parseVCTR(int w1, int w2) {
		OpVector v = new OpVector();

		v.y = w1 & 0x03FF;
		v.x = w2 & 0x03FF;
		v.s = w1 >> 12;
		v.z = w2 >> 12;
		
		int sy = (w1 & 0x0400) >> 10;
		int sx = (w2 & 0x0400) >> 10;
		if(sy != 0) v.y = -v.y;
		if(sx != 0) v.x = -v.x;
		
		return v;
	}
	
	
	/**
	 * Extrahiert die X-, Y-Vektoren, Skalierung und Helligkeit aus dem SVEC Befehl.
	 * 
	 * @param w1 erstes Befehlswort
	 * @param w2 zweites Befehlswort
	 */
	public static OpVector parseSVEC(int w1) {
		OpVector v = new OpVector();

		v.y = w1 & 0x0300;
		v.x = (w1 & 0x0003) << 8;
		v.z = (w1 & 0x00F0) >> 4;
		v.s = ((w1 & 0x0800) >> 11) | ((w1 & 0x0008) >> 2) + 2;
		
		int sy = (w1 & 0x0400) >> 10;
		int sx = (w1 & 0x0004) >> 2;
		if(sy != 0) v.y = -v.y;
		if(sx != 0) v.x = -v.x;
		
		return v;
	}
	

	/**
	 * Extrahiert die X-, Y-Vektoren und den GSF aus dem LABS Befehl.
	 * 
	 * @param w1 Befehlswort
	 * @param w2 Befehlswort
	 */
	public static OpLocation parseLABS(int w1, int w2) {
		OpLocation a = new OpLocation();
		
		a.y = w1 & 0x03FF;
		a.x = w2 & 0x03FF;
		a.gsf = w2 >> 12;
		
		return a;
	}
	
	
	/**
	 * Die Methode konvertiert den bergebenen little-endian Speicher Inhalt
	 * in 16-Bit Worte.
	 * 
	 * @param buf Der Speicherinhalt (in little-endian) 
	 * @param off Start Offset
	 * @param len Anzahl der zu konvertierenden Bytes
	 * @return Das int-Array mit den 16-Bit Worten
	 */
	public static int[] parseMemory(byte[] buf, int off, int len) {
		int[] mem = new int[len/2];

		// Little Endian Byte Buffer -> 16-Bit Worte
		for(int i=0; i<mem.length; i++) {
			mem[i] = byteToInt(buf[i*2+off]) | (byteToInt(buf[i*2+1+off]) << 8);
		}
		
		return mem;
	}
	
	
	/**
	 * Die Methode gibt die Befehle im bergebenen Speicherinhalt aus.
	 * 
	 * @param mem Der Speicherinhalt, der ausgegeben werden soll
	 * @param addrOff Adressen Offset (2048 fr Vector ROM)
	 */
	public static void dumpMemory(int[] mem, int addrOff) {
		for(int i=0; i<mem.length-1; i++) {
			System.out.printf("0x%03X | 0x%04X ", i+addrOff, mem[i]);
			
			int opCode = (mem[i] & 0xF000) >> 12;
		
			if(opCode <= 10)
				System.out.printf("0x%04X  ", mem[i+1]);
			else 
				System.out.print("        ");

			if(opCode < 10) {
				OpVector v = parseVCTR(mem[i], mem[i+1]);
				System.out.printf("VCTR    ; x=%5d, y=%5d, z=%2d, s=%2d\n", v.x, v.y, v.z, v.s);
				i++;
				
			} else if(opCode == 10) {
				OpLocation a = parseLABS(mem[i], mem[i+1]);
				System.out.printf("LABS    ; x=%5d, y=%5d, s=%2d\n", a.x, a.y, a.gsf);
				i++;
				
			} else if(opCode == 11) {
				System.out.printf("HALT\n");
				
			} else if(opCode == 12) {
				System.out.printf("JSRL    ; pc=0x%03X\n", mem[i] & 0x0FFF);
				
			} else if(opCode == 13) {
				System.out.printf("RTSL\n");
				
			} else if(opCode == 14) {
				System.out.printf("JMPL    ; pc=0x%03X\n", mem[i] & 0x0FFF);
				
			} else if(opCode == 15) {
				OpVector v = parseSVEC(mem[i]);
				System.out.printf("SVEC    ; x=%5d, y=%5d, z=%2d, s=%2d\n", v.x, v.y, v.z, v.s);
				
			}
		}
	}
	
	
	private static int byteToInt(byte b) {
		return b<0 ? (int)b+256 : b;
	}
	
	
	public static int[] loadVectorRom() {
		byte[] buf = new byte[2048];
		
		try {
			FileInputStream fin = new FileInputStream("vectorrom.bin");
			fin.read(buf);
			fin.close();
		} catch(IOException e) {
			e.printStackTrace();
		}
		
		return MemoryParser.parseMemory(buf, 0, 2048);
	}

}
