package core.asteroid;

import gui.AsteroidScreen;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.util.ArrayList;

import util.IntAvgBuffer;
import util.Vector2D;
import core.bot.TargetSolution;
import core.opCodes.OpLocation;
import core.opCodes.OpVector;


public abstract class AsteroidObject {
	
	public static final int MAX_RESOLUTION_AGE = 8;
	

	private static int objectId = 1;
	
	protected int id = objectId++;
	protected int romAddress = 0; 
	protected OpLocation loc;
	protected IntAvgBuffer vxb;
	protected IntAvgBuffer vyb;
	protected int collisionTime = -1;
	protected int nextColCalc = 0;
	protected int age = 0;
	protected int hitRadius = 0;
	protected int hitTime = 0;
	protected boolean hitMarker = false;
	
	protected TargetSolution solution;
	
	protected ArrayList<OpVector> vecs;
	protected Color color;
	
	
	public AsteroidObject(OpLocation location, ArrayList<OpVector> vectors) {
		loc = location;
		vxb = new IntAvgBuffer(MAX_RESOLUTION_AGE);
		vyb = new IntAvgBuffer(MAX_RESOLUTION_AGE);
		
		setVectors(vectors);
		calculateHitRadius();
		color = getDefaultColor();
	}
	
	
	protected void calculateHitRadius() {
		int r = 0;
		if(hasCollision()) {
			Vector2D v = new Vector2D();
			
			// Vektor Grafik nach den Eckpunkten durchsuchen
			for(int i=0; i<vecs.size(); i++) {
				Vector2D w = vecs.get(i).scale(loc.gsf);
				v.x += w.x;
				v.y += w.y;
				int l = Math.round(v.getLength());
				if(l > r) r = l;
			}
		}
		hitRadius = r;
	}
	
	
	protected void setVectors(ArrayList<OpVector> vectors) {
		vecs = vectors;
	}
	
	
	public synchronized void drawObject(Graphics2D g, AsteroidScreen screen, boolean drawInfo) {
		float x = loc.x;
		float y = loc.y;
		
		g.setColor(color);
		
		for(int i=0; i<vecs.size(); i++) {
			OpVector o = vecs.get(i);
			Vector2D v = o.scale(loc.gsf);
			
			if(o.z > 0) g.drawLine(Math.round(x), Math.round(y), Math.round(x+v.x), Math.round(y+v.y));
			
			x += v.x;
			y += v.y;
		}

		if(drawInfo) {
			float vx = vxb.getAverage();
			float vy = vyb.getAverage();
			
			g.setStroke(new BasicStroke(1));
			
			g.drawLine(loc.x, loc.y, Math.round(loc.x+vx*30), Math.round(loc.y+vy*30));
			//g.drawOval(loc.x-hitRadius, loc.y-hitRadius, hitRadius*2, hitRadius*2);
			
			/*AffineTransform t = g.getTransform();
			t.scale(1, -1);
			g.setTransform(t);
			g.drawString(String.format("%s [%d]", getObjectName(), id), loc.x+20, -loc.y);
			//g.drawString(String.format("v=(%.1f|%.1f) |v|=%.1f", vx, vy, Math.sqrt(vx*vx+vy*vy)), screen.tX(loc.x)+10, screen.tY(loc.y)+15);
			t.scale(1, -1);
			g.setTransform(t);*/
		}
	}
	
	
	public synchronized void updateLocation(AsteroidObject obj, int packetInc) {
		setVectors(obj.getVectors());
		
		OpLocation location = obj.getLocation();
		
		float vxa = vxb.getAverage();
		float vya = vyb.getAverage();
		int vx = Math.round((float)(location.x - loc.x) / (float)packetInc);
		int vy = Math.round((float)(location.y - loc.y) / (float)packetInc);

		// Position updaten
		loc.x = location.x;
		loc.y = location.y;

		// Seitenwechsel ignorieren
		if(Math.abs(vx-vxa) > 10 || Math.abs(vy-vya) > 10) {
			return;
		}
		// Geschwindigkeiten updaten
		vxb.addElement(vx);
		vyb.addElement(vy);
		
		age += packetInc;
		if(nextColCalc > 0) nextColCalc -= packetInc;
		if(nextColCalc < 0) nextColCalc = 0;
		if(hitTime > 0) hitTime -= packetInc;
		if(hitTime <= 0) {
			hitTime = 0;
			hitMarker = false;
		}
	}
	
	
	public synchronized boolean isMatching(AsteroidObject obj, int packetInc) {
		OpLocation location = obj.getLocation();
		ArrayList<OpVector> vectors = obj.getVectors();
		
		int vx = Math.round(vxb.getAverage() * packetInc);
		int vy = Math.round(vyb.getAverage() * packetInc);
		
		// GSF vergleichen
		if(location.gsf != loc.gsf) return false;
		
		// Position vergleichen
		int maxDif = (int)(19 - vxb.size()*2 / (float)(MAX_RESOLUTION_AGE/8.0f));
		int xDif = Math.abs(location.x - loc.x - vx);
		if(xDif > 1000) xDif = 1024 - xDif;
		int yDif = Math.abs(location.y - loc.y - vy);
		if(yDif > 755) yDif = 768 - yDif;
		
		if(xDif > maxDif || yDif > maxDif) return false;
		
		// Vektoren vergleichen
		if(romAddress != 0) {
			if(romAddress != obj.getRomAddress()) return false;
		} else {
			if(vecs.size() != vectors.size()) return false;
			for(int i=0; i<vecs.size(); i++) {
				if(!vecs.get(i).equals(vectors.get(i))) return false;
			}
		}
		
		// Objekt passt
		return true;
	}
	
	
	public synchronized void updateTargetSolution(Ship ship, int latency) {
		if(solution == null) {
			solution = TargetSolution.getAccurateSolution(ship, this, latency);
		} else {
			solution.update(ship, latency);
		}
	}
	
	
	public synchronized TargetSolution getTargetSolution() {
		return solution;
	}


	public synchronized OpLocation getLocation() {
		return loc;
	}
	
	
	public synchronized Vector2D getVelocity() {
		return new Vector2D(vxb.getAverage(), vyb.getAverage());
	}


	public synchronized int getCollisionTime() {
		return collisionTime;
	}


	public synchronized void setCollisionTime(int collisionTime) {
		this.collisionTime = collisionTime;
		nextColCalc = 10;
	}
	
	
	public synchronized int getNextColCalc() {
		return nextColCalc;
	}


	public synchronized int getAge() {
		return age;
	}
	
	
	public synchronized int getHitRadius() {
		return hitRadius;
	}


	public synchronized ArrayList<OpVector> getVectors() {
		return vecs;
	}


	public synchronized int getRomAddress() {
		return romAddress;
	}


	public synchronized void setRomAddress(int romAddress) {
		this.romAddress = romAddress;
	}
	
	
	public synchronized int getObjectId() {
		return id;
	}
	
	
	public synchronized float calculateDistance(AsteroidObject o) {
		int x = loc.x-o.getLocation().x;
		int y = loc.y-o.getLocation().y;
		if(x > 512) x = 1024 - x;
		if(x < -512) x = x + 1024;
		if(y > 384) y = 768 - y;
		if(y < -384) y = y + 768;
		Vector2D dist = new Vector2D(x, y);
		return dist.getLength();
	}
	
	
	public String toString() {
		return String.format("%s [0x%03X]", getObjectName(), romAddress);
	}


	public Color getColor() {
		return color;
	}


	public void setColor(Color color) {
		this.color = color;
	}
	
	
	public synchronized void setHitTime(int hitTime) {
		this.hitTime = hitTime;
	}
	
	
	public synchronized int getHitTime() {
		return hitTime;
	}
	
	
	public synchronized void setHitPreliminary(boolean marker) {
		hitMarker = marker;
		if(marker && hitTime < 10) {
			if(getCollisionTime() >= 0) {
				hitTime = 10;
			} else {
				hitTime = 50;
			}
		}
	}
	
	
	public synchronized boolean isHitPreliminary() {
		return hitMarker;
	}


	public abstract Color getDefaultColor();
	
	
	public abstract String getObjectName();
	
	
	public abstract boolean isDestroyable();
	
	
	public abstract boolean hasCollision();
	
}
