package de.heise.asteroid;

import java.net.UnknownHostException;
import java.util.LinkedList;
import java.util.List;

import de.heise.asteroid.comm.Communicator;
import de.heise.asteroid.comm.FramePacket;
import de.heise.asteroid.comm.KeyPacket;
import de.heise.asteroid.comm.ServerConnection;
import de.heise.asteroid.engine.Engine;
import de.heise.asteroid.engine.GameStatus;
import de.heise.asteroid.player.ModifiedSamplePlayer;
import de.heise.asteroid.player.Player;
import de.heise.asteroid.util.Worker;

/**
 * The <code>Main</code> is the main command-line application for the
 * asteroids player.
 * <p>
 * It takes one optional command-line parameter: <code>server</code> (defaults
 * to 127.0.0.1)
 */
public class Main {
   private static final boolean syncComm = true;
   private static final int timeout = 0;
   private static final int gamesToPlay = 1;
   private static String serverName;
   private static List<FramePacket> fpQueue;
   private static List<KeyPacket> kpQueue;
   
   private static void runPlayer() throws Exception {
      GameStatus gameStatus = new GameStatus();
      ServerConnection serverConnection = new ServerConnection();
      try {
         serverConnection.connect(serverName);
         Communicator communicator = new Communicator(serverConnection);
         communicator.setQueues(fpQueue, kpQueue);
         Engine engine = new Engine(communicator, gameStatus, gamesToPlay);
         engine.setQueues(fpQueue, kpQueue);
         Player player = new ModifiedSamplePlayer(engine);
         engine.registerPlayer(player);
         // Set up and start worker threads
         Worker engineWorker = new Worker("Engine", engine);
         engineWorker.start();
         Worker commWorker = syncComm? null : new Worker("Communicator", communicator);
         if (commWorker != null) {
//            commWorker.setPriority(Thread.MAX_PRIORITY);
            commWorker.start();
         }
         if (timeout > 0) {
            try {
               System.out.println("Waiting " + timeout + " seconds...");
               Thread.sleep(timeout * 1000);
               System.out.println("Time is over!");
            } catch (InterruptedException e) {
               System.out.println("InterruptedException: " + e.getMessage());
            }
            if (commWorker != null) {
               commWorker.terminate();
            }
            engineWorker.terminate();
         } else {
            try {
               while (engine.getGamesCompleted() < gamesToPlay) {
                  Thread.sleep(3000);
               }
            } catch (InterruptedException e) {
               System.out.println("InterruptedException: " + e.getMessage());
            }
         }
      } catch (Exception e) {
         System.err.println("Uncaught exception: " + e.getMessage());
         throw e;
      } finally {
         serverConnection.disconnect();
      }
   }
   
   /**
    * The main method.
    * 
    * @param args the command-line parameters
    * @throws Exception
    */
   public static void main(String[] args) throws Exception, UnknownHostException {
      // takes one parameter (defaults to 127.0.0.1)
      if (args.length > 1) {
         System.out.println("usage: " + Main.class.getCanonicalName() + " <server>");
         return;
      }
      if (args.length == 1) {
         serverName = args[0];
      } else {
         serverName = "127.0.0.1"; // default server
      }
      fpQueue = syncComm ? null : new LinkedList<FramePacket>();
      kpQueue = new LinkedList<KeyPacket>();

      runPlayer();
   }
}
