package de.heise.asteroid.comm;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.LinkedList;
import java.util.List;

/**
 * The <code>ServerConnection</code> class implements the network
 * communication with the server.
 * <p>
 * It uses blocking UDP sockets.
 */
public class ServerConnection extends Thread {
   public final static int SERVER_PORT = 1979;
   
   private DatagramSocket socket;
   private InetAddress address;
   private List<FramePacket> fpPool;
   private List<KeyPacket> kpPool;
   private int fpCount;
   private int kpCount;
   private int packetsSent;
   private int packetsRcvd;

   /**
    * Initializes this connection with server <code>serverName</code>.
    * 
    * @param serverName the name of the server
    * @throws SocketException
    */
   public ServerConnection() throws SocketException {
      socket = new DatagramSocket();
      socket.setReceiveBufferSize(20 * FramePacket.SIZE);
      fpPool = new LinkedList<FramePacket>();
      kpPool = new LinkedList<KeyPacket>();
      fpCount = 0;
      kpCount = 0;
      packetsSent = 0;
      packetsRcvd = 0;
   }

   /**
    * Tries to connect to the server.
    * @param serverName
    * @throws UnknownHostException
    */
   public void connect(String serverName) throws UnknownHostException {
      address = InetAddress.getByName(serverName);
      socket.connect(address, SERVER_PORT);
   }
   
   /**
    * Closes the network connection. Must be called to free resources.
    */
   public void disconnect() {
      socket.close();
   }

   /**
    * Takes a <code>FramePacket</code> from the pool of unused packets.
    * If the pool is empty a new packet is created.
    * 
    * @return the packet (never <code>null<code>) 
    */
   private FramePacket getFramePacket() {
      synchronized (fpPool) {
         if (fpPool.isEmpty()) {
            ++fpCount;
            return new FramePacket();
         } else {
            return fpPool.remove(0);
         }
      }
   }
   
   /**
    * Returns a <code>FramePacket</code> to the pool of unused packets
    * @param fp the packet to dispose
    */
   public void disposeFramePacket(FramePacket fp) {
      synchronized (fpPool) {
         fpPool.add(fp);
      }
   }
   
   /**
    * Takes a <code>KeyPacket</code> from the pool of unused packets.
    * If the pool is empty a new packet is created.
    * 
    * @return the packet (never <code>null<code>) 
    */
   public KeyPacket getKeyPacket() {
      synchronized (kpPool) {
         if (kpPool.isEmpty()) {
            ++kpCount;
            return new KeyPacket();
         } else {
            return kpPool.remove(0);
         }
      }
   }
   
   /**
    * Returns a <code>KeyPacket</code> to the pool of unused packets
    * @param kp the packet to dispose
    */
   public void disposeKeyPacket(KeyPacket kp) {
      synchronized (kpPool) {
         kpPool.add(kp);
      }
   }
   
   /**
    * Receives a <code>FramePacket</code> from the server, possibly blocking
    * for the given timeout if a packet is not immediately available.
    * 
    * @return a <code>FramePacket</code> if any was received, otherwise returns null
    * @throws IOException
    * @see FramePacket
    */
   public FramePacket receive() throws IOException {
      FramePacket fp = getFramePacket();
      DatagramPacket datagramPacket = fp.getDatagramPacket();
      try {
         socket.receive(datagramPacket);
         fp.setTimeStamp(System.nanoTime());
         ++packetsRcvd;
      } catch (SocketTimeoutException e) {
         disposeFramePacket(fp);
         return null;
      } catch (IOException e) {
         System.err.println("Error receiving a frame packet: " + e.getMessage());
         throw e;
      }
      return fp;
   }

   /**
    * Sends a <code>KeyPacket</code> to the server. The packet is disposed of 
    * (i.e. recycled) immediately afterwards.
    * 
    * @param kp the packet to be sent to the server
    * @return <code>null</code> if packet was successfully sent, otherwise returns the unsent packet
    * @see KeyPacket
    */
   public KeyPacket send(KeyPacket kp) {
      try {
         socket.send(kp.getDatagramPacket());
         ++packetsSent;
         disposeKeyPacket(kp);
         return null;
      } catch (IOException e) {
         System.err.println("Error sending a keys packet: " + e.getMessage());
         return kp;
      }
   }

   /**
    * Returns the number of <code>KeyPacket</code> packets sent since connect
    * 
    * @return the number of packets sent
    */
   public int getPacketsSent() {
      return packetsSent;
   }

   /**
    * Returns the number of <code>FramePacket</code> packets received since connect
    * 
    * @return the number of packets received
    */
   public int getPacketsRcvd() {
      return packetsRcvd;
   }

   /**
    * Returns the number of <code>FramePacket</code> packets created so far.
    * 
    * @return the number of packets
    */
   public int getFramePacketCount() {
      return fpCount;
   }

   /**
    * Returns the number of <code>KeyPacket</code> packets created so far.
    * 
    * @return the number of packets
    */
   public int getKeyPacketCount() {
      return kpCount;
   }
}
