/**
 * File:    AngleTracker.java
 * Package: de.heise.asteroid.engine
 * Created: 19.06.2008 22:08:45
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.BitSet;

import de.heise.asteroid.ScreenVector;

/**
 * @author Chr. Moellenberg
 *
 */
public class AngleTracker {
   private static final int[] rawCoords = {
      0, 152, 296, 440, 584, 720, 856, 976, 1088, 1192, 1280, 1360, 1416, 1472, 1504, 1528, 1536
   };
   private static int[] coordX = new int[64];
   private static int[] coordY = new int[64];
   private static BitSet[] angleSets = new BitSet[64];
   private static int[] screenAngles = new int[256];

   static {
      System.out.print("Building angle tracker tables... ");
      for (int i = 0; i < 16; ++i) {
         coordX[i] = rawCoords[16 - i];
         coordY[i] = rawCoords[i];
         coordX[i + 16] = -rawCoords[i];
         coordY[i + 16] = rawCoords[16 - i];
         coordX[i + 32] = -rawCoords[16 - i];
         coordY[i + 32] = -rawCoords[i];
         coordX[i + 48] = rawCoords[i];
         coordY[i + 48] = -rawCoords[16 - i];
      }
      for (int sa = 0; sa < 64; ++sa) {
         angleSets[sa] = new BitSet(256);
      }
      angleSets[0].set(0, 4);
      angleSets[0].set(253, 256);
      angleSets[16].set(64);
      angleSets[32].set(125, 132);
      angleSets[48].set(192);
      for (int a = 1, i = 4; a < 16; ++a, i += 4) {
         angleSets[a].set(i, 4 + i);
         angleSets[a + 16].set(61 + i, 65 + i);
         angleSets[a + 32].set(128 + i, 132 + i);
         angleSets[a + 48].set(189 + i, 193 + i);
      }
      for (int sa = 0; sa < 64; ++sa) {
         BitSet set = angleSets[sa];
         for (int a = set.nextSetBit(0); a >= 0; a = set.nextSetBit(++a)) {
            screenAngles[a] = sa;
         }
      }
      System.out.println("done.");
   }

   private int screenAngle;
   private int angle;
   private BitSet angleSet;
   private boolean inSync;

   public AngleTracker() {
      angle = 0;
      screenAngle = 0;
      angleSet = (BitSet)angleSets[0].clone();
      inSync = false;
   }

   public int getAngle() {
      return angle;
   }

   public ScreenVector getDir() {
      return new ScreenVector(coordX[screenAngle], coordY[screenAngle]);
   }

   public boolean isInSync() {
      return inSync;
   }
   
   public void initializeAngle(int dx, int dy) {
      if (!matchScreenAngle(screenAngle, dx, dy)) {
         for (screenAngle = 63; screenAngle >= 0; --screenAngle) {
            if (matchScreenAngle(screenAngle, dx, dy)) {
               break;
            }
            if (screenAngle == 0) {
               System.out.printf("Screen angle initialization failed on (%d,%d)\n", dx, dy);
               return;
            }
         }
      }
      if ((screenAngle & 0x1f) == 16) {
         angle = screenAngle << 2;
         inSync = true;
      } else {
         angleSet = (BitSet)angleSets[screenAngle].clone();
         angle = angleSet.nextSetBit(0);
         inSync = false;
      }
   }
   
   public void updateAngle(int dx, int dy, int keys) {
      int a = angle;
      if (inSync) {
         if (dx == 0) {
            a = (dy < 0) ? 192 : 64; 
         } else {
            if ((keys & FrameProcessor.KEY_LEFT) != 0) {
               a = (a + 3) & 0xff;
            } else if ((keys & FrameProcessor.KEY_RIGHT) != 0) {
               a = (a - 3) & 0xff;
            }
         }
         int sa = screenAngles[a];
         if (matchScreenAngle(sa, dx, dy)) {
            angle = a;
            screenAngle = sa;
         } else {
            if (!matchScreenAngle(screenAngle, dx, dy)) {
               System.out.println("Screen angle mismatch");
               resynchAngle(dx, dy);
            }
         }
      } else {
         synchronizeAngle(dx, dy, keys);
      }
   }

   private void resynchAngle(int dx, int dy) {
      for (int offs = 1; offs <= 128; ++offs) {
         int a = (angle + offs * 3) & 0xff;
         int sa = screenAngles[a];
         if (matchScreenAngle(sa, dx, dy)) {
            angle = a;
            screenAngle = sa;
            return;
         }
         a = (angle - offs * 3) & 0xff;
         sa = screenAngles[a];
         if (matchScreenAngle(sa, dx, dy)) {
            angle = a;
            screenAngle = sa;
            return;
         }
      }
      initializeAngle(dx, dy);
   }

   private void synchronizeAngle(int dx, int dy, int keys) {
      BitSet set = new BitSet(256);
      int a = angle;
      if ((keys & FrameProcessor.KEY_LEFT) != 0) {
         a = (a + 3) & 0xff;
         for (int i = angleSet.nextSetBit(0); i >= 0; i = angleSet.nextSetBit(++i)) {
            set.set((i + 3) & 0xff);
         }
      } else if ((keys & FrameProcessor.KEY_RIGHT) != 0) {
         a = (a - 3) & 0xff;
         for (int i = angleSet.nextSetBit(0); i >= 0; i = angleSet.nextSetBit(++i)) {
            set.set((i - 3) & 0xff);
         }
      } else {
         set.or(angleSet);
      }
      int sa = screenAngles[a];
//      System.out.printf("OLD: Angle %d (screen: %d), angles: %s\n", angle, screenAngle, bitsetToString(angleSet));
//      System.out.printf("NEW: Angle %d (screen: %d), angles:%s\n", a, sa, bitsetToString(set));
      if (coordX[sa] != dx || coordY[sa] != dy) {
         System.out.println("Screen angle mismatch");
         inSync = false;
         sa = resynchScreenAngle(sa, dx, dy);
         if (sa == screenAngles[a]) {
            return; // same value, must be wrong ==> give up
         }
         if (sa != screenAngle) {
            angleSet = (BitSet)angleSets[screenAngle].clone();
            screenAngle = sa;
         }
      } else {
         screenAngle = sa;
         angleSet = (BitSet)angleSets[screenAngle].clone();
         angleSet.and(set);
      }
      if (angleSet.get(a)) {
         angle = a;
         inSync = (angleSet.cardinality() == 1);
      } else if (!angleSet.get(angle)) {
         a = angleSet.nextSetBit(0);
         if (a >= 0) {
            angle = a;
         } else {
            System.out.printf("Angle synchronization lost: %d\n", angle);
            inSync = false;
         }
      }
//      StringBuilder strb = new StringBuilder("screenAngle ").append(screenAngle).append(", Dir (");
//      strb.append(dx).append(',').append(dy).append("), angles:").append(bitsetToString(angleSet));
//      System.out.println(strb);
   }
   
   private static boolean matchScreenAngle(int sa, int dx, int dy) {
      return coordX[sa] == dx && coordY[sa] == dy;
   }

   private static int resynchScreenAngle(int sa, int dx, int dy) {
      for (int i = sa - 3; i <= sa + 60; ++i) {
         int sa_tmp = i & 0x3f;
         if (matchScreenAngle(sa_tmp, dx, dy)) {
            return sa_tmp;
         }
      }
      System.out.printf("Cannot find screen angle for (%d,%d), even after full search\n", dx, dy);
      return sa;
   }

   private static String bitsetToString(BitSet set) {
      if (set.cardinality() == 0) {
         return " none";
      }
      StringBuilder strb = new StringBuilder();
      for (int i = set.nextSetBit(0); i >= 0; i = set.nextSetBit(++i)) {
         strb.append(' ').append(i);
      }
      return strb.toString();
   }
}
