/**
 * File:    AsteroidSequenceTracker.java
 * Package: de.heise.asteroid
 * Created: 12.05.2008 02:52:48
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.LinkedList;
import java.util.List;

import de.heise.asteroid.model.Asteroid;

/**
 * @author Chr. Moellenberg
 *
 */
public class AsteroidSequenceTracker {
   public static final int MAX_ASTEROIDS = 27;
   private AsteroidTracker[] trackers;
   private int curSlot;
   private int maxSlot;
   private boolean inSync;
   private int curFrame;
   private int numObjects;
   
   public AsteroidSequenceTracker() {
      trackers = new AsteroidTracker[MAX_ASTEROIDS];
      reset();
   }
   
   public void reset() {
      maxSlot = 0;
      start(0);
   }

   public void start(int frameNo) {
      numObjects = 0;
      curSlot = 0;
      inSync = true;
      curFrame = frameNo;
      for (int s = 0; s < maxSlot; ++s) {
         AsteroidTracker t = trackers[s];
         if (t != null) {
            if (t.isExploding()) {
               if (t.getFramesRemaining(curFrame) <= 0) {
                  // explosion is over, get rid of Asteroid instance
                  t.dispose();
                  trackers[s] = null;
//                  System.out.println(dump());
               }
            }
         }
      }
   }

   public void trackTarget(int sx, int sy, int type, int size) {
      ++numObjects;
      if (curSlot >= maxSlot) {
         // highest existing index exceeded, make more space if possible
         if (maxSlot < MAX_ASTEROIDS) {
            trackers[maxSlot++] = null;
         } else {
            inSync = false;
            System.out.println("Asteroid table overflow");
            System.out.println(dump());
            return;
         }
      }
      AsteroidTracker t = trackers[curSlot];
      if (t != null) {
         // current slot is in use
         if (t.match(curFrame, sx, sy, type, size)) {
            t.update(sx, sy, type, size);
         } else {
            inSync = false;
            System.out.println("AsteroidSequenceTracker out of sync");
            System.out.println(dump());
         }
      } else {
         // current slot is not in use
         int nextSlot = getNextSlot(curSlot);
         t = trackers[nextSlot];
         if (t != null) {
            if (t.match(curFrame, sx, sy, type, size)) {
               t.update(sx, sy, type, size);
               curSlot = nextSlot;
            } else {
               t = null;
            }
         }
         if (t == null) {
            if (GameStatus.isExplosionType(type)) {
               System.out.printf("[%d] Spurious explosion detected\n", curSlot);
            }
            trackers[curSlot] = new AsteroidTracker(curFrame, sx, sy, type, size, curSlot);
//            System.out.println(dump());
         }
      }
      ++curSlot;
   }
   
   private int getNextSlot(int slot) {
      for (int s = slot; s < maxSlot; ++s) {
         if (trackers[s] != null) {
            return s;
         }
      }
      return slot;
   }

   /**
    * Returns the synchronization state of this <code>AsteroidSequenceTracker</code>
    * @return <code>true</code> if the tracker "thinks" it is synchronized, otherwise <code>false</code>
    */
   public boolean isInSync() {
      return inSync;
   }

   public List<Asteroid> getAsteroids(List<Asteroid> list, boolean inclExpl) {
      if (list == null) {
         list = new LinkedList<Asteroid>();
      } else {
         list.clear();
      }
      for (int i = 0; i < maxSlot; ++i) {
         if (trackers[i] != null && (inclExpl || !trackers[i].isExploding())) {
            list.add(trackers[i].getAsteroid());
         }
      }
      return list;
   }
   
   public String dump() {
      StringBuilder strb = new StringBuilder("|");
      for (int s = 0; s < maxSlot; ++s) {
         if (trackers[s] == null) {
            strb.append(" - |");
         } else {
            if (trackers[s].isExploding()) {
               strb.append(" # |");
            } else {
               switch (trackers[s].getAsteroid().getType()) {
                  case GameStatus.TYPE_1:
                     strb.append(" 0 |");
                     break;
                  case GameStatus.TYPE_2:
                     strb.append(" 1 |");
                     break;
                  case GameStatus.TYPE_3:
                     strb.append(" 2 |");
                     break;
                  case GameStatus.TYPE_4:
                     strb.append(" 3 |");
                     break;
               }
            }
         }
      }
      return strb.toString();
   }

   public void cleanup() {
      maxSlot = curSlot;
      if (numObjects == 0) {
         cleanTable();
      }
   }

   private void cleanTable() {
      for (int s = 0; s < maxSlot; ++s) {
         AsteroidTracker t = trackers[s];
         if (t != null) {
            t.dispose();
            trackers[s] = null;
         }
      }
      maxSlot = 0;
   }
}
