/**
 * File:    AsteroidTracker.java
 * Package: de.heise.asteroid.engine
 * Created: 12.06.2008 21:01:12
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import de.heise.asteroid.Position;
import de.heise.asteroid.ScreenVector;
import de.heise.asteroid.model.Asteroid;
import de.heise.asteroid.util.SimpleMath;

/**
 * @author Chr. Moellenberg
 *
 */
public class AsteroidTracker {
   private int[] xBuf;
   private int[] yBuf;
   private int bufPos;
   private int baseFrame;
   private int curFrame;
   private int px;
   private int py;
   private int vx;
   private int vy;
   private int sqTol;
   private Position pos;
   private ScreenVector speed;
   private int type;
   private int size;
   private int slot;
   private Asteroid asteroid;
   private Explosion explosion;

   public AsteroidTracker(int frameNo, int sx, int sy, int t, int s, int sl) {
      xBuf = new int[9];
      yBuf = new int[9];
      baseFrame = frameNo;
      curFrame = frameNo;
      xBuf[0] = sx;
      yBuf[0] = sy;
      bufPos = 1;
      px = sx * 8;
      py = sy * 8;
      vx = 0;
      vy = 0;
      sqTol = 128; // initial tolerance, may need further testing and adjustment
      pos = null;
      speed = ScreenVector.NULL_VECTOR;
      type = t;
      size = s;
      slot = sl;
      explosion = null;
      asteroid = new Asteroid(getPos(), speed, t, s);
      if (GameStatus.isExplosionType(t)) {
         startExplosion(sx, sy, t, s);
      }
//      System.out.printf("[%d] New asteroid %s\n", slot, asteroid);
   }

   public void dispose() {
//      System.out.printf("[%d] End of explosion: %s\n", slot, asteroid);
      asteroid = null;
   }

   /**
    * Returns the speed vector of the tracked target. This may be (0, 0) 
    * if the speed is not yet known, or if the target is exploding.
    * 
    * @return the speed vector of the tracked target
    */
   public ScreenVector getSpeed() {
      return speed;
   }

   /**
    * Returns the position of the tracked target. The tracker will try its 
    * best to return at least a good estimate of the real position.
    * 
    * @return the position of the tracked target
    */
   public Position getPos() {
      if (pos == null) {
         pos = new Position(px, py);
      }
      return pos;
   }

   public Asteroid getAsteroid() {
      return asteroid;
   }

   /**
    * Checks if the tracked target is, within the given tolerance, at the 
    * given screen position and also has the given type and size
    * 
    * @param frameNo the frame number
    * @param sx the horizontal screen position
    * @param sy the vertical screen position
    * @param t the target type
    * @param s the target size
    * @return true if the target is visible at (sx,sy), false otherwise
    */
   public boolean match(int frameNo, int sx, int sy, int t, int s) {
      curFrame = frameNo;
      if (GameStatus.isExplosionType(t)) {
         if (explosion != null) {
            return explosion.match(curFrame, sx, sy, t, s);
         }
         int x = px >> 3;
         int y = py >> 3;
         if (sx != x || sy != y) {
            System.out.printf("[%d] Explosion seen at (%d,%d) but Asteroid expected at (%d,%d)\n", slot, sx, sy, x, y);
            return false;
         }
      } else {
         if (t != type) {
//            System.out.printf("[%d] Asteroid type %d does not match %d\n", slot, t, type);
            return false;
         }
         if (s != size) {
//            System.out.printf("[%d] Asteroid size %d does not match %d\n", slot, s, size);
            return false;
         }
         int x = SimpleMath.normalize((px + vx * (curFrame - baseFrame)) >> 3, 0, 1024);
         int y = SimpleMath.normalize((py + vy * (curFrame - baseFrame)) >> 3, 128, 896);
         int dx = SimpleMath.normalize(sx - x, -512, 512);
         int dy = SimpleMath.normalize(sy - y, -384, 384);
         if (dx * dx + dy * dy > sqTol) {
//            System.out.printf("[%d] Position (%d,%d) out of sq. tolerance %d, distance is (%d,%d)\n", slot, sx, sy, sqTol, dx, dy);
            return false;
         }
      }
      return true;
   }

   /**
    * Updates this tracker with the latest screen position, type, and size of 
    * the tracked object.
    * 
    * @param sx the horizontal screen position
    * @param sy the vertical screen position
    * @param t the target type
    * @param s the target size
    */
   public void update(int sx, int sy, int t, int s) {
      if (GameStatus.isExplosionType(t)) {
         if (explosion == null) {
            startExplosion(sx, sy, t, s);
         }
      } else {
         if (explosion == null) {
            updatePosition(sx, sy);
         } else {
            System.out.printf("[%d] Asteroid was already exploding, but is not now\n", slot);
         }
      }
      asteroid.update(getPos(), speed);
   }

   public int getFramesRemaining(int frameNo) {
      return explosion != null ? explosion.getFramesRemaining(frameNo) : Integer.MAX_VALUE;
   }

   public boolean isExploding() {
      return explosion != null;
   }

   private void updatePosition(int sx, int sy) {
      if (bufPos < 9) {
         if (curFrame != baseFrame + bufPos) {
            bufPos = 0;
            baseFrame = curFrame;
         }
         xBuf[bufPos] = sx;
         yBuf[bufPos] = sy;
         px = sx << 3;
         py = sy << 3;
         if (++bufPos >= 9) {
            vx = SimpleMath.normalize(xBuf[8] - xBuf[0], -512, 512);
            vy = SimpleMath.normalize(yBuf[8] - yBuf[0], -384, 384);
            adjustPosition();
            baseFrame = curFrame;
            speed = new ScreenVector(vx, vy);
//            System.out.printf("[%d] Asteroid at %d,%d has speed %d,%d\n", slot, px, py, vx, vy);
            sqTol = 2;
         }
      } else {
         px = SimpleMath.normalize(px + vx * (curFrame - baseFrame), 0, 8192);
         py = SimpleMath.normalize(py + vy * (curFrame - baseFrame), 1024, 7168);
         baseFrame = curFrame;
      }
      pos = null;
   }

   private void adjustPosition() {
      boolean foundX = false;
      boolean foundY = false;
      for (int offs = 0; offs < 8; ++offs) {
         if (!foundX && backtrack(vx, offs, xBuf, 0, 8192)) {
            px += offs;
            foundX = true;
         }
         if (!foundY && backtrack(vy, offs, yBuf, 1024, 7168)) {
            py += offs;
            foundY = true;
         }
         if (foundX && foundY) {
            return;
         }
      }
      System.out.printf("[%d] Co-ordinate backtracking failed\n", slot);
   }

   private boolean backtrack(int v, int offs, int[] buf, int min, int max) {
      int p = (buf[8] << 3) + offs;
      for (int i = 7; i >= 0; --i) {
         p -= v;
         if (p < min) {
            p += max - min;
         } else if (p >= max) {
            p -= max - min;
         }
         if (p >> 3 != buf[i]) {
            return false;
         }
      }
      return true;
   }

   private void startExplosion(int sx, int sy, int t, int s) {
      vx = 0;
      vy = 0;
      speed = ScreenVector.NULL_VECTOR;
      explosion = new Explosion(curFrame, sx, sy, t, s);
      asteroid.setExploding(true);
      int index = explosion.getIndex();
      if (index < 0) {
         System.out.printf("[%d] Unable to synchronize explosion sequence\n", slot);
         explosion = null;
      }
//      System.out.printf("[%d] Asteroid exploding at (%d,%d), animation index %d\n", slot, px, py, index);
   }
}
