/**
 * File:    Engine.java
 * Package: de.heise.asteroid
 * Created: 19.05.2008 22:55:39
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.List;
import java.util.ListIterator;

import de.heise.asteroid.comm.Communicator;
import de.heise.asteroid.comm.FramePacket;
import de.heise.asteroid.comm.KeyPacket;
import de.heise.asteroid.player.Player;
import de.heise.asteroid.util.WorkerRunnable;

/**
 * The <code>Engine</code> class implements main player loop.
 * 
 * @author Chr. Moellenberg
 */
public class Engine implements WorkerRunnable {
   public static final int STATE_UNKNOWN = 0;
   public static final int STATE_WAIT_START = 1;
   public static final int STATE_WAIT_PLAY = 2;
   public static final int STATE_PLAYING = 3;
   public static final int STATE_GAME_OVER = 4;
   public static final int STATE_HIGHSCORE = 5;
   public static final int STATE_ERROR = 99;

   private Communicator comm;
   int frameCnt;
   private FrameInterpreter frameInterpreter;
   private TextInterpreter textInterpreter;
   private List<FramePacket> fpQueue;
   private List<KeyPacket> kpQueue;
   private int state;
   private FrameProcessor[] stateHandlers;
   private GameStatus status;
   private boolean syncComm;
   private int gamesCompleted;
   private int gamesToPlay;

   /**
    * Initializes the player <code>Engine</code> with the given communication 
    * engine and game status.
    * 
    * @param c the <code>Communicator</code>
    * @param s the <code>GameStatus</code>
    */
   public Engine(Communicator c, GameStatus s, int numGames) {
      comm = c;
      status = s;
      gamesToPlay = numGames;
      frameInterpreter = new FrameInterpreter(status);
      textInterpreter = new TextInterpreter(status);
      fpQueue = null;
      kpQueue = null;
      syncComm = true;
      state = STATE_UNKNOWN;
      stateHandlers = new FrameProcessor[6];
      FrameProcessor idle = new IdleProcessor(this);
      stateHandlers[STATE_UNKNOWN] = idle;
      stateHandlers[STATE_WAIT_START] = new GameStarter(this);
      stateHandlers[STATE_WAIT_PLAY] = idle;
      stateHandlers[STATE_PLAYING] = null;
      stateHandlers[STATE_GAME_OVER] = idle;
      stateHandlers[STATE_HIGHSCORE] = null;
   }

   /**
    * Sets the packet queues. Please note that, if there is no 
    * <code>FramePacket</code> queue, the engine will try to 
    * communicate synchronously. This method must be called before 
    * the worker thread is started.
    * 
    * @param fpq the <code>FramePacket</code> queue to set
    * @param kpq the <code>KeyPacket</code> queue to set
    */
   public void setQueues(List<FramePacket> fpq, List<KeyPacket> kpq) {
      fpQueue = fpq;
      kpQueue = kpq;
   }

   /**
    * Registers a player strategy with this engine.
    * 
    * @param p the <code>Player</code> to register
    */
   public void registerPlayer(Player p) {
      stateHandlers[STATE_PLAYING] = p;
      stateHandlers[STATE_HIGHSCORE] = new HallOfFameEditor(this, p.getInitials());
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadInitialize()
    */
   public void workerInitialize() {
      gamesCompleted = 0;
      syncComm = (fpQueue == null);
      System.out.println("Engine started in " + (syncComm ? "synchronous" : "asynchronous") + " mode.");
      frameCnt = 0;
      state = STATE_UNKNOWN;
      System.out.println("Initial state: " + getStateString(state));
      if (syncComm) {
         comm.workerInitialize();
      }
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadMainLoop()
    */
   public boolean workerMainLoop() {
      FramePacket fp = syncComm ? comm.receiveFramePacket() : getFramePacket();
      if (fp != null) {
         if (syncComm) {
            comm.transmitKeyPacket();
         }
         int frameNo = fp.getRealFrameNo();
         textInterpreter.interpretFrame(fp);
         int newState = textInterpreter.getState();
         if (newState != state) {
            if (newState == STATE_ERROR) {
               return false;
            }
            if (state == STATE_HIGHSCORE || (state == STATE_GAME_OVER && newState != STATE_HIGHSCORE)) {
               ++gamesCompleted;
            }
            System.out.printf("State transition: %s --> %s\n", getStateString(state), getStateString(newState));
            state = newState;
            frameInterpreter.transitState(state);
            if (stateHandlers[state] != null) {
               comm.flushKeyQueue();
               stateHandlers[state].initialize();
            } else {
               System.out.println("ERROR: No FrameProcessor for state " + getStateString(state));
               state = STATE_UNKNOWN;
            }
         }
         frameInterpreter.interpretFrame(fp);
         comm.recycle(fp);
         if (stateHandlers[state] != null) {
            stateHandlers[state].process(status, frameNo);
         }
         ++frameCnt;
         return gamesCompleted < gamesToPlay;
      }
      return false;
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadFinalize()
    */
   public void workerFinalize() {
      System.out.printf("Engine stopped after %d frames.\n", frameCnt);
   }

   /**
    * Takes the next <code>FramePacket</code> from the queue.
    * 
    * @return the <code>FramePacket</code>
    */
   private FramePacket getFramePacket() {
      FramePacket fp = null;
      if (fpQueue != null) {
         synchronized (fpQueue) {
            while (fpQueue.isEmpty()) {
               try {
                  fpQueue.wait();
               } catch (InterruptedException e) {
                  return null;
               }
            }
            fp = fpQueue.remove(0);
         }
      }
      return fp;
   }
   
   public void sendKeys(byte keys, int frameNo) {
      if (kpQueue != null) {
         KeyPacket kp = comm.getKeyPacket();
         kp.setKeys(keys);
         kp.setTargetFameNo(frameNo);
         synchronized (kpQueue) {
            if (kpQueue.isEmpty()) {
               kpQueue.add(kp);
               kp = null;
            } else {
               ListIterator<KeyPacket> it = kpQueue.listIterator();
               while (it.hasNext()) {
                  KeyPacket p = it.next();
                  int tfn = p.getTargetFameNo();
                  if (tfn == frameNo) {
                     p.setKeys(keys); // Overwrite
                  } else if (tfn > frameNo) {
                     it.previous();
                     it.add(kp);
                     kp = null;
                  }
               }
            }
         }
         if (kp != null) {
            comm.disposeKeyPacket(kp);
         }
      }
   }

   public int getGamesCompleted() {
      return gamesCompleted;
   }

   public void dumpKeyQueue() {
      comm.dumpKeyQueue();
   }
   
   private static String getStateString(int stat) {
      switch (stat) {
         case STATE_UNKNOWN:
            return "UNKNOWN";
         case STATE_WAIT_START:
            return "WAIT_START";
         case STATE_WAIT_PLAY:
            return "WAIT_PLAY";
         case STATE_PLAYING:
            return "PLAYING";
         case STATE_GAME_OVER:
            return "GAME_OVER";
         case STATE_HIGHSCORE:
            return "HIGHSCORE";
         default:
            return "?????";
      }
   }
}
