package de.heise.asteroid.engine;

import java.util.LinkedList;
import java.util.List;

import de.heise.asteroid.model.Asteroid;
import de.heise.asteroid.model.Saucer;
import de.heise.asteroid.model.Ship;
import de.heise.asteroid.model.Shot;


/**
 * The <code>GameStatus</code> class represents game status which can be
 * decoded from a <code>FramePacket</code>.
 * <p>
 * It describes the currently visible ship, saucer, asteroids and shots.
 */
public class GameStatus {
   public static final int TYPE_1 = 0;
   public static final int TYPE_2 = 1;
   public static final int TYPE_3 = 2;
   public static final int TYPE_4 = 3;
   public static final int EXPL_0 = 4;
   public static final int EXPL_1 = 5;
   public static final int EXPL_2 = 6;
   public static final int EXPL_3 = 7;
   public static final int SAUCER = 8;

   public final static int MAX_ASTEROIDS = 30;
   public final static int MAX_SHOTS = 20;
   public final static int MAX_STRINGS = 10;

   private List<Asteroid> asteroids;
   private List<Shot> shots;
   private Saucer saucer;
   private Ship ship;
   private int lifeCount = 0;
   int playerNo;
   int[] score;
   private String initials;

   public GameStatus() {
      asteroids = new LinkedList<Asteroid>();
      shots = new LinkedList<Shot>();
      saucer = null;
      ship = null;
      lifeCount = 0;
      initials = null;
      score = new int[3];
      playerNo = 0;
   }

   public void clear() {
      asteroids.clear();
      shots.clear();
      saucer = null;
      ship = null;
      lifeCount = 0;
   }

   /* (non-Javadoc)
    * @see java.lang.Object#toString()
    */
   @Override
   public String toString() {
      StringBuilder str = new StringBuilder();
      if (ship != null) {
         str.append(ship.toString()).append('\n');
      }
      if (saucer != null) {
         str.append(saucer.toString()).append('\n');
      }
      for (Asteroid a : asteroids) {
         str.append(a.toString()).append('\n');
      }
      for (Shot s : shots) {
         str.append(s.toString()).append('\n');
      }
      return str.toString();
   }

   /**
    * Returns the current ship.
    * @return the current ship (may be invalid)
    */
   public Ship getShip() {
      return ship;
   }

   /**
    * Returns the current saucer.
    * @return the current saucer (may be invalid)
    */
   public Saucer getSaucer() {
      return saucer;
   }

   /**
    * Returns the current list of asteroids.
    * @return the current list of asteroids
    */
   public List<Asteroid> getAsteroids() {
      return asteroids;
   }

   /**
    * Returns the number of asteroids in the list.
    * @return the max. number of asteroids in the list
    */
   public int getNumAsteroids() {
      return asteroids.size();
   }

   /**
    * Returns the current list of shots.
    * @return the current list of shots
    */
   public List<Shot> getShots() {
      return shots;
   }

   /**
    * Sets the number of lives remaining.
    * @param lc the number to set
    */
   public void setLifeCount(int lc) {
      lifeCount  = lc;
   }
   
   /**
    * Returns the number of lives remaining in this game, including the 
    * current life.
    * 
    * @return the number of lives remaining
    */
   public int getLifeCount() {
      return lifeCount;
   }

   /**
    * Returns the current setting of the initials on the 
    * hall-of-fame screen.
    * 
    * @return the initials
    */
   public String getInitials() {
      return initials;
   }

   /**
    * Sets the initials currently found on the hall-of-fame screen.
    *  
    * @param initials the initials to set
    */
   public void setInitials(String initials) {
      this.initials = initials;
   }

   public void setPlayerNo(int p) {
      playerNo = p;
   }

   public void setScore(int player, int s) {
      if (player >= 0 && player <= 2) {
         score[player] = s;
      } else {
         System.out.println("Cannot set score - invalid player number");
      }
   }

   public void addAsteroids(List<Asteroid> astList) {
      asteroids.addAll(astList);
   }

   public void addShots(List<Shot> shotList) {
      shots.addAll(shotList);
   }

   public void setSaucer(Saucer s) {
      saucer = s;
   }

   public void setShip(Ship s) {
      ship = s;
   }

   /**
    * Returns if a given asteroid type is an explosion.
    * 
    * @param t the asteroid type
    * @return true if t is an explosion type, otherwise false
    */
   public static boolean isExplosionType(int t) {
      return t >= EXPL_0 && t <= EXPL_3;
   }
}
