/**
 * File:    ShipTracker.java
 * Package: de.heise.asteroid.engine
 * Created: 16.06.2008 19:33:45
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.BitSet;

import de.heise.asteroid.Position;
import de.heise.asteroid.ScreenVector;
import de.heise.asteroid.model.Ship;
import de.heise.asteroid.util.SimpleMath;

/**
 * @author Chr. Moellenberg
 *
 */
public class ShipTracker {
   private static final int[] rawCoords = {
      0, 152, 296, 440, 584, 720, 856, 976, 1088, 1192, 1280, 1360, 1416, 1472, 1504, 1528, 1536
   };
   private static int[] coordX = new int[64];
   private static int[] coordY = new int[64];
   private static BitSet[] angleSets = new BitSet[64];
   private static int[] screenAngles = new int[256];

   static {
      System.out.print("Building ship angle tables... ");
      for (int i = 0; i < 16; ++i) {
         coordX[i] = rawCoords[16 - i];
         coordY[i] = rawCoords[i];
         coordX[i + 16] = -rawCoords[i];
         coordY[i + 16] = rawCoords[16 - i];
         coordX[i + 32] = -rawCoords[16 - i];
         coordY[i + 32] = -rawCoords[i];
         coordX[i + 48] = rawCoords[i];
         coordY[i + 48] = -rawCoords[16 - i];
      }
      for (int sa = 0; sa < 64; ++sa) {
         angleSets[sa] = new BitSet(256);
      }
      angleSets[0].set(0, 4);
      angleSets[0].set(253, 256);
      angleSets[16].set(64);
      angleSets[32].set(125, 132);
      angleSets[48].set(192);
      for (int a = 1, i = 4; a < 16; ++a, i += 4) {
         angleSets[a].set(i, 4 + i);
         angleSets[a + 16].set(61 + i, 65 + i);
         angleSets[a + 32].set(128 + i, 132 + i);
         angleSets[a + 48].set(189 + i, 193 + i);
      }
      for (int sa = 0; sa < 64; ++sa) {
         BitSet set = angleSets[sa];
         for (int a = set.nextSetBit(0); a >= 0; a = set.nextSetBit(++a)) {
            screenAngles[a] = sa;
         }
      }
      System.out.println("done.");
   }

   private static final int DELAY = 3;
   private static final int MAX_SPEED = 16383;
   private Ship ship;
   private int curFrame;
   private int px;
   private int py;
   private int vx; // caution - ship speed is in 256ths of screen, not in 8ths!
   private int vy;
   private boolean visible;
   private Position pos;
   private ScreenVector speed;
   private int shipDetected;
   private int xTmp;
   private int yTmp;
   private int keyBuf[];
   private int keyIndex;
   private boolean thrustEnable;
   private AngleTracker angleTracker;

   public ShipTracker() {
      angleTracker = new AngleTracker();
      keyBuf = new int[DELAY];
      keyIndex = 0;
      reset();
   }
   
   public void reset() {
      ship = null;
      pos = null;
      start(0);
   }

   public void start(int frameNo) {
      curFrame = frameNo;
      shipDetected = 0;
      visible = false;
   }

   public void cleanup() {
      if (!visible) {
         for (int i = 0; i < DELAY; ++i) {
            keyBuf[i] = 0;
         }
         keyIndex = 0;
         vx &= 0xff; // preserve fractional value
         vy &= 0xff;
      }
   }

   public Ship getShip() {
      return visible ? ship : null;
   }

   public void trackShip(int sx, int sy, int dx, int dy, int keys) {
      keyBuf[keyIndex++] = keys;
      keyIndex %= DELAY;
      switch (shipDetected) {
         case 0:
            xTmp = dx;
            yTmp = dy;
            ++shipDetected;
            break;
         case 1:
            ++shipDetected;
            if (ship == null) {
               px = sx << 3;
               py = sy << 3;
               vx = 0;
               vy = 0;
               pos = null;
               speed = null;
               angleTracker.initializeAngle(xTmp - dx, yTmp - dy);
               ship = new Ship(getPos(), angleTracker.getDir(), angleTracker.getAngle());
            } else {
               angleTracker.updateAngle(xTmp - dx, yTmp - dy, keyBuf[keyIndex]);
               updateSpeed(sx, sy, angleTracker.getAngle(), keyBuf[keyIndex]);
               updatePosition(sx, sy);
            }
            visible = true;
            break;
      }
   }
   
   public Position getPos() {
      if (pos == null) {
         pos = new Position(px, py);
      }
      return pos;
   }
   
   public ScreenVector getSpeed() {
      if (speed == null) {
         speed = new ScreenVector(vx >> 5, vy >> 5);
      }
      return speed;
   }
   
   private void updatePosition(int sx, int sy) {
      px = sx << 3;
      py = sy << 3;
      pos = null;
      ship.update(getPos(), angleTracker.getDir(), angleTracker.getAngle(), getSpeed());
   }

   private void updateSpeed(int sx, int sy, int angle, int keys) {
      vx = ((sx << 3) - px) << 5;
      vy = ((sy << 3) - py) << 5;
      speed = null;
      if (false) { // if (thrustEnable)
         if ((keys & FrameProcessor.KEY_THRUST) != 0) {
            // thrust
            vx += SimpleMath.astCos[angle] * 2;
            if (vx > MAX_SPEED) {
               vx = MAX_SPEED;
            } else if (vx < -MAX_SPEED) {
               vx = -MAX_SPEED;
            }
            vy += SimpleMath.astSin[angle] * 2;
            if (vy > MAX_SPEED) {
               vy = MAX_SPEED;
            } else if (vy < -MAX_SPEED) {
               vy = -MAX_SPEED;
            }
         } else {
            // brake
            if (vx != 0) {
               vx -= (vx / 256) * 2 - 1;
            }
            if (vy != 0) {
               vy -= (vy / 256) * 2 - 1;
            }
         }
         speed = null;
      }
   }

   public void setThrustEnable(boolean enable) {
      thrustEnable = enable;
   }
}
