/**
 * File:    FrameInterpreter.java
 * Package: de.heise.asteroid
 * Created: 18.05.2008 18:47:39
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.LinkedList;
import java.util.List;

import de.heise.asteroid.Position;
import de.heise.asteroid.comm.FramePacket;

/**
 * @author Chr. Moellenberg
 *
 */
public class TextInterpreter {
   List<Text> strings;
   int state;
   GameStatus status;
   
   public TextInterpreter(GameStatus stat) {
      status = stat;
      strings = new LinkedList<Text>();
      state = Engine.STATE_UNKNOWN;
   }
   
   public int getState() {
      return state;
   }

   /**
    * Collects state information about the game by interpreting the vector ram contained 
    * in a <code>FramePacket</code> and inspecting messages which appear on screen.
    * 
    * @param fp the <code>FramePacket</code> to interpret
    */
   public void interpretFrame(FramePacket fp) {
      int px = 0, py = 0;
      int tx = 0, ty = 0;
      StringBuilder strb = new StringBuilder();
      strings.clear();

      final int initialJump = fp.getRamAt(0);
      if (initialJump != 0xe001 && initialJump != 0xe201) {
         System.err.println(String.format("Bad instruction in vector RAM[0]: %04x", initialJump));
         state = Engine.STATE_ERROR; // should not occur, first instruction is always a jump
         return;
      }
      
      for (int pc = 1; true;) {
         final int currentWord = fp.getRamAt(pc++);
         int op = currentWord >> 12;
         switch (op) {
            case 0xa: { // LABS
               finalizeText(strb, tx, ty);
               final int nextWord = fp.getRamAt(pc++);
               py = currentWord & 0x3ff;
               px = nextWord & 0x3ff;
               break;
            }
            case 0xb: // HALT
               checkState();
               return;
            case 0xc: // JSRL
               final int addr = currentWord & 0xfff;
               if (addr >= 0xa78 && addr <= 0xb63) {
                  if (strb.length() == 0) {
                     tx = px;
                     ty = py;
                  }
                  strb.append(getCharByAddr(addr));
                  break;
               } else {
                  finalizeText(strb, tx, ty);
               }
               break;
            case 0xd: // RTSL
               System.err.println(String.format("Unexpected RTSL instruction in vector RAM[%03x]", pc - 1));
               return;
            case 0xe: // JMPL
               System.err.println(String.format("Unexpected JMPL instruction in vector RAM[%03x]", pc - 1));
               return;
            case 0xf: // SVEC
               break;
            default:  // VCTR
               break;
         }
      }
   }
   
   private void finalizeText(StringBuilder strb, int sx, int sy) {
      if (strb.length() > 0) {
         String s = strb.toString();
         strb.setLength(0);
         strings.add(new Text(s, new Position(sx, sy))); // intentionally using screen co-ordinates!
      }
   }
   
   private void checkState() {
      int newState = Engine.STATE_UNKNOWN;
      for (Text t : strings) {
         String s = t.getText().trim();
         int x = t.getPos().getX();
         int y = t.getPos().getY();
         if (s.startsWith("STARTKNO") ||        // STARTKNOEPFE DRUECKEN
            s.startsWith("HOECHSTE")) {         // HOECHSTERGEBNIS
            status.setPlayerNo(0);
            newState = Engine.STATE_WAIT_START;
         } else if (s.equals("SPIELENDE")) {    // SPIELENDE
            newState = Engine.STATE_GAME_OVER;
         } else if (s.equals("SPIELER 1")) {    // SPIELER 1
            status.setPlayerNo(1);
            newState = Engine.STATE_WAIT_PLAY;
         } else if (s.equals("SPIELER 2")) {    // SPIELER 2
            status.setPlayerNo(2);
            newState = Engine.STATE_WAIT_PLAY;
         } else if (s.startsWith("IHR ERGE") || // IHR ERGEBNIS IST EINES DER ZEHN BESTEN 
                    s.startsWith("BITTE GE") || // BITTE GEBEN SIE IHRE INITIALEN EIN
                    s.startsWith("ZUR BUCH") || // ZUR BUCHSTABENWAHL ROTATE DRUECKEN
                    s.startsWith("WENN BUC")) { // WENN BUCHSTABE OK HYPERSPACE DRUECKEN 
            newState = Engine.STATE_HIGHSCORE;
         } else if (y == 876) {
            switch (x) {
               case 100:
                  status.setScore(1, parseNumber(s));    // player 1 score
                  break;
               case 480:
                  status.setScore(0, parseNumber(s));    // highscore
                  break;
               case 768:
                  status.setScore(2, parseNumber(s));    // player 2 score
                  break;
               default:
                  System.out.println("Undetected: " + t);
                  break;
            }
         } else if (newState == Engine.STATE_HIGHSCORE && x == 400 & y == 228) {
            status.setInitials(t.getText());
         } else if (!(newState == Engine.STATE_WAIT_START && x == 380 & y <= 668)) {
            System.out.println("Undetected: " + t);
         }
      }
      if (newState == Engine.STATE_UNKNOWN) {
         if (state == Engine.STATE_WAIT_PLAY) {
            state = Engine.STATE_PLAYING;
         }
      } else {
         state = newState;
      }
   }
   
   private int parseNumber(String s) {
      int number = 0;
      for (int i = 0; i < s.length(); ++i) {
         int digit = "O123456789".indexOf(s.charAt(i));
         if (digit < 0) {
            return -1;
         }
         number = 10 * number + digit;
      }
      return number;
   }

   private static char getCharByAddr(int addr) {
      switch (addr) {
         case 0xa78:
            return 'A';
         case 0xa80:
            return 'B';
         case 0xa8d:
            return 'C';
         case 0xa93:
            return 'D';
         case 0xa9b:
            return 'E';
         case 0xaa3:
            return 'F';
         case 0xaaa:
            return 'G';
         case 0xab3:
            return 'H';
         case 0xaba:
            return 'I';
         case 0xac1:
            return 'J';
         case 0xac7:
            return 'K';
         case 0xacd:
            return 'L';
         case 0xad2:
            return 'M';
         case 0xad8:
            return 'N';
         case 0xadd:
            return 'O';
         case 0xae3:
            return 'P';
         case 0xaea:
            return 'Q';
         case 0xaf3:
            return 'R';
         case 0xafb:
            return 'S';
         case 0xb02:
            return 'T';
         case 0xb08:
            return 'U';
         case 0xb0e:
            return 'V';
         case 0xb13:
            return 'W';
         case 0xb1a:
            return 'X';
         case 0xb1f:
            return 'Y';
         case 0xb26:
            return 'Z';
         case 0xb2c:
            return ' ';
         case 0xb2e:
            return '1';
         case 0xb32:
            return '2';
         case 0xb3a:
            return '3';
         case 0xb41:
            return '4';
         case 0xb48:
            return '5';
         case 0xb4f:
            return '6';
         case 0xb56:
            return '7';
         case 0xb5b:
            return '8';
         case 0xb63:
            return '9';
         default:
            return '?';
      }
   }
}
