/**
 * File:    SimpleMath.java
 * Package: de.heise.asteroid.util
 * Created: 22.06.2008 19:13:07
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.util;

/**
 * @author Chr. Moellenberg
 *
 */

public class SimpleMath {
   public static final double PI2 = Math.PI * 2.0;
   public static final double PI_2 = Math.PI * 0.5;
   public static final double PI_180 = Math.PI / 180.0;
   public static final int[] astCos = new int[256];
   public static final int[] astSin = new int[256];
   public static final double[] astAlpha = new double[256];
   static {
      System.out.print("Building trigonometric tables... ");
      astSin[64] = astCos[0] = 127;
      astSin[192] = astCos[128] = -127;
      astSin[0] = astSin[128] = astCos[64] = astCos[192] = 0;
      for (int a = 1; a < 64; ++a) {
         astCos[256 - a] = astCos[a] = (int)Math.round(127.0 * Math.cos((double)a / 128.0 * Math.PI));
         astCos[128 - a] = astCos[128 + a] = -astCos[a];
         astSin[128 - a] = astSin[a] = (int)Math.round(127.0 * Math.sin((double)a / 128.0 * Math.PI));
         astSin[128 + a] = astSin[256 - a] = -astSin[a];
      }
      for (int a = 0; a < 256; ++a) {
         astAlpha[a] = Math.atan2((double)astSin[a], (double)astCos[a]);
         if (astAlpha[a] < 0) {
            astAlpha[a] += PI2;
         }
      }
      System.out.println("done.");
//      dumpTables();
//      plausiCheck();
   }
   
   private static void plausiCheck() {
      double epsilon = 0.04;
      for (int a = 1; a < 128; ++a) {
         if (astCos[a] != astCos[256 - a]) {
            System.out.printf("astCos is asymmetric a=+/-%d\n", a);
         }
         if (astSin[64 + a] != astSin[(64 - a) & 0xff]) {
            System.out.printf("astSin is asymmetric a=64+/-%d\n", a);
         }
      }
      double maxDeltaSin = 0.0;
      double maxDeltaCos = 0.0;
      double maxDeltaAlpha = 0.0;
      double alpha = 0.0;
      for (int a = 1; a < 256; ++a) {
         if (astAlpha[a] <= alpha) {
            System.out.printf("astAlpha is not monotonous: alpha[%d]=%f, alpha[%d]=%f\n", a, astAlpha[a], a - 1, alpha);
         }
         alpha = astAlpha[a];
         double realAlpha = (double)a / 128.0 * Math.PI;
         double deltaAlpha = realAlpha - alpha;
         if (deltaAlpha * deltaAlpha > maxDeltaAlpha * maxDeltaAlpha) {
            maxDeltaAlpha = deltaAlpha;
         }
         double deltaSin = astSin[a] == 0 ? 0 : (1.0 - 127.0 * Math.sin(realAlpha) / astSin[a]);
         double deltaCos = astCos[a] == 0 ? 0 : (1.0 - 127.0 * Math.cos(realAlpha) / astCos[a]);
         if (deltaSin * deltaSin > maxDeltaSin * maxDeltaSin) {
            maxDeltaSin = deltaSin;
         }
         if (deltaCos * deltaCos > maxDeltaCos * maxDeltaCos) {
            maxDeltaCos = deltaCos;
         }
         if (deltaSin * deltaSin > epsilon * epsilon) {
            System.out.printf("astSin exceeds allowed error at a=%d, delta=%f\n", a, deltaSin);
         }
         if (deltaSin * deltaSin > epsilon * epsilon) {
            System.out.printf("astCos exceeds allowed error at a=%d, delta=%f\n", a, deltaSin);
         }
      }
      System.out.printf("Max. alpha deviation is %f deg\n", maxDeltaAlpha / PI_180);
      System.out.printf("Max. sin() deviation is %.2f %%\n", maxDeltaSin * 100.0);
      System.out.printf("Max. cos() deviation is %.2f %%\n", maxDeltaCos * 100.0);
   }

   private static void dumpTables() {
      System.out.println("Angle astSin astCos astAlpha realAlpha");
      for (int a = 0; a < 256; ++a) {
         System.out.printf("%3d:   %4d   %4d   %6.2f    %6.2f\n", a, astSin[a], astCos[a], astAlpha[a] / PI_180, (double)a / 32.0 * 45.0);
      }
   }

   public static int normalize(int p, int min, int max) {
      while (p < min) {
         p += max - min;
      }
      while (p >= max) {
         p -= max - min;
      }
      return p;
   }
}
