require 'java'

include_class 'java.awt.event.WindowListener'
include_class 'javax.swing.WindowConstants'
include_class 'javax.swing.JFrame'
include_class 'javax.swing.JPanel'
include_class 'javax.swing.JLabel'
include_class 'java.awt.Color'
include_class 'java.awt.Polygon'

OBJECT_SIZES = {
  :small_asteroid  => 10,
  :medium_asteroid => 20,
  :large_asteroid  => 40,
  :small_saucer    => 8,
  :large_saucer    => 20
}

VELOCITY_SCALE = 20
SHOT_SCALE = 68

BLACK = Color.new(0,0,0)
GREY  = Color.new(192,192,192)
WHITE = Color.new(255,255,255)
RED   = Color.new(255,0,0)
GREEN = Color.new(0,255,0)

class DrawingPanel < JPanel
  def paintComponent(g)
    setBackground(WHITE)
    g.clearRect(0,0,self.width,self.height)

    @draw_objects.each do |obj|
      size = OBJECT_SIZES[obj[:class]] || 1
      if obj[:blacklisted]
        g.setColor(GREY)
      elsif obj[:current_target]
        g.setColor(RED)
      else
        g.setColor(BLACK)
      end

      case obj[:kind]

      when :asteroid
        x,y = coords(obj[:nvx]-size, obj[:nvy]+size)
        if obj[:lock_on]
          g.setColor(GREEN) if obj[:valid_target]
          g.fillOval(x, y, size, size)
        else
          g.drawOval(x, y, size, size)
        end

      when :saucer
        x,y = coords(obj[:nvx]-size, obj[:nvy]+size)
        if obj[:lock_on]
          g.setColor(GREEN) if obj[:valid_target]
          g.fillRect(x, y, size, size)
        else
          g.drawRect(x, y, size, size)
        end

      when :ship
        dx,dy = coords(obj[:nvx]+obj[:initial_dx], obj[:nvy]+obj[:initial_dy])
        right_x,right_y = coords(obj[:nvx]-obj[:initial_dy]/2.0-obj[:initial_dx]/2.0, obj[:nvy]+obj[:initial_dx]/2.0-obj[:initial_dy]/2.0)
        left_x,left_y = coords(obj[:nvx]+obj[:initial_dy]/2.0-obj[:initial_dx]/2.0, obj[:nvy]-obj[:initial_dx]/2.0-obj[:initial_dy]/2.0)
        mx,my = coords(obj[:nvx]+obj[:initial_dx]+obj[:mx]*SHOT_SCALE, obj[:nvy]+obj[:initial_dy]+obj[:my]*SHOT_SCALE)
        # ship
        p = Polygon.new
        p.addPoint(dx,dy)
        p.addPoint(right_x,right_y)
        p.addPoint(left_x,left_y)
        g.drawPolygon(p)
        # shot vector
        g.setColor(RED)
        g.drawLine(dx,dy, mx,my)

      when :shot
        g.setColor(RED) if obj[:tag].to_s =~ /^u/ # ufo shot
        x,y = coords(obj[:nvx], obj[:nvy])
        g.fillOval(x, y, 2, 2)

      when :velocity
        if obj[:mx] && obj[:my]
          g.setColor(GREEN)
          x,y = coords(obj[:nvx], obj[:nvy])
          g.drawLine(x, y, *coords(obj[:nvx]+obj[:mx]*VELOCITY_SCALE, obj[:nvy]+obj[:my]*VELOCITY_SCALE))
        end

      end
    end
  end
  
  def <<(o)
    @draw_objects << o
  end
  
  def clear
    @draw_objects = []
  end
  
  def initialize
    @draw_objects = []
  end

protected

  def coords(game_x, game_y)
    screen_x = (game_x+512) / 2
    screen_y = 512 - (game_y+512) / 2
    return screen_x, screen_y
  end
  
end

class Visualizer < JFrame
  include WindowListener
  
  def draw_objects # is being called as observer when ringbuffer changes
    return unless @drawing_panel

    @drawing_panel.clear
    game = @ringbuffer.get_last
    state = @player_state.get

    return unless game[:ship_present] || !game[:enabled]
    
    game[:asteroids].each do |a|
      @drawing_panel << { :kind => :asteroid, :current_target => (state[:current_target] == a[:tag]), :valid_target => state[:valid_targets].include?(a[:tag]), :blacklisted => state[:blacklist].include?(a[:tag]) }.merge!(a)
      @drawing_panel << { :kind => :velocity }.merge!(a)
    end
    
    game[:shots].each do |s|
      @drawing_panel << { :kind => :shot }.merge!(s)
      @drawing_panel << { :kind => :velocity }.merge!(s)
    end
    
    if game[:saucer_present]
      @drawing_panel << { :kind => :saucer, :current_target => (state[:current_target] == game[:saucer][:tag]), :valid_target => state[:valid_targets].include?(game[:saucer][:tag]), :blacklisted => state[:blacklist].include?(game[:saucer][:tag]) }.merge!(game[:saucer])
      @drawing_panel << { :kind => :velocity }.merge!(game[:saucer])
    end

    if game[:ship_present]
      @drawing_panel << { :kind => :ship }.merge!(game[:ship]).merge!(Vector::SHOT_ANGLES_BY_INDEX[game[:shot_angle_index]]) 
      @drawing_panel << { :kind => :velocity }.merge!(game[:ship])
    end

  rescue # show exceptions without terminating visualization altogether
    puts $!.message if $debug
    puts $!.backtrace.join("\n") if $debug
  ensure
#    @message.text = "t=#{'%d' % (game[:duration]||0)}s, Cal #{game[:angle_calibrated] ? '+' : '-'}, Score: #{game[:score]}, Score/s: #{'%.0f' % (game[:score] / game[:duration]) rescue 0}, Lives: #{game[:lives]}, Strategy: #{state[:strategy] || :idle}, Lat. #{game[:latency]}#{'*' if game[:lost_frames] > 0}" if game
    @message.text = "[#{game[:sequence]||0}], Cal #{game[:angle_calibrated] ? '+' : '-'}, Score: #{game[:score]}, Score/s: #{'%.0f' % (game[:score] / game[:duration]) rescue 0}, Shots: #{game[:fired_shot_count]}, Strategy: #{state[:strategy] || :idle}, Lat. #{game[:latency]}#{'*' if game[:lost_frames] > 0}" if game
    @drawing_panel.repaint if @drawing_panel
  end
  
  # terminate given thread on windowClosing event
  def windowClosing(e)
    @main_thread.terminate
  end

  # also define ignored events to adhere to the WindowListener interface
  def windowOpened(e); end
  def windowClosed(e); end
  def windowIconified(e); end
  def windowDeiconified(e); end
  def windowActivated(e); end
  def windowDeactivated(e); end
  
  def initialize(ringbuffer, player_state, main_thread)
    super "Asteroids Visualization"

    @ringbuffer = ringbuffer
    @ringbuffer.register_observer(self, :draw_objects)
    @player_state = player_state
    @main_thread = main_thread

    contentPane.layout = nil
    setDefaultCloseOperation(DISPOSE_ON_CLOSE)  
    setSize(512, 532)

    @message = JLabel.new
    contentPane.add(@message)
    @message.setBounds(0,0,512,20)
    @drawing_panel = DrawingPanel.new
    @drawing_panel.setBounds(0,20,512,512)
    contentPane.add(@drawing_panel)

    addWindowListener(self)
    setVisible(true)
  end

end
