#include "inhabitants.h"
#include <cassert>
#include <cmath>
#include <iostream>
using namespace std;

Inhabitant::Inhabitant(const short& xPos, const short& yPos, const short& sz)
 : curr(0)
 , xSpeed(0)
 , ySpeed(0)
 , age(0)
 , uptodate(true)
{
	assert(xPos >= 0 && xPos <= xRes);
	assert(yPos >= 0 && yPos <= yRes);
	x[0] = xPos;
	y[0] = yPos;
	// initialization is always a good idea even if -1 is never checked for
	for(int i = 1; i < history; i++) {
		x[i] = -1;
		y[i] = -1;
	}
	size = sz;
}

Inhabitant::~Inhabitant()
{
}

void Inhabitant::position(short* xPos, short* yPos, short* sz) const
{
	assert(xPos != 0);
	assert(yPos != 0);
	*xPos = x[curr];
	*yPos = y[curr];
	if(sz != 0)
		*sz = size;
}

void Inhabitant::speed(float* xSp, float* ySp) const
{
	assert(xSp != 0);
	assert(ySp != 0);
	*xSp = xSpeed;
	*ySp = ySpeed;
}

const bool Inhabitant::upToDate() const
{
	return uptodate;
}

void Inhabitant::outOfDate()
{
	uptodate = false;
}

/**
 * @returns WillHit if the inhabitant collides with the thing in the next futureFrames,
 *	    MissLeft if the thing and inhabitant approach but the thing passes left of
 *	    the inhabitant, MissRight in any other case
 */
Inhabitant::Destiny Inhabitant::willHit(const Inhabitant& thing, int* numFrames, float* smallestDistance, const bool sniper) const
{
	short thingPosX = 0, thingPosY = 0, thingSize = 0;
	float thingSpeedX = 0, thingSpeedY = 0;
	thing.position(&thingPosX, &thingPosY, &thingSize);
	thing.speed(&thingSpeedX, &thingSpeedY);
	short dirX = 0, dirY = 0;
	int distX = 0, distY = 0;
	float nearestApproach = distance(x[curr], y[curr], thingPosX, thingPosY);
	if(smallestDistance != 0)
		*smallestDistance = nearestApproach;
	if(numFrames != 0)
		*numFrames = 0;
	if(sniper)
		while(thingSize > 8)
			thingSize /= 2;
	for(int i = 1;; i++) {
		if(numFrames != 0)
			(*numFrames)++;
		short futureX = compensateScreenWrap(x[curr] + i * xSpeed, 1024);
		short futureY = compensateScreenWrap(y[curr] + i * ySpeed, 768);
		short futureThingX = compensateScreenWrap(thingPosX + i * thingSpeedX, 1024);
		short futureThingY = compensateScreenWrap(thingPosY + i * thingSpeedY, 768);
		float newDistance = distance(futureX, futureY, futureThingX, futureThingY);
		if(newDistance < size + thingSize)
			return WillHit;
		if(newDistance < nearestApproach) {
			nearestApproach = newDistance;
			if(smallestDistance != 0)
				*smallestDistance = nearestApproach;
		} else { // inhabitant and thing missed each other,
			dirX = (x[curr] + history * xSpeed) - x[curr];
			dirY = (y[curr] + history * ySpeed) - y[curr];
			distX = compensateScreenWrap(futureX - futureThingX, 1024, 512);
			distY = compensateScreenWrap(futureY - futureThingY, 768, 384);
			break;
		}
	}
	// inhabitant and thing missed each other,
	// we want to know on what side
	// Vektor, Kreuzprodukt
	if(dirX * distY - dirY * distX > 0) {
		return MissLeft;
	}
	return MissRight;
}

float Inhabitant::distance(const short& x1, const short& y1, const short& x2, const short& y2) const
{
	assert(x1 <= xRes && y1 <= yRes);
	assert(x2 <= xRes && y2 <= yRes);
	int distX = compensateScreenWrap(x1 - x2, 1024, 512);
	int distY = compensateScreenWrap(y1 - y2, 768, 384);
	return sqrt(distX * distX + distY * distY);
}

short Inhabitant::compensateScreenWrap(const short& dist, const short& bounds, const short& tolerance) const
{
	assert(bounds > 0);
	assert(tolerance > 0);
	assert(dist <= bounds);
	if(dist < -bounds + tolerance) // exit right or top
		return dist + bounds;
	else if(dist > bounds - tolerance) // exit left or bottom
		return dist - bounds;
	return dist;
}

short Inhabitant::compensateScreenWrap(const short& pos, const short& bounds) const
{
	short result = pos;
	if(pos > bounds)
		do {
			result -= bounds;
		} while(result > bounds);
	else if(pos < 0)
		do {
			result += bounds;
		} while(result < 0);
	return result;
}

void Inhabitant::speedMonitoring(const short& xPos, const short& yPos, float* xSp, float* ySp) const
{
	assert(xPos >= 0 && xPos <= xRes);
	assert(yPos >= 0 && yPos <= yRes);
	assert(xSp != 0);
	assert(ySp != 0);
	if(age < history) {
		/* Can't use xRes and yRes here:
		   "undefined reference to `Inhabitant::xRes'"
		   with "g++ (GCC) 4.2.3 (Debian 4.2.3-4)" */
		*xSp = compensateScreenWrap(xPos - x[oldestEntry()], 1024, (age + 1) * jumpiness) / (1.0 * age + 1);
		*ySp = compensateScreenWrap(yPos - y[oldestEntry()], 768, (age + 1) * jumpiness) / (1.0 * age + 1);
	} else {
		*xSp = 0;
		*ySp = 0;
	}
}

/**
 * Sets curr to the next position in the x and y array and stores the values there
 * @param xPos new x position to store
 * @param yPos new y position to store
 * @param xSp new speed in x direction
 * @param ySp new speed in y direction
 *
 * If something gets stored, the inhabitant is up to date.
 * @see upToDate()
 */
void Inhabitant::store(const short& xPos, const short& yPos, const float& xSp, const float& ySp, const short& sz)
{
	assert(xPos >= 0 && xPos <= xRes);
	assert(yPos >= 0 && yPos <= yRes);
	if(++curr == history)
		curr = 0;
	x[curr] = xPos;
	y[curr] = yPos;
	xSpeed = xSp;
	ySpeed = ySp;
	size = sz;
	uptodate = true;
	if(age + 1 < history)
		age++;
	else if(age == history)
		age = 0;
}

const unsigned char Inhabitant::oldestEntry() const
{
	assert(age < history);
	return age <= curr ?
			curr - age :
			curr + history - age;
}

void Inhabitant::invalidate()
{
	curr = 0;
	/* age is normally one lower, so this is used
	   to signal that no useful data is available */
	age = history;
	xSpeed = 0;
	ySpeed = 0;
	uptodate = false;
}

Living::Living(const short& xPos, const short& yPos, const short& sz)
 : Inhabitant(xPos, yPos, sz)
{
}

Living::~Living()
{
}

/**
 * Dead::couldBeAt() without checks
 */
void Living::set(const short& xPos, const short& yPos, const short& sz)
{
	assert(xPos >= 0 && xPos <= xRes);
	assert(yPos >= 0 && yPos <= yRes);
	float xSp = 0;
	float ySp = 0;
	speedMonitoring(xPos, yPos, &xSp, &ySp);
	store(xPos, yPos, xSp, ySp, sz);
}


Dead::Dead(const short& xPos, const short& yPos, const short& sz)
 : Inhabitant(xPos, yPos, sz)
{
}

Dead::~Dead()
{
}

/**
 * If the moving object could be at the given coordinates, store them and return true
 * @param xPos The new x coordinate
 * @param yPos The new y coordinate
 *
 * The speed in x and y direction is computed from the oldest valid value.
 * This increases the precision.
 * @return true if the object could be at the given screen position, false otherwise
 */
bool Dead::couldBeAt(const short& xPos, const short& yPos, const short& sz)
{
	assert(xPos >= 0 && xPos <= xRes);
	assert(yPos >= 0 && yPos <= yRes);
	assert(size > 0);
	if(sz != size)
		return false;
	/* Can't use xRes and yRes here:
		"undefined reference to `Inhabitant::xRes'"
		with "g++ (GCC) 4.2.3 (Debian 4.2.3-4)" */
	short xDistance = compensateScreenWrap(xPos - x[curr], 1024, jumpiness);
	short yDistance = compensateScreenWrap(yPos - y[curr], 768, jumpiness);
	if (xDistance < jumpiness && xDistance > -jumpiness
	 && yDistance < jumpiness && yDistance > -jumpiness) {
		// the new position is near the last known one
		float xSp = 0;
		float ySp = 0;
		speedMonitoring(xPos, yPos, &xSp, &ySp);
		// need positive values for comparison
		if ((xSpeed == 0 && ySpeed == 0)
		 || (xSp * xSp - xSpeed * xSpeed < jumpiness * jumpiness
		 &&  xSp * xSp - xSpeed * xSpeed > -jumpiness * jumpiness
		 &&  xSp * xSpeed >= 0 // same direction
		 &&  ySp * ySp - ySpeed * ySpeed < jumpiness * jumpiness
		 &&  ySp * ySp - ySpeed * ySpeed > -jumpiness * jumpiness
		 &&  ySp * ySpeed >= 0)) {
			// The speed matches, too. This is likely to be a known space object.
			store(xPos, yPos, xSp, ySp, size);
			return true;
		}
	}
	return false;
}

Shot::Shot(const short& xPos, const short& yPos)
 : Dead(xPos, yPos, 1)
{
}

Shot::~Shot()
{
}

bool Shot::couldBeAt(const short& xPos, const short& yPos)
{
	return Dead::couldBeAt(xPos, yPos, 1);
}

Saucer& Saucer::instance()
{
	static Saucer s;
	return s;
}

Saucer::Saucer()
 : Living(0, 0, 0)
{
	invalidate();
}

Saucer::~Saucer()
{
}

void Saucer::set(const short& xPos, const short& yPos, const short& sz)
{
	Living::set(xPos, yPos, sz);
}

Ship& Ship::instance()
{
	static Ship s;
	return s;
}

Ship::Ship()
 : Living(0, 0, shipSize)
 , xDirection(0)
 , yDirection(0)
{
	invalidate();
}

Ship::~Ship()
{
}

void Ship::set(const short& xPos, const short& yPos, const short& xDir, const short& yDir)
{
	assert(xDirection <= 1536);
	assert(yDirection <= 1536);
	Living::set(xPos, yPos, shipSize);
	xDirection = xDir;
	yDirection = yDir;
}

Ship::Action Ship::aim(const Inhabitant& thing)
{
	/* xDirection negative: Nose to the left
	   yDirection negative: Nose to the bottom
	   max xDirection and yDirection 1536
	   avaerage shot speed about 8.00
	   ship speed adds to shot speed
	*/
	/* create a fake shot that starts behind the ship and goes
	   to the direction a real shot would. Simulate movement to get the speed.*/
	Shot s(compensateScreenWrap(x[curr] - history * (xSpeed + 8.0 * (xDirection / 1536.0)), 1024)
	     , compensateScreenWrap(y[curr] - history * (ySpeed + 8.0 * (yDirection / 1536.0)), 768));
	for(int i = history - 1; i > -2; i--)
		// couldBeAt must always succeed here, not interested in return value
		s.couldBeAt(compensateScreenWrap(x[curr] - i * (xSpeed + 8.0 * (xDirection / 1536.0)), 1024)
			  , compensateScreenWrap(y[curr] - i * (ySpeed + 8.0 * (yDirection / 1536.0)), 768));
	Destiny d = s.willHit(thing, 0, 0, true);
	if(d == WillHit)
		return Shoot;
	else if(d == MissLeft)
		return TurnRight;
	return TurnLeft;
}

bool Ship::tooFarAwayFrom(const Inhabitant& thing)
{
	short thingX = 0;
	short thingY = 0;
	short thingSz = 0;
	thing.position(&thingX, &thingY, &thingSz);
	if(distance(x[curr], y[curr], thingX, thingY) > 200 && thingSz < 12)
		return true;
	/*if(distance(x[curr], y[curr], thingX, thingY) > 400 && thingSz < 25)
		return true;*/
	if(distance(x[curr], y[curr], thingX, thingY) > 400 && thingSz < 35)
		return true;
	return false;
}

bool Ship::thrust(bool slow)
{
	if(xSpeed * xSpeed + ySpeed * ySpeed < (slow ? 8 : 16))
		return true;
	return false;
}

Asteroid::Asteroid(const short& xPos, const short& yPos, const Asteroid::Form& f, const short& sz)
 : Dead(xPos, yPos, sz)
{
	form = f;
}

Asteroid::~Asteroid()
{
}

bool Asteroid::couldBeAt(const short& xPos, const short& yPos, const short& sz, const Asteroid::Form& f)
{
	if(f != form)
		return false;
	return Dead::couldBeAt(xPos, yPos, sz);
}

void Asteroid::position(short* xPos, short* yPos, Asteroid::Form* f, short* sz) const
{
	assert(f != 0);
	Inhabitant::position(xPos, yPos, sz);
	*f = form;
}

short Asteroid::distanceToShip()
{
	short shipX;
	short shipY;
	Ship::instance().position(&shipX, &shipY);
	return distance(x[curr], y[curr], shipX, shipY);
}
