#include "pilot.h"
#include <cassert>
#include <cmath>
#include <iostream>
#include <iomanip>
using namespace std;

Pilot& Pilot::instance()
{
	static Pilot p;
	return p;
}

Pilot::Pilot()
 : curr(0)
 , firedInPrevFrame(false)
 , firedInThisFrame(false)
{
}

Pilot::~Pilot()
{
}

void Pilot::updateSpace(const unsigned short* vector_ram)
{
	int dx, dy, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;
	Asteroid::Form aType;

	if ((vector_ram[0] & 0xff00) != 0xe000 && (vector_ram[0] & 0xff00) != 0xe200)
		return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

	int pc = 1;
	for (;;)
	{
		int op = vector_ram[pc] >> 12;
		switch (op)
		{
		case 0xa: // LABS
			vy = (vector_ram[pc] & 0x3ff) - 128; // y has an offset
			vx = vector_ram[pc+1] & 0x3ff;
			vs = vector_ram[pc+1] >> 12;
			break;
		case 0xb: // HALT
			return;
		case 0xc: // JSRL
			if(vector_ram[pc] == 0xc929) {
				Saucer::instance().set(vx, vy, vs == 15 ? 20 : 10);
				break;
			} else if(vector_ram[pc] == 0xc8f3)
				aType = Asteroid::Type1;
			else if(vector_ram[pc] == 0xc8ff)
				aType = Asteroid::Type2;
			else if(vector_ram[pc] == 0xc90d)
				aType = Asteroid::Type3;
			else if(vector_ram[pc] == 0xc91a)
				aType = Asteroid::Type4;
			else // should never happen
				break;
			asteroidPos(vx, vy, aType, vs == 0 ? 40 : vs == 15 ? 20 : 8);
			break;
		case 0xd: // RTSL
			return;
		case 0xe: // JMPL
			return;
		default:
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;
			vz = vector_ram[pc+1] >> 12;
			if (dx == 0 && dy == 0 && vz == 15) {
				shotPos(vx, vy);
			}
			if (op == 6 && vz == 12 && dx != 0 && dy != 0)
			{
				switch (shipdetect)
				{
				case 0:
					v1x = dx;
					v1y = dy;
					++shipdetect;
					break;
				case 1:
					++shipdetect;
					Ship::instance().set(vx, vy, v1x - dx, v1y - dy);
					break;
				}
			}
			else if (shipdetect == 1)
				shipdetect = 0;

			break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}
}

void Pilot::react(KeysPacket& keys)
{
	for(vector<Shot*>::iterator i = shot.begin(); i < shot.end(); i++) {
		if(!(*i)->upToDate()) {
			delete(*i);
			shot.erase(i);
		}
	}
	for(vector<Asteroid*>::iterator i = asteroid.begin(); i < asteroid.end(); i++) {
		if(!(*i)->upToDate()) {
			delete(*i);
			asteroid.erase(i);
		}
	}
	firedInThisFrame = false;
	keys.clear();   // alle Tasten loslassen
	if(Ship::instance().upToDate()) {
		if(nearestShotIsDangerous()) {
			keys.hyperspace(true);
			firedInPrevFrame = false;
		}
		else
			steer(keys, analyseSpace(keys));
	} else {
		Ship::instance().invalidate();
		firedInPrevFrame = false;
	}
	for(unsigned int i = 0; i < shot.size(); i++)
		shot[i]->outOfDate();
	Saucer::instance().outOfDate();
	Ship::instance().outOfDate();
	for(unsigned int i = 0; i < asteroid.size(); i++)
		asteroid[i]->outOfDate();
}

void Pilot::shotPos(int xPos, int yPos)
{	
	for(unsigned int i = 0; i < shot.size(); i++) {
		if(shot[i]->couldBeAt(xPos, yPos))
			return;
	}
	shot.push_back(new Shot(xPos, yPos));
}

void Pilot::asteroidPos(int xPos, int yPos, Asteroid::Form f, int sz)
{	
	for(unsigned int i = 0; i < asteroid.size(); i++) {
		if(asteroid[i]->couldBeAt(xPos, yPos, sz, f))
			return;
	}
	asteroid.push_back(new Asteroid(xPos, yPos, f, sz));
}

bool Pilot::nearestShotIsDangerous()
{
	for(unsigned int i = 0; i < shot.size(); i++) {
		float speedX = 0;
		float speedY = 0;
		shot[i]->speed(&speedX, &speedY);
		if (speedX * speedX + speedY * speedY > 60) { // "slow" shots are not dangerous
			int numFrames = 0;
			if(shot[i]->willHit(Ship::instance(), &numFrames) == Inhabitant::WillHit && numFrames < 3)
				return true;
		}
	}
	return false;
}

Inhabitant* Pilot::analyseSpace(KeysPacket& keys)
{
	Inhabitant* target = 0;
	Inhabitant* backup = 0;
	float nearestDist = 1024 + 768;
	short framesTillCollision = 1000;
	int numFrames = 0;
	for(unsigned int i = 0; i < asteroid.size(); i++) {
		short astX = 0, astY = 0, astSz = 0;
		Asteroid::Form astF = Asteroid::Type1;
		asteroid[i]->position(&astX, &astY, &astF, &astSz);
		float smallestDistance = 0;
		if(Ship::instance().aim(*asteroid[i]) == Ship::Shoot
		&& firedInPrevFrame == false
		&& alreadyShot(asteroid[i]) == false) {
			keys.fire(true); // Abstauber
			firedInPrevFrame = true;
			firedInThisFrame = true;
		}
		if(asteroid[i]->willHit(Ship::instance(), &numFrames, &smallestDistance) == Inhabitant::WillHit) {
			if(numFrames < 2) {
				keys.hyperspace(true);
				firedInPrevFrame = firedInThisFrame;
				return 0;
			}
			if(numFrames < framesTillCollision
			&& (alreadyShot(asteroid[i]) == false
			    || astSz > 8)) /* only stop shooting if the smallest asteroid is already shot) */ {
				framesTillCollision = numFrames;
				if(asteroid[i]->distanceToShip() < nearestDist)
					nearestDist = smallestDistance;
				if(Ship::instance().tooFarAwayFrom(*asteroid[i]))
					backup = asteroid[i];
				else
					target = asteroid[i];
			}
		} else if(framesTillCollision > 400
		&& asteroid[i]->distanceToShip() < nearestDist
		&& (alreadyShot(asteroid[i]) == false
		   || astSz > 8)) {
			nearestDist = smallestDistance;
			if(Ship::instance().tooFarAwayFrom(*asteroid[i]))
				backup = asteroid[i];
			else
				target = asteroid[i];
		}
	}
	if(Saucer::instance().upToDate()) {
		if(Ship::instance().aim(Saucer::instance()) == Ship::Shoot
		&& firedInPrevFrame == false
		&& alreadyShot(&Saucer::instance()) == false) {
			keys.fire(true); // Abstauber
			firedInThisFrame = true;
			firedInPrevFrame = true;
		}
		if(Saucer::instance().willHit(Ship::instance(), &numFrames) == Inhabitant::WillHit
		&& numFrames < 2) {
			keys.hyperspace(true);
			firedInPrevFrame = firedInThisFrame;
			return 0;
		}
		if(framesTillCollision > 400
		&& alreadyShot(&Saucer::instance()) == false) {
			target = &Saucer::instance();
		}
	} else
		Saucer::instance().invalidate();
	if(target == 0 && backup != 0)
		return backup;
	return target;
}

bool Pilot::alreadyShot(const Inhabitant* toCheck) const
{
	for(unsigned int i = 0; i < shot.size(); i++)
		if(shot[i]->willHit(*toCheck) == Inhabitant::WillHit)
			return true;
	return false;
}

void Pilot::steer(KeysPacket& keys, const Inhabitant* target)
{
	Asteroid centerOfGalaxy(512, 384, Asteroid::Type1, 8);
	if(target == 0)
		target = &centerOfGalaxy;
	if(Ship::instance().tooFarAwayFrom(*target))
		keys.thrust(Ship::instance().thrust((target == &centerOfGalaxy) ? true : false));
	switch(Ship::instance().aim(*target)) {
		case Ship::Shoot:
			if(!firedInPrevFrame && target != &centerOfGalaxy) {
				keys.fire(true);
				firedInPrevFrame = true;
				return;
			}
			break;
		case Ship::TurnRight:
			keys.right(true);
			break;
		default:
			keys.left(true);
			break;
	}
	firedInPrevFrame = firedInThisFrame;
}
