//
//  APIOChannel.m
//  APlayer
//
//  Created by Holger Sadewasser on 4/20/08.
//  Copyright 2008. All rights reserved.
//

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#import "parameter.h"
#import "APIOChannel.h"
#import "APTracker.h"


static void socketCallback(CFSocketRef iSocket,
                           CFSocketCallBackType iType,
                           CFDataRef iAddress,
                           const void *iDataPtr,
                           void *iUserInfoPtr) {
  
  switch( iType )
  {
    case kCFSocketDataCallBack:
      [(APIOChannel *)iUserInfoPtr dataReceived:(NSData *)iDataPtr];
      break;
    case kCFSocketWriteCallBack:
      [(APIOChannel *)iUserInfoPtr readyToSend];
      break;
  }
}


// methods that are used internally by this class, but should not be needed externally
// are defined here to keep them "private". Anyone who knows about them can still call
// them of course, but by putting the prototypes here, it emphasizes that they are only
// meant for internal use.
@interface APIOChannel(_private_methods)


@end


@implementation APIOChannel

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Initializers/Clean up
// -------------------------------------------------------------------------------------

- (id) initWithAddress:(const char *)iAddress port:(const in_port_t)iPort; {
  self = [super init];
  if (self != nil) {
    
    mDataCallbackObject = nil;
    mWriteCallbackObject = nil;
 
    mFlgDataToSend = NO;
    mFlgDataReceived = NO;
    
    APKeysPacket_t keysPacket = {{'c', 't', 'm', 'a', 'm', 'e'}, '@', 0};
    mData = [[NSMutableData dataWithBytes:(const void *)&keysPacket length:sizeof(keysPacket)] retain];
    mKeysPacketPtr = [mData mutableBytes];
    
    int sockHandle = socket(AF_INET, SOCK_DGRAM, 0);
    struct sockaddr_in sa;
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(inet_addr(iAddress));
    sa.sin_port = htons(iPort);
    bind(sockHandle, (struct sockaddr *)&sa, sizeof(sa));
    CFSocketContext socketContext = {0, self, NULL, NULL, NULL};
    mSocket = CFSocketCreateWithNative(NULL, sockHandle,
                                        kCFSocketDataCallBack + kCFSocketWriteCallBack, socketCallback, &socketContext); // CFSocketRef
    
    NSData * address = [NSData dataWithBytes:(const void *)&sa length:sizeof(sa)];
    CFSocketError error = CFSocketConnectToAddress(mSocket,(CFDataRef)address,0.0);
    if ( error != 0 )
      NSLog(@"Connect error %d\n", error);
    
    mSource = CFSocketCreateRunLoopSource(NULL, mSocket, 0); // CFRunLoopSourceRef
    CFRunLoopAddSource(CFRunLoopGetCurrent(), mSource, kCFRunLoopDefaultMode);
    
  }
  return self;
}


-(id)initWithPort:(const in_port_t)iPort; {
  return [self initWithAddress:"127.0.0.1" port:iPort];
}


-(void) dealloc {
  CFRunLoopRemoveSource(CFRunLoopGetCurrent(), mSource, kCFRunLoopDefaultMode);
  CFRelease(mSource);
  CFRelease(mSocket);
  [mData release];
  [mDataCallbackObject release];
  [mWriteCallbackObject release];
  [super dealloc];  
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Instance Methods
// -------------------------------------------------------------------------------------

-(void)setDataCallbackObject:(id <IOChannelDataCallback>)iObject {
  [iObject retain];
  [mDataCallbackObject release];
  mDataCallbackObject = iObject;
}


-(void)setWriteCallbackObject:(id <IOChannelWriteCallback>)iObject {
  [iObject retain];
  [mWriteCallbackObject release];
  mWriteCallbackObject = iObject; 
}


-(void)dataReceived:(NSData *)iData {
  if ( mDataCallbackObject != nil ) [mDataCallbackObject dataReceived:iData];
}


-(void)readyToSend {
  [self sendKeys:'@' timeToLive:-1];
}


-(CFSocketError)sendKeys:(uint8_t)iKeys timeToLive:(int)iTimeToLive {
  
  CFSocketError result;
  
  mKeysPacketPtr->keys = iKeys;
  mKeysPacketPtr->ping++;
  result = CFSocketSendData(mSocket,NULL,(CFDataRef)mData,0.0);
  
  if ( result == kCFSocketSuccess && mWriteCallbackObject != nil )
    [mWriteCallbackObject didSendKeys:mKeysPacketPtr->keys ping:mKeysPacketPtr->ping timeToLive:iTimeToLive];
  else
    NSLog(@"Error sending keys: %d\tkeys: %d\tping: %d", result, (int)mKeysPacketPtr->keys, (int)mKeysPacketPtr->ping);
  
  return result;
}


-(void)close {
  CFSocketInvalidate(mSocket);
  CFRunLoopRemoveSource(CFRunLoopGetCurrent(), mSource, kCFRunLoopDefaultMode);
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Private Methods
// -------------------------------------------------------------------------------------


@end
