//
//  APPlayer.m
//  APlayer
//
//  Created by Holger Sadewasser on 4/20/08.
//  Copyright 2008. All rights reserved.
//

#import <unistd.h>
#import "APPlayer.h"
#import "APTarget.h"
#import "APTargetList.h"
#import "APKeystroke.h"
#import "tools.h"
#import "parameter.h"


extern BOOL gFlgLogging;
extern BOOL gFlgLogShots;

//int gNumOfTargets = 0;


// methods that are used internally by this class, but should not be needed externally
// are defined here to keep them "private". Anyone who knows about them can still call
// them of course, but by putting the prototypes here, it emphasizes that they are only
// meant for internal use.
@interface APPlayer(_private_methods)

-(double)_distanceFromShip:(APShip *)iShip toNextObjectIn:(NSArray *)iObjects;
-(int)_rotationToWaitPosition;
-(NSMutableArray *)_targetsFromHits:(NSMutableArray *)xListOfHits collisions:(NSMutableArray *)xListOfCollisions;
-(NSMutableArray *)_optimizedTargetsFromObjects:(NSArray *)iObjects tagged:(int)iTagged withHits:(NSMutableArray *)iListOfHits frames:(int)iFramesOfLevel duration:(double *)xDuration;

@end


@implementation APPlayer

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Initializers/Clean up
// -------------------------------------------------------------------------------------

-(id)initWithIOChannel:(APIOChannel *)iIOChannel maxFrames:(unsigned int)iMaxFrames {
  self = [super init];
  if (self != nil) {
    mIOChannel = iIOChannel;
    [mIOChannel retain];
    [mIOChannel setDataCallbackObject:self];
    mTracker = [[APTracker alloc] initWithIOChannel:iIOChannel];
    mProphet = [[APProphet alloc] init];
    mKeySequence = [[APKeySequence alloc] init];
    mTargets = [[NSMutableArray alloc] initWithCapacity:40];
    mMaxFrames = iMaxFrames;
    mFlgSaucerConsidered = NO;
    mFlgWaiting = NO;
    mFlgSynchronized = NO;
  }
  return self;
}


-(void)dealloc {
  [mTargets release];
  [mKeySequence release];
  [mProphet release];
  [mTracker release];
  [mIOChannel setDataCallbackObject:nil];
  [mIOChannel release];
  [super dealloc];
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Instance Methods
// -------------------------------------------------------------------------------------

-(void)dataReceived:(NSData *)iPacket {
  
  static uint8_t last_keys = 0;
  static int level = 0;
  static int framesOfLevel = 0;
  static BOOL flgObjectsPresent = NO;
  static BOOL flgFirstCall = YES;
  static BOOL flgSynchronizing = NO;
//  int savedFrame = 0;
  uint8_t keys = 0;
  NSArray * objects = [mTracker interpretScreen:iPacket];
  NSMutableArray * listOfHits = nil;
  NSMutableArray * listOfCollisions = nil;
  NSMutableArray * listOfTargets = nil;
  APTarget * target;
  APKeystroke * keystroke;
  int repeat, totalRepeat;
  int size;
  APObjectTypes_t type;
  int count, i;
  int rotation;
  int timeToLive;
  BOOL flgSaucerFound;
  APObject * object;
//  NSMutableArray * optimizedTargets;
//  double duration;
  
  
  // Check if the game is over
  if ( [mTracker lostFrames] > 10 ) {
//    NSLog(@"player: snapshots: %d", [mTracker->mSnapshots count]);
//    if ( [mTracker->mSnapshots count] > 0 )
//      NSLog(@"player: objects: %d", [[mTracker->mSnapshots objectAtIndex:0] count]);
//    if ( [NSKeyedArchiver archiveRootObject:mTracker->mSnapshots toFile:@"/Users/holger/MAME/snapshots"] == YES )
//      NSLog(@"Snaptshots successfully archived");
//    else
//      NSLog(@"Snaptshots not archived");
    [mIOChannel close];
    return;
  }

  if ( flgFirstCall == YES && [mTracker shipPresent] == YES ) {
    [mTracker setSynchronize:YES];
    [mKeySequence addShotToTarget:nil repeat:1];
    [mKeySequence addKeys:cKeyNone repeat:9];
    flgFirstCall = NO;
    flgSynchronizing = YES;
  }

  if ( flgObjectsPresent == NO && [mTracker numberOfAsteroids] >= 4 ) {
    ++level;
    framesOfLevel = 0;
    flgObjectsPresent = YES;
    NSLog(@"level %d", level);
  } else if ( flgObjectsPresent == YES && [mTracker numberOfAsteroids] == 0 && [mTracker saucerPresent] == NO ) {
    flgObjectsPresent = NO;
    NSLog(@"%d frames", framesOfLevel);
  }
  ++framesOfLevel;
  
  if ( [mTracker shipPresent] == NO && flgSynchronizing == NO ) {
    // When there is no ship we don't want to send any packets
    if ( last_keys != cKeyNone ) {
      // make sure we release all keys if there is no ship or no objects on the screen
      last_keys = cKeyNone;
      [mKeySequence clear];
      [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:2 target:nil]];
    }
  } else if ( [mTracker numberOfAsteroids] == 0 && [mTracker saucerPresent] == NO && flgSynchronizing == NO ) {
    if ( mFlgWaiting == NO ) {
      [mKeySequence clear];
      rotation = [self _rotationToWaitPosition];
      if ( rotation != 0 ) {
        if ( rotation >= 0 ) {
          keys = cKeyLeft;
          repeat = rotation;
        } else if ( rotation < 0 ) {
          keys = cKeyRight;
          repeat = -rotation;
        }
        [mKeySequence addKeys:keys repeat:repeat];
      }
      [mKeySequence addKeys:cKeyNone repeat:2];
      mFlgWaiting = YES;
    } else if ( mFlgSynchronized == NO && [mKeySequence count] == 0 ) {
      [mTracker setSynchronize:YES];
      [mKeySequence addShotToTarget:nil repeat:1];
      mFlgSynchronized = YES;
    }
  } else {
    
    if ( mFlgWaiting == YES ) {
      [mKeySequence clear];
      mFlgWaiting = NO;      
      mFlgSynchronized = NO;
    }
    
    // Check if we have to do a hyperspace jump
    if ( [self _distanceFromShip:[mTracker ship] toNextObjectIn:objects] <= 0.0 ) {
      
      [mKeySequence addHyperspace];
      
    } else {
//      if ( [mTracker numberOfAsteroids] == 1 ) {
//
//        if ( gFlgLogShots == NO ) {
//          [mKeySequence clear];
//          if ( [[mTracker ship] angleByte] < 128 ) {
//            keys = cKeyRight;
//            repeat = [[mTracker ship] angleByte];
//          } else {
//            keys = cKeyLeft;
//            repeat = 256 - [[mTracker ship] angleByte];
//          }
//          if ( repeat > 0 ) {
//            [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:keys repeat:repeat]];
//          }
//        }
//        if ( [mKeySequence count] == 0 && [mTracker numberOfOwnShots] == 0 ) {
//          [mKeySequence addShotToTarget:nil repeat:1];
//          [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:4]];
//          [mKeySequence addShotToTarget:nil repeat:1];
//          [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:4]];
//          [mKeySequence addShotToTarget:nil repeat:1];
//          [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:4]];
//          [mKeySequence addShotToTarget:nil repeat:1];
//          [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyLeft repeat:1]];
//          [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:1]];
//        }
//        gFlgLogShots = YES;
//        
//      } else {
      gFlgLogShots = NO;
      
      listOfCollisions = [mProphet collisionsFromTracker:mTracker frames:framesOfLevel];
      [listOfCollisions sortUsingSelector:@selector(compareDurationToTarget:)];
//      if ( [listOfCollisions count] > 0 )
//        NSLog(@"collisions %d dist %f", [listOfCollisions count], [[listOfCollisions objectAtIndex:0] durationOfShot]);
      if ( [listOfCollisions count] > 0 && [[listOfCollisions objectAtIndex:0] durationOfShot] <= 1.0 ) {
        [mKeySequence addHyperspace];
//        NSLog(@"hyperspace");
//        NSLog(@"type %d", [[[listOfCollisions objectAtIndex:0] object] type]);
      } else {
        
        if ( [mKeySequence count] > 0 && flgSynchronizing == NO ) {
          // Check if the current target still exists. If not remove all keystrokes and
          // select the next target
          keystroke = [mKeySequence currentKeys];
          target = [keystroke target];
          if ( target != nil ) {
            object = [target object];
            if ( [objects containsObject:object] == NO ) {
              [mKeySequence clear];
            } else {
              // If the target is the saucer check if it has changed its heading
              if ( [object type] == cSaucer ) {
                if ( [object headingX] != [target headingX] || [object headingY] != [target headingY] ) {
                  [mKeySequence clear];
                }
              }
            }
          }

        }
        if ( [mKeySequence count] == 0 ) {

          flgSynchronizing = NO;

          if ( listOfHits == nil )
            // Determine list of possible hits
            listOfHits = [mProphet hitsFromTracker:mTracker ship:nil frames:framesOfLevel maxOffset:5];
          
          if ( [listOfHits count] > 0 ) {

//            if ( [mTracker numberOfAsteroids] <= 4 ) {
//              optimizedTargets = [self _optimizedTargetsFromObjects:[mTracker objects] tagged:0 withHits:listOfHits frames:framesOfLevel duration:&duration];
//              NSLog(@"optimized targets: %d  duration: %f", [optimizedTargets count], duration);
//              if ( savedFrame == 0 )
//                savedFrame = framesOfLevel;
//            }
            
            // Determine list of actual targets from list of possible hits
            listOfTargets = [self _targetsFromHits:listOfHits collisions:listOfCollisions];
            
            totalRepeat = 0;
            count = [listOfTargets count] - 1;
            for ( i = 0; i < count; i++ ) {
              target = [listOfTargets objectAtIndex:i];
              
              if ( mFlgSaucerConsidered == NO && [[target object] type] == cSaucer )
                mFlgSaucerConsidered = YES;
              
              if ( target->mFrameOfShot > 0 ) {
                
                if      ( [target rotationDirection] > 0 ) keys |= cKeyLeft;
                else if ( [target rotationDirection] < 0 ) keys |= cKeyRight;
                
                keys &= ~cKeyFire;
                repeat = [target frameOfShot] - totalRepeat;
                if ( repeat > 0 )
                  [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:keys repeat:repeat target:target]];
              }
              keys |= cKeyFire;
              [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:keys repeat:1 target:target]];
              
              totalRepeat += repeat + 1;
            }
            
            if ( count >= 0 ) {
              target = [listOfTargets objectAtIndex:count];
              type = [[target object] type];
              size = [[target object] size];
              
              if ( mFlgSaucerConsidered == NO && type == cSaucer )
                mFlgSaucerConsidered = YES;
              
              repeat = [target frameOfShot] - totalRepeat;
              if ( repeat > 0 ) {
                
                if      ( [target rotationDirection] > 0 ) keys = cKeyLeft;
                else if ( [target rotationDirection] < 0 ) keys = cKeyRight;
                else keys = cKeyNone;
                
                [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:keys repeat:repeat target:target]];
              }
              repeat = [target waitingTime];
              if ( repeat > 0 ) {
//                NSLog(@"waiting %d frames", repeat);
                [mKeySequence addKeys:cKeyNone repeat:repeat target:target];
              }
              [mKeySequence addShotToTarget:target repeat:[[target object] numberOfShots]];
            }
          } else {
            if ( last_keys != cKeyNone ) {
              // make sure we release all keys if there is nothing else to do
              last_keys = cKeyNone;
              [mKeySequence addKeystroke:[APKeystroke keystrokeWithKeys:cKeyNone repeat:1 target:nil]];
            }
          }
        } else {
          if ( [mTracker saucerPresent] == YES && mFlgSaucerConsidered == NO ) {
            if ( listOfHits == nil )
              listOfHits = [mProphet hitsFromTracker:mTracker ship:nil frames:framesOfLevel maxOffset:5];
            count = [listOfHits count];
            flgSaucerFound = NO;
            for ( i = 0; i < count && flgSaucerFound == NO; i++ ) {
              target = [listOfHits objectAtIndex:i];
              if ( [[target object] type] == cSaucer ) {
                mFlgSaucerConsidered = [mKeySequence insertShotToTarget:target];
                flgSaucerFound = YES;
              }
            }
          }
        }
      }
      //      }
    }
  }
  
  if ( [mKeySequence count] > 0 ) {
    // The first object in the sequence is the next key stroke
    keystroke = [mKeySequence currentKeys];
    
    // Set the time to live if we shoot at a target
//    if ( (last_keys & cKeyFire) && !([keystroke keys] & cKeyFire) ) {
//      if ( last_target != nil && [[last_target object] timeToLive] < 0 )
//        [[last_target object] setTimeToLive:(int)([last_target durationOfShot] + 1.5)];
//    }
    
    keys = [keystroke keys];
    target = [keystroke target];
    timeToLive = (int)([target durationOfShot] + 0.5);
//    if ( (last_keys & cKeyFire) && target != nil && [mTracker numberOfOwnShots] < 4 ) {
//    if ( (last_keys & cKeyFire) && target != nil ) {
//      timeToLive = (int)([target durationOfShot] + 0.5);
    if ( (keys & cKeyFire) && target != nil && [mTracker numberOfOwnShots] < 4 ) {
      if ( [[target object] timeToLive] < 0 ) {
        [[target object] setTimeToLive:timeToLive + 2];
      }
    } else {
      timeToLive = -1;
    }    
    
    // Send the keys
//    last_keys &= ~cKeyFire;
    [mIOChannel sendKeys:keys timeToLive:timeToLive];
    last_keys = keys;
    
    // Decrease repeat counter and remove key stroke from sequence if this was the last stroke
    [mKeySequence sent];
    if ( [mKeySequence count] == 0 )
      mFlgSaucerConsidered = NO;
  }

//- Begin Records shot angles ---------------------
//  uint8_t keys = 0;
//  static int count = 0;
//  static int shotCount = 0;
//  static int state = 0;
//  
//  if ( ++count >= 5 ) {
//    if ( mTracker->mFlgShipPresent == YES ) {
////         ((APTracker *)mDataCallbackObject)->mFlgSaucerPresent == NO ) {
//      switch (state) {
//        case 0:
//          if ( mTracker->mNumShots < 4 ) {
//            keys |= 2;
//            state = 1;
//            shotCount = 0;
//          }
//          break;
//        case 1:
//          keys &= ~2;
//          if ( ++shotCount >= 10 ) {
//            keys |= 0x10;
//            state = 0;
//          }
//            break;
//      }
//      }
//  }
//  [mIOChannel sendKeys:keys];
//- End Records shot angles ---------------------    

}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Private Methods
// -------------------------------------------------------------------------------------

-(double)_distanceFromShip:(APShip *)iShip toNextObjectIn:(NSArray *)iObjects {
 
  APObject * object;
//  APObject * nextObject;
  int count = [iObjects count];
  int i;
  double distX, distY, dist, minDist;
  
  minDist = 1e10;
  for ( i = 0; i < count; ++i ) {
    object = [iObjects objectAtIndex:i];
    if ( [object type] != cShip && [object allowsHyperspace] == YES ) {
      distX = distXdouble( [object posX], [iShip posX] );
      distY = distYdouble( [object posY], [iShip posY] );      
      dist = distX*distX + distY*distY - object->mRadiusC2 - iShip->mRadius2;
      if ( dist < minDist ) {
        minDist = dist;
//        nextObject = object;
      }
    }
  }
//  if ( minDist <= 0.0 && nextObject != nil ) {
//    for ( i = 0; i < count; ++i ) {
//      object = [iObjects objectAtIndex:i];
//      NSLog(@"type: %d  x: %f  y: %f  frames: %d  hit: %d", [object type], [object posX], [object posY], [object frames], (object == nextObject));
//    }
//  }
  return minDist;
}

-(int)_rotationToWaitPosition {

  APShip * ship = [mTracker ship];
  double x, y, newAngle, curAngle, dAngle;
  int rotation;
  
//  NSLog(@"sx: %f  sy: %f", [ship posX], [ship posY]);
  x = [ship posX] - 524.0;
  y = [ship posY] - 524.0;
  if ( x == 0.0 && y == 0.0 )
    newAngle = M_PI_2;
  else
    newAngle = acos(x / sqrt(x*x + y*y));
  if ( y < 0.0 ) newAngle = 2.0*M_PI - newAngle;
  curAngle = (double)[ship angleByte] * 3.0 * M_PI / 128.0;
  while( curAngle >= 2.0*M_PI ) curAngle -= 2.0 * M_PI;
  dAngle = newAngle - curAngle;
  if ( dAngle > M_PI) {
    dAngle -= 2.0 * M_PI;
  } else if ( dAngle < -M_PI ) {
    dAngle += 2.0 * M_PI;
  }
  
  rotation = (int)(dAngle / (3.0 * M_PI / 128.0) + 0.5);
  
//  NSLog(@"x: %f  y: %f  na: %f  ca: %f  da: %f  rot: %d", x, y, newAngle, curAngle, dAngle, rotation);
  return rotation;
}

-(NSMutableArray *)_targetsFromHits:(NSMutableArray *)xListOfHits collisions:(NSMutableArray *)xListOfCollisions {

  APTarget * target;
//  APTarget * target2;
  APTarget * lastTarget;
  NSMutableArray * targets = [[NSMutableArray alloc] initWithCapacity:30];
  NSMutableArray * intermediates = [[NSMutableArray alloc] initWithCapacity:30];
  int count, i;
//  int count2, j;
  int numberOfShots;
  int shotsForObject;
  unsigned lastFrameOfShot;
  int previousFrame = -1;
//  int shotAvailable[4];
  
  numberOfShots = 4 - [mTracker numberOfOwnShots];
    
  // If there is the saucer among the hits we select it as main target
  lastTarget = nil;
  count = [xListOfHits count];
  for ( i = 0; i < count && lastTarget == nil; i++ ) {
    target = [xListOfHits objectAtIndex:i];
    if ( [[target object] type] == cSaucer ) {
      lastTarget = target;
    }
  }
  
  // If there is no saucer among the hits we select the asteroid that we can hit within the smallest number of frames
//  if ( lastTarget == nil ) {
//    [xListOfCollisions sortUsingSelector:@selector(compareDurationToTarget:)];
//    count = [xListOfCollisions count];
//    for ( i = 0; i < count && lastTarget == nil; i++ ) {
//      target = [xListOfCollisions objectAtIndex:i];
//      if ( [target durationOfShot] < MIN_DIST_OF_COLLISIONS && [[target object] type] != cShot && [[target object] type] != cShip ) {
//        
//        count2 = [xListOfHits count];
//        for ( j = 0; j < count2 && lastTarget == nil; j++ ) {
//          target2 = [xListOfHits objectAtIndex:j];
//          if ( [[target2 object] isEqual:[target object]] == YES ) {
//            lastTarget = target2;
//          }
//        }
//        
//      }
//    }
//  }
  
  // If there is no target yet we select the asteroid that we can hit within the smallest number of frames
  if ( lastTarget == nil ) {
    [xListOfHits sortUsingSelector:@selector(compareTotalDurationToTarget:)];
    lastTarget = [xListOfHits objectAtIndex:0];
  }

//  shotsForObject = [[target object] numberOfShots];
//  if ( shotsForObject > 1 ) {
//    --shotsForObject;
//  }
//  numberOfShots -= shotsForObject;

  //  [mTracker shotAvailability:shotAvailable];
//  i = [[target object] numberOfShots];
//  if ( i >= 0 && i <= 4 ) {
//    if ( shotAvailable[i-1] > [target frameOfShot] )
//      return [targets autorelease];
//  }
//  numberOfShots -= i;
  
  numberOfShots -= [[target object] numberOfShots];
  
  // Subsequent targets must have a distance of at least 2 frames
  lastFrameOfShot = [lastTarget frameOfShot] - 1;

  if ( numberOfShots > 0 ) {
    [xListOfHits sortUsingSelector:@selector(compareFramesToTarget:)];
    count = [xListOfHits count];
    for ( i = 0; i < count; i++ ) {
      target = [xListOfHits objectAtIndex:i];
      if ( [target rotationDirection] == [lastTarget rotationDirection] &&
           [target frameOfShot] < lastFrameOfShot &&
           [target waitingTime] == 0 ) {
        [intermediates addObject:target];
      }
    }
    count = [intermediates count];
//    if ( count > 0 )
//      NSLog(@"%d intermediates %d shots", count, numberOfShots - 1);
    if ( count > 1 )
      [intermediates sortUsingSelector:@selector(compareTotalDurationToTarget:)];
    for ( i = 0; i < count && numberOfShots > 0; i++ ) {
      target = [intermediates objectAtIndex:i];
      shotsForObject = [[target object] numberOfShots];
      if ( shotsForObject > 1 ) {
        shotsForObject;
      }
      if ( (previousFrame < 0 || [target frameOfShot] > previousFrame + 1) && shotsForObject <= numberOfShots ) {
        [targets addObject:target];
        previousFrame = [target frameOfShot];
        numberOfShots -= [[target object] numberOfShots];
      }
    }
  }
  [targets addObject:lastTarget];
  
  [intermediates release];
  
  return [targets autorelease];
}


-(NSMutableArray *)_optimizedTargetsFromObjects:(NSArray *)iObjects tagged:(int)iTagged withHits:(NSMutableArray *)iListOfHits frames:(int)iFramesOfLevel duration:(double *)xDuration {

  NSMutableArray * hits;
  NSMutableArray * tmpObjects;
  NSMutableArray * minTargets = nil;
  NSMutableArray * tmpTargets;
  APTarget * target;
  APTarget * minTarget = nil;
  APObject * object;
  NSEnumerator * enumerator;
  double minDuration = 1e10;
  double duration = 0.0;
  int deltaFrames;
  uint8_t angleByte;
  uint8_t tmpAngleByte;
  APShip * ship = [mTracker ship];
  
  angleByte = [ship angleByte];
  
  if ( iListOfHits == nil )
    hits = [mProphet hitsFromTracker:mTracker ship:nil frames:iFramesOfLevel maxOffset:5];
  else
    hits = iListOfHits;
  
  enumerator = [hits objectEnumerator];
  while( (target = [enumerator nextObject]) ) {
    object = [target object];
    [object tag];
    if ( [iObjects count] > iTagged ) {
      tmpObjects = [iObjects copy];
      
      // Move objects to the frame where we can shoot again
      deltaFrames = [target frameOfShot] + [object numberOfShots] * 2;
      [tmpObjects makeObjectsPerformSelector:@selector(moveByFrames:) withObject:[NSNumber numberWithInt:deltaFrames]];
      
      // Rotate the ship to the angle where we can shoot again
      tmpAngleByte = angleByte + [target rotationDirection] * [target frameOfShot];
      [ship setAngleByte:tmpAngleByte];
      
      tmpTargets = [self _optimizedTargetsFromObjects:tmpObjects tagged:iTagged+1 withHits:nil frames:iFramesOfLevel+deltaFrames duration:&duration];
    } else
      duration = 0.0;
    duration += [target totalDuration];
    
    if ( duration < minDuration ) {
      minDuration = duration;
      minTarget = target;
      minTargets = tmpTargets;
    }
    
    [object untag];
  }
  
  if ( minTarget != nil ) {
    if ( minTargets == nil ) {
      minTargets = [[[NSMutableArray alloc] initWithCapacity:10] autorelease];
    }
    [minTargets insertObject:minTarget atIndex:0];
    *xDuration = minDuration;
  } else {
    *xDuration = 0.0;
  }
  
  [ship setAngleByte:angleByte];
  
  return minTargets;
}


@end
