//
//  APShip.m
//  APlayer
//
//  Created by Holger Sadewasser on 4/26/08.
//  Copyright 2008. All rights reserved.
//

#import "parameter.h"
#import "APShip.h"
#import "angles.h"
#import "tools.h"


// methods that are used internally by this class, but should not be needed externally
// are defined here to keep them "private". Anyone who knows about them can still call
// them of course, but by putting the prototypes here, it emphasizes that they are only
// meant for internal use.
@interface APShip(_private_methods)


@end


@implementation APShip

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Class methods
// -------------------------------------------------------------------------------------

+(void)initialize
{
  static BOOL initialized = NO;
  int i;
  double angle;
  double sina, cosa;
  
  if (!initialized) {
    for ( i = 0; i < 256; i++ ) {
      gAngles[i].intXd = (double)gAngles[i].intX / 8.0;
      gAngles[i].intYd = (double)gAngles[i].intY / 8.0;
      gAngles[i].angle = acos((double)gAngles[i].intX / sqrt((double)(gAngles[i].intX*gAngles[i].intX + gAngles[i].intY*gAngles[i].intY)));
      angle = (double)i * 3.0 * M_PI / 128.0;
      sina = round(127.0*sin(angle));
      cosa = round(127.0*cos(angle));
      gAngles[i].startX = (floor(cosa/2.0) + floor(cosa/4.0))/8.0;
      gAngles[i].startY = (floor(sina/2.0) + floor(sina/4.0))/8.0;
    }
    initialized = YES;
  }
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Initializers/Clean up
// -------------------------------------------------------------------------------------

-(id)init {
  self = [super initWithType:cShip];
  return self;
}

-(id)initWithKeysBuffer:(uint8_t *)iKeysBuf {
  
  int count;
  uint8_t curIndex, prevIndex;
  unsigned long long key, keyPrev, keyCur, keyOld;
  NSMutableIndexSet * indexSet;
  NSNumber * keyNumber;
  
  self = [super initWithType:cShip];
  if (self != nil) {
    mViewX = 0;
    mViewY = 0;
    mAngleByte = 0;
    mFlgNeedsCalibration = NO;
    mKeysBuf = iKeysBuf;
    mAngles = gAngles;
    
    mAngleByteDict = [[NSMutableDictionary alloc] initWithCapacity:512];
    mAngleByteRangeDict = [[NSMutableDictionary alloc] initWithCapacity:256];
    
    curIndex = 0;
    prevIndex = 255;
    for (count = 0; count < 256; count++) {
      if ( mAngles[prevIndex].visX != mAngles[curIndex].visX || mAngles[prevIndex].visY != mAngles[curIndex].visY || mAngles[prevIndex].rotation != mAngles[curIndex].rotation) {
        keyPrev = (((unsigned)mAngles[prevIndex].visX & 0xffff) << 16) | ((unsigned)mAngles[prevIndex].visY & 0xfff) | (mAngles[prevIndex].rotation << 12);
        keyCur  = (((unsigned)mAngles[curIndex].visX & 0xffff) << 16) | ((unsigned)mAngles[curIndex].visY & 0xfff);
        key = (keyPrev << 32) | keyCur;
        
        [mAngleByteDict setObject:[NSNumber numberWithInt:mAngles[curIndex].angleByte] forKey:[NSNumber numberWithLongLong:key]];
        
        keyOld = key;
        keyPrev &= ~0xf000;
        keyCur  |= (mAngles[curIndex].rotation << 12);
        key = (keyCur << 32) | keyPrev;

        [mAngleByteDict setObject:[NSNumber numberWithInt:mAngles[prevIndex].angleByte] forKey:[NSNumber numberWithLongLong:key]];

//        NSLog(@"pa: %3d px: %5d  py: %5d  pr: %d ca: %3d cx: %5d  cy: %5d  cr: %d px: %10#x  py: %10#x  cx: %10#x  cy: %10#x k1 = %#qx k2 = %#qx", mAngles[prevIndex].angleByte, mAngles[prevIndex].visX, mAngles[prevIndex].visY, mAngles[prevIndex].rotation, mAngles[curIndex].angleByte, mAngles[curIndex].visX, mAngles[curIndex].visY, mAngles[curIndex].rotation, mAngles[prevIndex].visX, mAngles[prevIndex].visY, mAngles[curIndex].visX, mAngles[curIndex].visY, keyOld, key);
      }
      ++curIndex;
      ++prevIndex;
    }
    
    for (count = 0; count < 256; count++) {
      key = (((unsigned)mAngles[count].visX & 0xffff) << 16) | ((unsigned)mAngles[count].visY & 0xfff) | (mAngles[count].rotation << 12);
//      NSLog(@"cx: %d  cy: %d  cx: %#x  cy: %#x  r: %d  key = %#qx", mAngles[count].visX, mAngles[count].visY, mAngles[count].visX, mAngles[count].visY, mAngles[count].rotation, key);
      keyNumber = [NSNumber numberWithLongLong:key];
      indexSet = [mAngleByteRangeDict objectForKey:keyNumber];
      if ( indexSet == nil ) {
        [mAngleByteRangeDict setObject:[[[NSMutableIndexSet alloc] initWithIndex:mAngles[count].angleByte] autorelease] forKey:keyNumber];
      } else {
        [indexSet addIndex:mAngles[count].angleByte];
      }
//      NSLog(@"key: %#qx  angle byte: %d", key, mAngles[count].angleByte);
    }

//    NSEnumerator * keyEnum = [mAngleByteRangeDict keyEnumerator];
//    unsigned indexBuf[256];
//    NSRange range;
//    unsigned num;
//    int visX, visY, rot;
//    while ((keyNumber = [keyEnum nextObject])) {
//      indexSet = [mAngleByteRangeDict objectForKey:keyNumber];
//      if ( indexSet != nil ) {
//        range = NSMakeRange(0,256);
//        num = [indexSet getIndexes:indexBuf maxCount:256 inIndexRange:&range];
//        key = [keyNumber longLongValue];
//        visX = (short)(key >> 16);
//        visY = key & 0xfff;
//        if ( visY & 0x800 ) {
//          visY |= ~0xfff;
//        }
//        rot = (key >> 12) & 0xf;
//        switch (num) {
//          case 0:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d", key, visX, visY, rot); break;
//          case 1:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d", key, visX, visY, rot, indexBuf[0]); break;
//          case 2:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1]); break;
//          case 3:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1], indexBuf[2]); break;
//          case 4:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1], indexBuf[2], indexBuf[3]); break;
//          case 5:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d  %3d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1], indexBuf[2], indexBuf[3], indexBuf[4]); break;
//          case 6:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d  %3d  %3d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1], indexBuf[2], indexBuf[3], indexBuf[4], indexBuf[5]); break;
//          case 7:
//            NSLog(@"key: %018#qx  x: %5d  y: %5d  r: %d  %3d  %3d  %3d  %3d  %3d  %3d  %3d", key, visX, visY, rot, indexBuf[0], indexBuf[1], indexBuf[2], indexBuf[3], indexBuf[4], indexBuf[5], indexBuf[6]); break;
//          default:
//            NSLog(@"key: %018#qx  %d indexes", key, num); break;
//            break;
//        }
//      } else {
//        NSLog(@"key: %018#qx has no index set", [keyNumber longLongValue]);
//      }
//    }
  }
  return self;
}

-(id)initWithCoder:(NSCoder *)iCoder {
  
  if ( [iCoder allowsKeyedCoding] == YES) {
    self = [super initWithCoder:iCoder];
    
    mViewX = [iCoder decodeIntForKey:@"APviewX"];
    mViewY = [iCoder decodeIntForKey:@"APviewY"];
    mFlgNeedsCalibration = [iCoder decodeBoolForKey:@"APflgNeedsCalibration"];
    mFlgAngleReset = [iCoder decodeBoolForKey:@"APflgAngleReset"];
    mAngleByte = (uint8_t)[iCoder decodeIntForKey:@"APangleByte"];
    
    mAngles = gAngles;
    
    mKeysBuf = NULL;
    mAngleByteDict = nil;
    mAngleByteRangeDict = nil;
    
  } else
    [NSException raise:NSInvalidUnarchiveOperationException format:@"Only supports NSKeyedUnarchiver coders"];
  
  return self;
}

-(void)encodeWithCoder:(NSCoder *)iCoder {
  
  if ( [iCoder allowsKeyedCoding] == YES) {
    [super encodeWithCoder:iCoder];
    
    [iCoder encodeInt:mViewX forKey:@"APviewX"];
    [iCoder encodeInt:mViewY forKey:@"APviewY"];
    [iCoder encodeBool:mFlgNeedsCalibration forKey:@"APflgNeedsCalibration"];
    [iCoder encodeBool:mFlgAngleReset forKey:@"APflgAngleReset"];
    [iCoder encodeInt:(int)mAngleByte forKey:@"APangleByte"];

  } else
    [NSException raise:NSInvalidArchiveOperationException format:@"Only supports NSKeyedArchiver coders"];
}

-(void)dealloc {
  [mAngleByteDict release];
  [mAngleByteRangeDict release];
  [super dealloc];
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Instance Methods
// -------------------------------------------------------------------------------------

-(void)setPosX:(int)iPosX posY:(int)iPosY viewX:(int)iViewX viewY:(int)iViewY {
  
  mInitPosX = iPosX;
  mInitPosY = iPosY;
  mTempPosX = mInitPosX;
  mTempPosY = mInitPosY;
  mViewX = iViewX;
  mViewY = iViewY;
  mDeltaX[0] = iPosX;
  mDeltaY[0] = iPosY;

  [super _setPosX:iPosX posY:iPosY];
//  NSLog(@"Ship x: %d y: %d vx: %d vy: %d", iPosX, iPosY, iViewX, iViewY);

}


-(void)updateWithPosX:(int)iPosX posY:(int)iPosY viewX:(int)iViewX viewY:(int)iViewY frame:(uint8_t)iFrame frames:(uint8_t)iFrames ping:(uint8_t)iPing lastPing:(uint8_t)iLastPing {

//  double distX, distY;
//  double divX, divY;
  unsigned long long key, keyPrev, keyCur;
  NSNumber * value;
  uint8_t keys;
  NSIndexSet * indexSet;
  
  if ( mTotalFrames > 1 ) {

//    distX = (double)iPosX - mPosX;
//    if ( distX > 512 ) distX -= 1024;
//    else if ( distX < -512 ) distX += 1024;
//    
//    distY = (double)iPosY - mPosY;
//    if ( distY > 384 ) distY -= 768;
//    else if ( distY < -384 ) distY += 768;
//
//    divX = distX - mHeadingX;
//    divY = distY - mHeadingY;
//    
//    if ( divX*divX + divY*divY > TRACKING_HEADING_DIV ) {
//      mInitPosX = mPosX;
//      mInitPosY = mPosY;
//      mTempPosX = mInitPosX;
//      mTempPosY = mInitPosY;
//      mInitFrame = 1;
//      mTempFrame = mInitFrame;
//      mTotalFrames = mInitFrame;
//    }      
  }

//  double oldx = mPosX;
//  double oldy = mPosY;
//  int oldViewX = mViewX;
//  int oldViewY = mViewY;
  int oldAngleByte = mAngleByte;
  
  [super updateWithPosX:iPosX posY:iPosY frames:iFrames];

  value = nil;
  if ( iViewX != mViewX || iViewY != mViewY ) {

    keyPrev = (((unsigned)mViewX & 0xffff) << 16) | ((unsigned)mViewY & 0xfff) | (mAngles[mAngleByte].rotation << 12);
    keyCur  = (((unsigned)iViewX & 0xffff) << 16) | ((unsigned)iViewY & 0xfff);
    key = (keyPrev << 32) | keyCur;
    value = [mAngleByteDict objectForKey:[NSNumber numberWithLongLong:key]];
    if ( value != nil ) {
      mAngleByte = [value intValue];
      mFlgNeedsCalibration = NO;
      mFlgAngleReset = NO;
      
      keys = mKeysBuf[(uint8_t)(iPing-1)]; 
//      NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  synchronized angle byte %3d  vx: %5d  vy: %5d", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, (int)mAngleByte, iViewX, iViewY);

#ifdef LOG_ANGLE_BYTE
      NSLog(@"synchronized");
#endif
    }
    
    mViewX = iViewX;
    mViewY = iViewY;
  }
  
  if ( value == nil ) {
    // This works only with no latency
    if ( iLastPing != iPing )
      keys = mKeysBuf[(uint8_t)(iLastPing-1)]; 
    else
      keys = mKeysBuf[(uint8_t)(iPing-1)]; 
    if ( keys & 0x10 ) ++mAngleByte;
    if ( keys & 0x08 ) --mAngleByte;
    key = (((unsigned)iViewX & 0xffff) << 16) | ((unsigned)iViewY & 0xfff) | (mAngles[mAngleByte].rotation << 12);
    indexSet = [mAngleByteRangeDict objectForKey:[NSNumber numberWithLongLong:key]];
    if ( indexSet != nil ) {
      if ( [indexSet containsIndex:(unsigned int)mAngleByte] == NO ) {

#ifdef LOG_ANGLE_BYTE
        NSLog(@"out of sync: keys & 0x18: %#x  angle byte: %d  r: %d  ox: %d  oy: %d  vx: %d  vy: %d  key: %#qx", (keys & 0x18), mAngleByte, mAngles[mAngleByte].rotation, oldViewX, oldViewY, iViewX, iViewY, key);
#endif
        key &= ~0xf000;
        key |=  (mAngles[oldAngleByte].rotation << 12);
        indexSet = [mAngleByteRangeDict objectForKey:[NSNumber numberWithLongLong:key]];
        if ( indexSet != nil ) {
          if ( [indexSet containsIndex:(unsigned int)oldAngleByte] == NO ) {
            mAngleByte = [indexSet firstIndex];
            mFlgAngleReset = YES;

//            NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  first angle from set    %3d  vx: %5d  vy: %5d  key: %#qx", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, mAngleByte, mViewX, mViewY, key);

#ifdef LOG_ANGLE_BYTE
            NSLog(@"new angle byte: %d  key: %#qx", mAngleByte, key);
#endif
          } else {
            mAngleByte = oldAngleByte;
            mFlgAngleReset = YES;
//            NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  kept old angle byte     %3d  vx: %5d  vy: %5d", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, (int)oldAngleByte, mViewX, mViewY);

#ifdef LOG_ANGLE_BYTE
            NSLog(@"kept old angle byte");
#endif
          }
          mFlgNeedsCalibration = YES;
        } else {
          NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  no angle bytes           for vx: %5d  vy: %5d  r: %d  key: %#qx", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, iViewX, iViewY, mAngles[oldAngleByte].rotation, key);
        }
      } else {
        mFlgAngleReset = NO;
//        NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  new angle byte          %3d  vx: %5d  vy: %5d  key: %#qx", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, mAngleByte, mViewX, mViewY, key);
      }
    } else {
      NSLog(@"f: %3d df: %d lp: %3d p: %3d k: %4#x  no angle bytes           for vx: %5d  vy: %5d  r: %d  key: %#qx", (int)iFrame, (int)iFrames, (int)iLastPing, (int)iPing, keys, iViewX, iViewY, mAngles[mAngleByte].rotation, key);
    }
  }

  mKeysBuf[(uint8_t)(iPing-1)] = '@'; 

#ifdef LOG_ANGLE_BYTE
  if ( oldAngleByte != mAngleByte )
    NSLog(@"old angle byte: %d  new angle byte: %d", oldAngleByte, (int)mAngleByte);
#endif
//  if ( iPosX != oldx || iPosY != oldy )
//    NSLog(@"Ship x: %lf y: %lf hx: %lf hy: %lf vx: %lf vy: %lf f:%d\n", mPosX, mPosY, mHeadingX, mHeadingY, mViewX, mViewY, mTotalFrames);
}


-(APAngleStruct_t *)angles {
  return gAngles;
}


-(uint8_t)angleByte {
  return mAngleByte;
}


-(void)setAngleByte:(uint8_t)iAngleByte {
  mAngleByte = iAngleByte;
}


-(double)timeOfCollisionWithPosX:(double)a0 posY:(double)b0 dirX:(double)iDirX dirY:(double)iDirY radius:(double)iRadius headingStable:(BOOL)iFlgHeadingStable {
  
  double a = iDirX;
  double b = iDirY;
  
  double x0 = a0 + distXdouble( a0, mPosX );
  double y0 = b0 + distYdouble( b0, mPosY );
  
  //  double x0 = mPosX;
  //  double y0 = mPosY;
  
  double dx, dy;
  double q1, q2, n, r;
  double c, d, e, dist2;
  
  if ( a == 0.0 && b == 0.0 ) return -1.0;
  
  dx = a0 - x0;
  dy = b0 - y0;
  dist2 = dx*dx + dy*dy;
    
  // If the distance to the object is more than 100 pixel we calculate the hit only
  // if the object's direction is known with sufficient precision, i.e. we have
  // tracked the object with more than 8 frames.
  if ( iFlgHeadingStable == NO ) {
    if ( dist2 > 90000 ) return -1.0;
  }
  
  r = mRadiusC + iRadius;
  n = a*a + b*b;
  c = (a*dx + b*dy) / n;
  d = (dist2 - r*r) / n;
  e = c*c - d;
  if ( e < 0.0 ) return -1.0;
  e = sqrt(e);
  q1 = -c - e;
  q2 = -c + e;
  
  if ( q1 >= 0.0 && q2 >= 0.0 ) return q1;
  return -1.0;
  if ( q1 < 0.0 && q2 < 0.0 ) return -1.0;
  return 0.0;
}


-(void)synchronizeAngleByteFromHeadingX:(int)iHeadingX headingY:(int)iHeadingY {

  int i;
  BOOL flgDone = NO;
  
  for ( i = 0; i < 256 && flgDone == NO; i++ ) {
    if ( gAngles[i].intX == iHeadingX && gAngles[i].intY == iHeadingY ) {
//      if ( (int)mAngleByte != i )
//        NSLog(@"Angle byte synchronized from %d to %d", (int)mAngleByte, i);
      mAngleByte = i;
      flgDone = YES;
    }
  }
}


#if defined WITH_WINDOW || defined INTERSECTION_TEST
-(void)draw {
  [super draw];  
  [[NSColor greenColor] set];
  [NSBezierPath strokeLineFromPoint:NSMakePoint((float)mPosX,(float)mPosY) toPoint:NSMakePoint((float)(mPosX+50.0*mAngles[mAngleByte].intX),(float)(mPosY+50.0*mAngles[mAngleByte].intY))];
  [[NSColor redColor] set];
  [NSBezierPath strokeLineFromPoint:NSMakePoint((float)mPosX,(float)mPosY) toPoint:NSMakePoint((float)(mPosX+mAngles[mAngleByte].visX),(float)(mPosY+mAngles[mAngleByte].visY))];
}
#endif


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Private Methods
// -------------------------------------------------------------------------------------


@end
