// player.cpp: Beispielspieler fr Asteroids
// Harald Bgeholz / c't

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#if defined(WINDOWS)
#include <winsock2.h>
#else
// 2 Includes fr socket()
#include <sys/types.h>
#include <sys/socket.h>
// 2 Includes fr inet_addr()
#include <netinet/in.h>
#include <arpa/inet.h>
// 2 Includes fr fcntl()
#include <unistd.h>
#include <fcntl.h>
// fr memset()
#define INVALID_SOCKET -1
#define WSAGetLastError() errno
#endif

#include "player.h"

#define HYPERSPACE_DISTANCE_THRESHOLD	34

#define MINIMUM_SHOT_VELOCITY		5.0
#define MAXIMUM_SHOT_VELOCITY		14.0

#define SMALL_THREAT_TIME_TO_IMPACT		75
#define MEDIUM_THREAT_TIME_TO_IMPACT	90
#define LARGE_THREAT_TIME_TO_IMPACT		120

#define MAXIMUM_POSITION_TRACKING_ERROR	50
#define MINIMUM_SQUARED_VELOCITY		1
#define MAXIMUM_SQUARED_VELOCITY		1000
#define MAXIMUM_INTEGER					0x7FFFFFFF

#define ONE_DEGREE_IN_RADIANS				0.0175
#define SHIP_TURN_GRANULARITY_IN_DEGREES	5.625
#define SHIP_TURN_PER_FRAME					4.219

#define THRUST_FREQUENCY		9
#define MAXIMUM_SHIP_VELOCITY	1.0

#define SHOT_ESCAPE_TIME	20.0

#define MAXIMUM_TARGET_TIME 54

#define SHOT_FIRE_DELAY		2

// Distance at which small asteroids can be hit 100%
// (takes turn granularity into account)
#define SMALL_CERTAIN_HIT_DISTANCE	215
#define MEDIUM_CERTAIN_HIT_DISTANCE	430
#define LARGE_CERTAIN_HIT_DISTANCE	860

// TODO: Keep a target only for X frames, the pick a new one.
//	X ~= 40-60


void Player::Run(void)
{
	FramePacket frame;
	KeysPacket keys;
	GameStatus game;
	char prevframe = 0;
	int t = 0, thrust_ctr = 0;
//	static int shot_stat[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

	game.ship_target = NULL;

	for (;;)
	{
		++t;         // Zeit
		++keys.ping; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung
		SendPacket(keys);
		ReceivePacket(frame);

		if (frame.frameno != ++prevframe || frame.ping != keys.ping)
		{
			printf("Latenz %d. %d Frames verloren.\n", keys.ping - frame.ping, frame.frameno - prevframe);
			prevframe = frame.frameno;
		}

		InterpretScreen(frame, game);
		game.update_predictions();
		game.track_objects();
//		shot_stat[game.n_player_shots]++;
		game.update_predictions();
		game.update_objects();
		game.tracked_ship.update_angle();

		if(game.nasteroids > 26)
		{	printf("WARNING: 27 asteroids !\n");	}

		keys.clear();   // alle Tasten loslassen

		/* Weil der Bildschirm 4:3-Format hat, werden von dem technisch mglichen Koordinatenbereich
	fr die y-Achse nur die Werte 128 bis 895 genutzt, whrend die x-Koordinaten Werte
	zwischen 0 und 1023 annehmen. Grere Zahlen sind rechts beziehungsweise oben.	*/
		if (game.ship_present)
		{
			keys.fire(game.fire_shot());

			game.pick_target();

			game.turn_ship(keys);

			keys.thrust(game.thrust_ship());

			keys.hyperspace(game.enter_hyperspace());

		}
	}
}


/* ****************************************************
	Implementation of GameStatus
   **************************************************** */

GameStatus::GameStatus(void)
{
	opportunity_shot = false;
	target_threat = false;;
	fire_counter = 0;
	thrust_counter = 0;
	same_target_counter = 0;
	is_thrust_frame = false;
	ship_target = NULL;
	opportunity_target = NULL;
}

void GameStatus::turn_ship(KeysPacket &keys)
{
	double angle_is, angle_tolerance;
	CVector target;

	tracked_ship.m_dir.last_turn[0] = 0;

	if(!saucer_present && (nasteroids == 0))
		{
// Orient ship so that it points towards the second_nearest corner
		if(			(tracked_ship.m_pos.abs_predicted.get_x() > 511) &&
					(tracked_ship.m_pos.abs_predicted.get_y() > 511) )
		{	target.set_xy(1023, 895);	}
		else if(	(tracked_ship.m_pos.abs_predicted.get_x() <= 511) &&
					(tracked_ship.m_pos.abs_predicted.get_y() > 511) )
		{	target.set_xy(0, 895);	}
		else if(	(tracked_ship.m_pos.abs_predicted.get_x() <= 511) &&
					(tracked_ship.m_pos.abs_predicted.get_y() <= 511) )
		{	target.set_xy(0, 128);	}
		else
		{	target.set_xy(1023, 128);	}
		target = target - tracked_ship.m_pos.abs_predicted;
		target.normalize();
		}
	else if(ship_target == NULL)
		{	return;	}
	else
		{	target = ship_target->m_dir.target;	}

//	angle_is = tracked_ship.m_dir.predicted.angle_to(ship_target->m_dir.target);
	angle_is = tracked_ship.m_dir.predicted.angle_to(target);
	angle_tolerance = 0.01 * ONE_DEGREE_IN_RADIANS * SHIP_TURN_GRANULARITY_IN_DEGREES;
	
	tracked_ship.m_dir.last_turn[0] = 0;

	if (angle_is > angle_tolerance)
		{
		tracked_ship.m_dir.last_turn[0] = LEFT_TURN;
		keys.left(true);
		}
	else if(angle_is < - angle_tolerance)
		{	
		tracked_ship.m_dir.last_turn[0] = RIGHT_TURN;
		keys.right(true);
		}


}

bool GameStatus::fire_shot(void)
{
//	double angle_diff, angle_hit;
	bool can_fire;
	static int shots_fired = 0;

	if(fire_counter > 0)
		{	fire_counter--;	}

	can_fire = ship_can_fire();

//
	if((ship_target == NULL) && (nasteroids == 0) &&
		(!saucer_present))
		{
		if(can_fire)
			{
			fire_counter = SHOT_FIRE_DELAY + 14;
			return(true);
			}
		return(false);
		}
	if(ship_target == NULL)
	{	return(false);	}
	if(	!can_fire )
	{	return(false);	}


// 1. Only fire if ship is pointed towards target (with some tolerance, also depending on target size
//		and distance)

/*	angle_diff	= abs(tracked_ship.m_dir.predicted.angle_to(ship_target->m_dir.target));

	angle_hit	= atan(ship_target->m_radius / ship_target->m_pos.intercept.distance_to(tracked_ship.m_pos.abs_predicted));
	angle_hit	= abs(angle_hit);
	angle_hit	= ACCURACY_FACTOR * angle_hit;

	if( angle_diff <= angle_hit)*/
	if(tracked_ship.can_hit(ship_target))
		{
		fire_counter = 0;
		ship_target->m_time_to_hit = (int) 
			(ship_target->m_shot_travel_time + 1.0);
		ship_target->m_incoming_shots++;
		fire_counter = SHOT_FIRE_DELAY;
		shots_fired++;
		if(shots_fired % 50 == 0)
			{	printf("Shots fired: %i\n", shots_fired);	}
		return(true);
		}

	check_opportunity_shot();
	if(opportunity_shot == true) 
		{
		if(opportunity_target != NULL)
			{
			opportunity_target->m_time_to_hit = (int) (0.92 * opportunity_target->m_shot_travel_time);
			opportunity_target->m_incoming_shots++;
			}
		printf("FIRE_SHOT: Taking opportunity shot !!!\n");
		fire_counter = SHOT_FIRE_DELAY;
		shots_fired++;
		if(shots_fired % 50 == 0)
			{	printf("Shots fired: %i\n", shots_fired);	}
		return(true);
		}
	return(false);
}

bool GameStatus::ship_can_fire()
{
	int i, j;
	CVector pos_shot, pos_asteroid;

	if(fire_counter != 0)
		{	return(false);	}
	if(n_player_shots < 4)
		{	return(true);	}

// TODO: Calculate if ship can fire two frames from now.
//	For each player shot:
	for(i = 0; i < MAX_SHOTS; i++)
		{
		if(	(tracked_shot[i].m_is_active == false) ||
			(tracked_shot[i].m_type != player_shot)	)
		{	continue;	}
//		Calculate predicted position 2 frames ahead.
		pos_shot =	tracked_shot[i].m_pos.abs_predicted +
					tracked_shot[i].m_vel.abs_averaged;
		pos_shot.torus_wrap_absolute();
//		For each active asteroid
		for(j = 0; j < MAX_ASTEROIDS; j++)
			{
			if(tracked_asteroid[j].m_is_active == false)
				{	continue;	}
//			Calculate predicted position 2 frames ahead
			pos_asteroid =	tracked_asteroid[j].m_pos.abs_predicted +
							tracked_asteroid[j].m_vel.abs_averaged;
			pos_asteroid.torus_wrap_absolute();
//			Check distance to shot. If < threshold, shot should disappear
			if(pos_shot.distance_to(pos_asteroid) < tracked_asteroid[j].m_radius - 0.5)
				{
//				printf("SHIP_CAN_FIRE: Can fire early !\n");
				return(true);
				}
			}
		}
	return(false);
}

void GameStatus::check_opportunity_shot(void)
{
	int i;
//	double angle_diff, angle_hit;
	double opportunity_shot_travel_time;

//	Check for opportunity shots
	opportunity_shot = false;
	opportunity_shot_travel_time = 40.0;
	opportunity_target = NULL;
	if(nasteroids > 15)
	{	opportunity_shot_travel_time = 25.0;	}
	if(nasteroids > 20)
	{	opportunity_shot_travel_time = 20.0;	}
	opportunity_shot_travel_time += 8.0 * (4 - n_player_shots); 

	if(	(tracked_saucer.m_is_active) &&
		(&tracked_saucer != ship_target)	&&
		(tracked_saucer.enough_shots() == false))
		{
/*		angle_diff	= abs(tracked_ship.m_dir.predicted.angle_to(tracked_saucer.m_dir.target));
		angle_hit	= atan(tracked_saucer.m_radius / 
					tracked_saucer.m_pos.intercept.distance_to(tracked_ship.m_pos.abs_predicted));
		angle_hit	= abs(angle_hit);
		angle_hit	= ACCURACY_FACTOR * angle_hit; 
		if(	(angle_diff <= angle_hit) &&	*/
		if(	tracked_ship.can_hit(&tracked_saucer) &&
			(tracked_saucer.m_shot_travel_time < opportunity_shot_travel_time) )
			{	
			opportunity_shot = true;
			opportunity_target = &tracked_saucer;
			}
		}

	for(i = 0; i < MAX_ASTEROIDS; i++)
		{
		if(!tracked_asteroid[i].m_is_active)
			{	continue;	}
		if(	(!tracked_asteroid[i].m_velocity_known) &&
			(tracked_asteroid[i].m_radius < ASTEROID_RADIUS_LARGE))
			{	continue;	}
		if(&tracked_asteroid[i] == ship_target)
			{	continue;	}
		if(tracked_asteroid[i].enough_shots() == true)
			{	continue;	}

/*		angle_diff	= abs(tracked_ship.m_dir.predicted.angle_to(tracked_asteroid[i].m_dir.target));
		angle_hit	= atan(tracked_asteroid[i].m_radius / 
					tracked_asteroid[i].m_pos.intercept.distance_to(tracked_ship.m_pos.abs_predicted));
		angle_hit	= abs(angle_hit);
		angle_hit	= ACCURACY_FACTOR * angle_hit;

		if(	(angle_diff <= angle_hit) && */
		if(	tracked_ship.can_hit(&tracked_asteroid[i]) &&
			(tracked_asteroid[i].m_shot_travel_time < opportunity_shot_travel_time) &&
			(tracked_asteroid[i].m_time_to_hit == 0))
			{	
			opportunity_shot = true;
			if(opportunity_target == NULL)
			{	opportunity_target = &tracked_asteroid[i];	}
			else if(tracked_asteroid[i].m_shot_travel_time < opportunity_target->m_shot_travel_time)
			{	opportunity_target = &tracked_asteroid[i];	}

			if(	(tracked_asteroid[i].m_radius >= ASTEROID_RADIUS_MEDIUM) &&
				(tracked_asteroid[i].distance_from(tracked_ship) < 40.0) )
				{	
				opportunity_shot = false;
				break;
				}
			}
		}
}

// Maneuver the ship to the edge of the visible area
//	TODO: Stay close to the edge, but not _too_ close due to roid
//		respawn
#define TOO_CLOSE_LEFT		120
#define CLOSE_ENOUGH_LEFT	220
#define TOO_CLOSE_RIGHT		(1023 - TOO_CLOSE_LEFT)
#define CLOSE_ENOUGH_RIGHT	(1023 - CLOSE_ENOUGH_LEFT)
#define MIDDLE_X			511

#define TOP_Y		895
#define BOTTOM_Y	128
#define TOO_CLOSE_BOTTOM		(BOTTOM_Y + 120)
#define CLOSE_ENOUGH_BOTTOM		(BOTTOM_Y + 220)
#define TOO_CLOSE_TOP			(1023 - TOO_CLOSE_BOTTOM)
#define CLOSE_ENOUGH_TOP		(1023 - CLOSE_ENOUGH_BOTTOM)
#define MIDDLE_Y				511


bool GameStatus::thrust_ship(void)
{
	bool ship_points_right, ship_points_left;
	bool ship_points_up, ship_points_down;

	if(is_thrust_frame == false)
	{	return(false);	}

	if(thrust_counter < 0)
		{	thrust_counter = 0;	}

	thrust_counter++;

	if(thrust_counter < THRUST_FREQUENCY)
	{	return(false);	}

	ship_points_right = false;
	ship_points_left = false;
	if(tracked_ship.m_dir.predicted.get_x() > 0.8)
	{	ship_points_right = true;	}
	if(tracked_ship.m_dir.predicted.get_x() < -0.8)
	{	ship_points_left = true;	}

	ship_points_up = false;
	ship_points_down = false;
	if(tracked_ship.m_dir.predicted.get_y() > 0.8)
	{	ship_points_up = true;	}
	if(tracked_ship.m_dir.predicted.get_y() < -0.8)
	{	ship_points_down = true;	}

// If close to edge already
	if(	(ship_points_right == true) &&
		((tracked_ship.m_pos.abs_predicted.get_x() <= TOO_CLOSE_LEFT) ||
		 (((tracked_ship.m_pos.abs_predicted.get_x() > MIDDLE_X) &&
		  (tracked_ship.m_pos.abs_predicted.get_x() < CLOSE_ENOUGH_RIGHT)))))
	{
		thrust_counter = 0;
		return(true);
	}

	if(	(ship_points_left == true) &&
		((tracked_ship.m_pos.abs_predicted.get_x() >= TOO_CLOSE_RIGHT) ||
		(((tracked_ship.m_pos.abs_predicted.get_x() <= MIDDLE_X) &&
		  (tracked_ship.m_pos.abs_predicted.get_x() > CLOSE_ENOUGH_LEFT)))))
	{
		thrust_counter = 0;
		return(true);
	}

	if(	(ship_points_up == true) &&
		((tracked_ship.m_pos.abs_predicted.get_y() <= TOO_CLOSE_BOTTOM) ||
		(((tracked_ship.m_pos.abs_predicted.get_y() > MIDDLE_Y) &&
		  (tracked_ship.m_pos.abs_predicted.get_y() < CLOSE_ENOUGH_TOP)))))
	{
		thrust_counter = 0;
		return(true);
	}

	if(	(ship_points_down == true) &&
		((tracked_ship.m_pos.abs_predicted.get_y() >= TOO_CLOSE_TOP) ||
		(((tracked_ship.m_pos.abs_predicted.get_y() <= MIDDLE_Y) &&
		  (tracked_ship.m_pos.abs_predicted.get_y() > CLOSE_ENOUGH_BOTTOM)))))
	{
		thrust_counter = 0;
		return(true);
	}

	return(false);
}

bool GameStatus::enter_hyperspace(void)
{
	int i, min_dist = 0x7FFFFFFF;

// Check for asteroid collision

// TODO: Do not hyperspace away from asteroids that are predicted to miss
	for(i = 0; i < MAX_ASTEROIDS; i++)
		{
		if(tracked_asteroid[i].m_is_active)
			{
			if(	tracked_asteroid[i].predicted_distance_from(tracked_ship) <= 
				tracked_asteroid[i].m_radius + tracked_ship.m_radius + 3.0)

				{
				printf("Hyperspace due to asteroid closer than %f\n", tracked_asteroid[i].distance_from(tracked_ship));
				if(!target_threat)
				{	printf("Hyperspace without THREAT !!!\n"); }
				return(true);
				}
			}
		}

// Check for shot collision
	for(i = 0; i < MAX_SHOTS; i++)
		{
		if(tracked_shot[i].m_is_active)
			{
			if(	(tracked_shot[i].m_velocity_known) &&
				(tracked_shot[i].m_type == saucer_shot) &&
				(tracked_shot[i].minimum_distance_from(tracked_ship) <= tracked_ship.m_radius) &&
				(tracked_shot[i].time_to_minimum_distance_from(tracked_ship) > 0) &&
				(tracked_shot[i].time_to_minimum_distance_from(tracked_ship) < SHOT_ESCAPE_TIME))
				{
				printf("Hyperspace due to shot estimated closer than %f\n", tracked_shot[i].minimum_distance_from(tracked_ship));
				return(true);
				}
			}
		}

// Check for UFO collision
	if(saucer_present)
		{
		if(	tracked_saucer.distance_from(tracked_ship) <= 
			tracked_saucer.m_radius + tracked_ship.m_radius)
			{
			printf("Hyperspace due to saucer closer than %f\n", tracked_saucer.distance_from(tracked_ship));
			return(true);
			}
		}
//	
	return(false);
}

void GameStatus::pick_target(void)
{
	int i;
	double distance, min_time_to_impact, target_score, min_target_score, time_to_min_dist;;
	double size_score, distance_score, turn_score, travel_score, attack_score;
	Tracked_Object *old_target;

	old_target = ship_target;
	ship_target = NULL;
	min_time_to_impact = LARGE_THREAT_TIME_TO_IMPACT;
	target_threat = false;

//	Update firing solutions for any active asteroid
	update_firing_solutions();

	for(i = 0; i < MAX_ASTEROIDS; i++)
		{
//	Find and target threatening asteroids
		if(!tracked_asteroid[i].m_is_active)
			{	continue;	}
		distance = (tracked_asteroid[i].minimum_distance_from(tracked_ship) - tracked_asteroid[i].m_radius);
		if(distance < 0)
			{	distance = 0;	}
		time_to_min_dist = tracked_asteroid[i].time_to_minimum_distance_from(tracked_ship);
		if(time_to_min_dist < 0.0)
		{	continue;	}

		if(	(tracked_asteroid[i].m_radius >= ASTEROID_RADIUS_LARGE) &&
			(time_to_min_dist > LARGE_THREAT_TIME_TO_IMPACT) )
		{	continue;	}
		if(	(tracked_asteroid[i].m_radius > ASTEROID_RADIUS_SMALL) && (tracked_asteroid[i].m_radius < ASTEROID_RADIUS_MEDIUM) &&
			(time_to_min_dist > MEDIUM_THREAT_TIME_TO_IMPACT) )
		{	continue;	}
		if(	(tracked_asteroid[i].m_radius <= ASTEROID_RADIUS_SMALL) &&
			(time_to_min_dist > SMALL_THREAT_TIME_TO_IMPACT) )
		{	continue;	}

		if(	(tracked_asteroid[i].time_to_minimum_distance_from(tracked_ship) > 0) &&
			(distance <= HYPERSPACE_DISTANCE_THRESHOLD + 4) &&
			(time_to_min_dist < min_time_to_impact) &&
			(tracked_asteroid[i].enough_shots() == false) )
			{
				target_threat = true;
				min_time_to_impact = (int) tracked_asteroid[i].time_to_minimum_distance_from(tracked_ship);
				ship_target = &tracked_asteroid[i];
			}
		}

//if(ship_target != NULL)
//{	printf("PICK_TARGET: Targeting THREAT !!!\n");	}

//	Find and target nonthreatening asteroids
//	Prefer targets by size (prefer small), distance (prefer close) and angular difference
//		from ship facing (prefer small)
	min_target_score = 1000000.0;
	if(ship_target == NULL)
		{
		
		for(i = 0; i < MAX_ASTEROIDS; i++)
			{
			if(!tracked_asteroid[i].m_is_active)
				{	continue;	}
			if(	(!tracked_asteroid[i].m_velocity_known) &&
				(tracked_asteroid[i].m_radius < ASTEROID_RADIUS_LARGE))
				{	continue;	}

			attack_score = 0;
			if( tracked_asteroid[i].enough_shots() == true) 
			{	attack_score += 1000.0;		}
//	Size score
			size_score = 0.0;
			if(tracked_asteroid[i].m_radius >= ASTEROID_RADIUS_MEDIUM)
				{
				size_score = 0.0;
				if(tracked_asteroid[i].m_radius >= ASTEROID_RADIUS_LARGE)
				{	size_score += 0.0;	}
				if(nasteroids > 23)
				{	size_score += 30.0;	}
			}
//	Distance score: Distance in pixels
			distance_score = tracked_asteroid[i].distance_from(tracked_ship) - tracked_asteroid[i].m_radius;
			if(distance_score < 0)
				{	distance_score = 0;	}
//	Turn score
			turn_score = tracked_asteroid[i].m_ship_turn_time;
			if(tracked_asteroid[i].m_easy_prey == false) // Penalize not exactly hittable
			{	turn_score += 150.0;		}
//	Shot travel score
			travel_score = tracked_asteroid[i].m_shot_travel_time;
			travel_score = 1.375 * travel_score;
			if(tracked_asteroid[i].m_shot_travel_time > 69.0)	// Penalize out of range
			{	travel_score += 200.0;	}
			if(nasteroids > 15)
			{	travel_score = travel_score * 1.375;	}	
// Total score
// TODO: Award penalty if asteroid is medium or large and too close
			target_score =	attack_score + 
							size_score +
							0.0 * distance_score +
							1.0 * turn_score +
							1.2 * travel_score;
//			printf("Scores: size: %3.1f, distance: %3.1f, angle: %3.1f, total: %3.1f\n", size_score, distance_score, angle_score, target_score);
//	Pick best target
			if(target_score < min_target_score)
				{
				min_target_score = target_score;
				ship_target = &tracked_asteroid[i];
				}
			}
//	Target saucer if not threatened by asteroid
		if (tracked_saucer.m_is_active)
			{
			if(tracked_saucer.m_type == saucer_large)
				{
// Treat larget saucer as just another target
				attack_score = 0;
				if( tracked_saucer.enough_shots() == true) 
				{	attack_score += 1000.0;		}
//
				distance_score = tracked_saucer.distance_from(tracked_ship) - tracked_saucer.m_radius;
				if(distance_score < 0)
					{	distance_score = 0;	}
//	Turn score
				turn_score = tracked_saucer.m_ship_turn_time;
				if(tracked_saucer.m_easy_prey == false) // Penalize not exactly hittable
				{	turn_score += 150.0;		}
//	Shot travel score
			travel_score = tracked_saucer.m_shot_travel_time;
			travel_score = 1.375 * travel_score;
				if(tracked_saucer.m_shot_travel_time > 69.0)	// Penalize out of range
				{	travel_score += 150.0;	}
				if(nasteroids > 15)
				{	travel_score = travel_score * 1.375;	}	
// Total score
// TODO: Award penalty if asteroid is medium or large and too close
				target_score =	attack_score + 
								0.0 * distance_score +
								1.0 * turn_score +
								1.0 * travel_score;
				if(target_score < min_target_score)
					{
					min_target_score = target_score;
					ship_target = &tracked_saucer;
					}
			}
			else
			{
				if((nasteroids == 0)  || (tracked_saucer.enough_shots() == false))
					{	ship_target = &tracked_saucer;	}
			}
//			printf("PICK_TARGET: Saucer present !!\n");
			}
		}
// Update same_target_counter
	if(old_target != NULL)
		{
		if(	(ship_target == old_target) &&
			(ship_target->m_type == old_target->m_type)	)
			{	same_target_counter++;	}
		else
			{	same_target_counter = 0;	}
		}
if((ship_target == NULL) && (nasteroids != 0))
{	printf("PICK_TARGET: NO target selected despite %i asteroids available !!!\n", nasteroids);	}
/*
if(saucer_present && (ship_target != &tracked_saucer) && !target_threat)
{	
	printf("PICK_TARGET: SAUCER present, but not targetted despite no threat!!\n");}
*/
}

void GameStatus::update_firing_solutions(void)
{
	int i;

	if(!ship_present)
		{	return;	}

	for(i = 0; i < MAX_ASTEROIDS; i++)
		{	
		if(tracked_asteroid[i].m_is_active == true)
			{
			tracked_ship.newton_firing_solution_to(&tracked_asteroid[i]);
			}
		}
	if(tracked_saucer.m_is_active == true)
		{
		tracked_ship.newton_firing_solution_to(&tracked_saucer);
		}
}

void GameStatus::clear(void)
{
	ship_present = false;
	saucer_present = false;
	nasteroids = 0;
	nshots = 0;
}

void GameStatus::update_predictions(void)
{
	int i;

	tracked_ship.predict_position();
	tracked_saucer.predict_position();

	for(i = 0; i < MAX_SHOTS; i++)
		{	tracked_shot[i].predict_position();	}

	for(i = 0; i < MAX_ASTEROIDS; i++)
		{	tracked_asteroid[i].predict_position();	}
}


void GameStatus::track_objects(void)
{
	track_ship();
	track_saucer();
	track_asteroids();
	track_shots();
}

void GameStatus::update_objects(void)
{
	int i;

	for(i = 0; i < MAX_ASTEROIDS; i++)
		{
		if(tracked_asteroid[i].m_is_active == false)
		{	continue;	}
		if(tracked_asteroid[i].m_time_to_hit > 0) 
			{	tracked_asteroid[i].m_time_to_hit--;	}
		else if(tracked_asteroid[i].m_time_to_hit == 0)
			{	tracked_asteroid[i].m_incoming_shots = 0;	}
		else
			{	tracked_asteroid[i].m_time_to_hit = 0;	}
		}

	if(	tracked_saucer.m_is_active == true)
		{
		if(tracked_saucer.m_time_to_hit > 0)
			{	tracked_saucer.m_time_to_hit--;		}
		else if(tracked_saucer.m_time_to_hit == 0)
			{	tracked_saucer.m_incoming_shots = 0;	}
		else
			{	tracked_saucer.m_time_to_hit = 0;	}
		}
}


void GameStatus::track_ship(void)
{
	if (ship_present)
		{	tracked_ship.track(&ship);	}
	else
		{	tracked_ship.set_inactive();	}
}

void GameStatus::track_saucer(void)
{
	if (saucer_present)
		{	tracked_saucer.track(&saucer);	}
	else
		{	tracked_saucer.set_inactive();	}
}


void GameStatus::track_asteroids(void)
{
	int a_idx, ta_idx, min_dist_idx;
	double min_dist,  curr_dist;
	Tracked_Object *p_Ast;
	CVector screen_pos;

//	0. Don't track any asteroids if there aren't any asteroids on the screen;
//	TODO: Set all asteroids to inactive
	if(nasteroids == 0)
		{
		for(ta_idx = 0; ta_idx < MAX_ASTEROIDS; ta_idx++)
			{	tracked_asteroid[ta_idx].set_inactive();	}

		return;
		}
//	1. Track any active asteroids with known velocity
	for(ta_idx = 0; ta_idx < MAX_ASTEROIDS; ta_idx++)
		{
		p_Ast = &tracked_asteroid[ta_idx];
		min_dist_idx = -1;
		min_dist = MAXIMUM_INTEGER;
		if(	(p_Ast->m_is_active) &&
			(p_Ast->m_velocity_known)	)
			{
//	Find closest asteroid with the same size and type
			for(a_idx = 0; a_idx < nasteroids; a_idx++)
				{
				if(	(asteroids[a_idx].is_tracked == false)	&&
					(asteroids[a_idx].get_object_type() == p_Ast->m_type)	)
					{
					screen_pos.set_xy(asteroids[a_idx].x, asteroids[a_idx].y);

					curr_dist = p_Ast->predicted_distance_from(screen_pos.get_x(), screen_pos.get_y());

					if(curr_dist < min_dist)
						{
						min_dist = curr_dist;
						min_dist_idx = a_idx;
						}
					}
				}

			if(min_dist <= MAXIMUM_POSITION_TRACKING_ERROR)
				{	p_Ast->track(&asteroids[min_dist_idx]);	}
			else
				{	p_Ast->set_inactive();	}
			}
		}

//	2. Track any active asteroids with unknown velocity
	for(ta_idx = 0; ta_idx < MAX_ASTEROIDS; ta_idx++)
		{
		p_Ast = &tracked_asteroid[ta_idx];
		min_dist_idx = -1;
		min_dist = MAXIMUM_INTEGER;

		if(	(p_Ast->m_is_active) &&
			!(p_Ast->m_velocity_known)	)
			{
//	Find closest asteroid with the same size
			for(a_idx = 0; a_idx < nasteroids; a_idx++)
				{
				if(	(asteroids[a_idx].is_tracked == false)	&&
					(asteroids[a_idx].get_object_type() == p_Ast->m_type)	)
					{
					screen_pos.set_xy(asteroids[a_idx].x, asteroids[a_idx].y);

					curr_dist = p_Ast->distance_from(screen_pos.get_x(), screen_pos.get_y());

					if(curr_dist < min_dist)
						{
						min_dist = curr_dist;
						min_dist_idx = a_idx;
						}
					}
				}
			if(	(min_dist >= MINIMUM_SQUARED_VELOCITY) &&
				(min_dist <= MAXIMUM_SQUARED_VELOCITY)	)
				{	p_Ast->track(&asteroids[min_dist_idx]);	}
			else
				{	p_Ast->set_inactive();	}
			}
		}

//	3. Track any new asteroids
	for(a_idx = 0; a_idx < nasteroids; a_idx++)
		{
		if(asteroids[a_idx].is_tracked == false)
			{
			for(ta_idx = 0; ta_idx < MAX_ASTEROIDS; ta_idx++)
				{
				p_Ast = &tracked_asteroid[ta_idx];
				if(!(p_Ast->m_is_active))
					{
					p_Ast->track(&asteroids[a_idx]);
					break;
					}
				}
			}
		}
}

void GameStatus::track_shots(void)
{
	int s_idx, ts_idx, min_dist_idx; 
	double min_dist,  curr_dist;
	CVector screen_pos;
	Tracked_Object *p_Shot;

//	0. Don't track any shot if there aren't any asteroids on the screen;
//	TODO: Set all shots to inactive
	n_player_shots = 0;
	if(nshots == 0)
		{
		for(ts_idx = 0; ts_idx < MAX_SHOTS; ts_idx++)
			{	tracked_shot[ts_idx].set_inactive();	}
		return;
		}
//	1. Track any active shots with known velocity

// TODO: Algorithm still loses track when tracking shot (esp saucer)
// TODO: New player shots start <20 units away from ship. USE THIS INFORMATION
	for(ts_idx = 0; ts_idx < MAX_SHOTS; ts_idx++)
		{
		p_Shot = &tracked_shot[ts_idx];
		min_dist_idx = -1;
		min_dist = MAXIMUM_INTEGER;
		if(	(p_Shot->m_is_active) &&
			(p_Shot->m_velocity_known)	)
			{
			for(s_idx = 0; s_idx < nshots; s_idx++)
				{
				if(	shots[s_idx].is_tracked == false)
					{
					screen_pos.set_xy(shots[s_idx].x, shots[s_idx].y);
					curr_dist = p_Shot->predicted_distance_from(screen_pos.get_x(), screen_pos.get_y());
					if(	(curr_dist < min_dist) &&
						(curr_dist <= p_Shot->m_vel.abs_averaged.length() + 10.0 ) &&
						(curr_dist >= p_Shot->m_vel.abs_averaged.length() - 10.0)	)
						{
						min_dist = curr_dist;
						min_dist_idx = s_idx;
						}
					}
				}
			if(	(min_dist <= p_Shot->m_vel.abs_averaged.length() + 10.0)	&&
				(min_dist >= p_Shot->m_vel.abs_averaged.length() - 10.0)	)
				{	p_Shot->track(&shots[min_dist_idx], s_idx, tracked_ship, tracked_saucer);	}
			else
				{
				if(p_Shot->m_type == saucer_shot)
					printf("Lost track of saucer shot, velocity known !\n");
				p_Shot->set_inactive();
				}
			}
		}

//	2. Track any active asteroids with unknown velocity
	for(ts_idx = 0; ts_idx < MAX_SHOTS; ts_idx++)
		{
		p_Shot = &tracked_shot[ts_idx];
		min_dist_idx = -1;
		min_dist = MAXIMUM_INTEGER;

		if(	(p_Shot->m_is_active) &&
			!(p_Shot->m_velocity_known)	)
			{
//	Find closest shot 
			for(s_idx = 0; s_idx < nshots; s_idx++)
				{
				if(	shots[s_idx].is_tracked == false)
					{
					screen_pos.set_xy(shots[s_idx].x, shots[s_idx].y);
					curr_dist = p_Shot->predicted_distance_from(screen_pos.get_x(), screen_pos.get_y());

					if(	(curr_dist < min_dist) &&
						(curr_dist <= MAXIMUM_SHOT_VELOCITY) &&
						(curr_dist >= MINIMUM_SHOT_VELOCITY)	)
						{
						min_dist = curr_dist;
						min_dist_idx = s_idx;
						}
					}
				}
			if(	(min_dist <= MAXIMUM_SHOT_VELOCITY) &&
				(min_dist >= MINIMUM_SHOT_VELOCITY)	)
				{	p_Shot->track(&shots[min_dist_idx], s_idx, tracked_ship, tracked_saucer);	}
			else
				{
				if(p_Shot->m_type == saucer_shot)
					printf("Lost track of saucer shot !\n");
				p_Shot->set_inactive();
				}
			}
		}

//	3. Track any new shots
	for(s_idx = 0; s_idx < nshots; s_idx++)
		{
		if(shots[s_idx].is_tracked == false)
			{
			for(ts_idx = 0; ts_idx < MAX_SHOTS; ts_idx++)
				{
				p_Shot = &tracked_shot[ts_idx];
				if(!(p_Shot->m_is_active))
					{
					p_Shot->track(&shots[s_idx], s_idx, tracked_ship, tracked_saucer);
					break;
					}
				}
			}
		}
//	Count player shots:
	for(ts_idx = 0; ts_idx < MAX_SHOTS; ts_idx++)
		{
		if((tracked_shot[ts_idx].m_is_active) && (tracked_shot[ts_idx].m_type == player_shot))
			{	n_player_shots++;	}
		}
	if(n_player_shots > 4)
	{	printf("TRACK_SHOTS: n_player_shots > 4. IMPOSSIBLE !!!!\n");}
}

/*	**************************************
	Implementation of Player 
	************************************** */

void Player::InterpretScreen(FramePacket &packet, GameStatus& game)
{
	unsigned short *vector_ram = (unsigned short *)packet.vectorram;
	int dx, dy, sf, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;

	game.clear();
	if ((unsigned char)packet.vectorram[1] != 0xe0 && (unsigned char)packet.vectorram[1] != 0xe2)
		return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

	if((unsigned char)	packet.vectorram[1] == 0xe2)
	{	game.is_thrust_frame = false;	}
	else
	{	game.is_thrust_frame = true;	}

	int pc = 1;
	for (;;)
	{
		int op = vector_ram[pc] >> 12;
		switch (op)
		{
		case 0xa: // LABS
			vy = vector_ram[pc] & 0x3ff;
			vx = vector_ram[pc+1] & 0x3ff;
			vs = vector_ram[pc+1] >> 12;
			break;
		case 0xb: // HALT
			return;
		case 0xc: // JSRL
			switch (vector_ram[pc] & 0xfff)
			{
			case 0x8f3:
				game.asteroids[game.nasteroids++].set(vx, vy, 1, vs);
				break;
			case 0x8ff:
				game.asteroids[game.nasteroids++].set(vx, vy, 2, vs);
				break;
			case 0x90d:
				game.asteroids[game.nasteroids++].set(vx, vy, 3, vs);
				break;
			case 0x91a:
				game.asteroids[game.nasteroids++].set(vx, vy, 4, vs);
				break;
			case 0x929:
				game.saucer_present = true;
				game.saucer.x = vx;
				game.saucer.y = vy;
				game.saucer.size = vs;
				break;
			}  
			break;
		case 0xd: // RTSL
			return;
		case 0xe: // JMPL
			/*
			pc = vector_ram[pc] & 0xfff;
			break;
			*/
			return;
		case 0xf: // SVEC
			/*
			dy = vector_ram[pc] & 0x300;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = (vector_ram[pc] & 3) << 8;
			if ((vector_ram[pc] & 4) != 0)
				dx = -dx;
			sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
			vz = (vector_ram[pc] & 0xf0) >> 4;
			*/
			break;
		default:
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;
			sf = op;
			vz = vector_ram[pc+1] >> 12;
			if (dx == 0 && dy == 0 && vz == 15)
				game.shots[game.nshots++].set(vx, vy);
			if (op == 6 && vz == 12 && dx != 0 && dy != 0)
			{
				switch (shipdetect)
				{
				case 0:
					v1x = dx;
					v1y = dy;
					++shipdetect;
					break;
				case 1:
					game.ship_present = true;
					game.ship.x = vx;
					game.ship.y = vy;
					game.ship.dir_x = v1x - dx;
					game.ship.dir_y = v1y - dy;
					++shipdetect;
					break;
				}
			}
			else if (shipdetect == 1)
				shipdetect = 0;

			break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}   

}


void Asteroid::set(int x, int y, int type, int sf)
{
	this->x = x;
	this->y = y;
	this->type = type;
	this->sf = sf;
	this->is_tracked = false;
}

void Shot::set(int x, int y)
{
	this->x = x;
	this->y = y;
	this->is_tracked = false;
}


/* ****************************************************
	Implementation of KeysPacket
   **************************************************** */


KeysPacket::KeysPacket(void)
{
	signature[0] = 'c';
	signature[1] = 't';
	signature[2] = 'm';
	signature[3] = 'a';
	signature[4] = 'm';
	signature[5] = 'e';
	keys = '@';
	ping = 0;
}



void KeysPacket::clear(void)
{
	keys = '@';
}

void KeysPacket::hyperspace(bool b)
{
	if (b)
		keys |= KEY_HYPERSPACE;
	else
		keys &= ~KEY_HYPERSPACE;
}

void KeysPacket::fire(bool b)
{
	if (b)
		keys |= KEY_FIRE;
	else
		keys &= ~KEY_FIRE;
}

void KeysPacket::thrust(bool b)
{
	if (b)
		keys |= KEY_THRUST;
	else
		keys &= ~KEY_THRUST;
}

void KeysPacket::left(bool b)
{
	if (b)
	{
		keys |= KEY_LEFT;
		right(false);
	}
	else
		keys &= ~KEY_LEFT;
}

void KeysPacket::right(bool b)
{
	if (b)
	{
		keys |= KEY_RIGHT;
		left(false);
	}
	else
		keys &= ~KEY_RIGHT;
}

/* ****************************************************
	Implementation of Player
   **************************************************** */

void Player::ReceivePacket(FramePacket &packet)
{
	sockaddr_in sender;
	int sender_size = sizeof sender;
	fd_set readfds, writefds, exceptfds;

	do
	{
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		FD_SET(sd, &exceptfds);
		select(sd+1, &readfds, &writefds, &exceptfds, NULL);
		int bytes_received = recv(sd, (char *)&packet, sizeof packet, 0);
		if (bytes_received != sizeof packet)
		{
			int err = WSAGetLastError();
			fprintf(stderr, "Fehler %d bei recvfrom().\n", err);
			exit(1);
		}
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		select(sd+1, &readfds, &writefds, &exceptfds, &zero);
	} while(FD_ISSET(sd, &readfds));
}

void Player::SendPacket(KeysPacket &packet)
{
	sockaddr_in server;
	memset(&server, 0, sizeof server);
	server.sin_family = AF_INET;
	server.sin_port = htons(1979);
	server.sin_addr.s_addr = server_ip;
	if (sizeof packet != sendto(sd, (char *)&packet, sizeof packet, 0, (sockaddr*)&server, sizeof server))
	{
#if defined(WINDOWS)
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stderr, "Fehler %d bei sendto().\n", err);
			exit(1);
		}
#else
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}
#endif
	}
}
