#include "Tracked_Object.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define	PI	3.141592
#define ONE_DEGREE_IN_RADIANS				0.0175
#define SHIP_TURN_GRANULARITY_IN_DEGREES	5.625
#define SHIP_TURN_PER_FRAME_IN_RADIANS		(4.219 * PI / 180.0)



double angle_byte_to_degrees[256] =
{	0.0, 3.6, 8.2, 12.7, 16.5, 21.2, 25.2, 29.5,
	33.5, 37.8, 41.7, 46.2, 50.8, 55.3, 59.4, 63.4,
	67.5, 71.6, 76.1, 79.9, 84.5, 89.1, 92.8, 97.2, 
	101.8, 105.5, 110.2, 114.1, 118.1, 122.8, 126.7, 131.1, 
	135.0, 139.4, 143.3, 147.9, 151.9, 156.6, 160.7, 164.5, 
	169.2, 172.8, 177.3, 181.9, 185.5, 189.9, 194.5, 198.2, 
	203.0, 206.9, 211.0, 215.4, 220.0, 223.8, 227.5, 232.2, 
	235.9, 240.3, 245.1, 248.8, 252.7, 257.3, 261.1, 265.6, 
	270.0, 273.6, 278.2, 282.7, 286.5, 291.2, 295.0, 299.0, 
	303.5, 307.8, 311.7, 316.2, 320.0, 324.5, 328.6, 332.7, 
	336.6, 341.6, 345.3, 349.9, 354.5, 358.1, 2.8, 7.2, 
	11.0, 15.5, 19.3, 23.4, 28.1, 32.8, 36.7, 40.6, 
	45.0, 49.4, 53.3, 57.2, 61.9, 66.6, 70.7, 74.5, 
	79.0, 82.8, 87.2, 91.9, 95.5, 100.1, 104.7, 108.4, 
	113.4, 117.3, 121.4, 125.6, 130.0, 133.8, 138.3, 142.2, 
	146.5, 151.0, 155.0, 158.8, 163.6, 167.3, 171.8, 176.6, 
	180.0, 184.4, 188.9, 192.7, 197.3, 201.2, 205.0, 209.7, 
	214.2, 217.8, 222.5, 226.2, 230.0, 234.6, 239.0, 243.1, 
	247.0, 251.8, 255.5, 260.1, 264.5, 268.1, 272.7, 277.2, 
	280.8, 285.5, 289.3, 293.4, 298.1, 302.2, 306.7, 310.6, 
	315.0, 318.9, 323.3, 327.2, 331.9, 335.9, 339.8, 344.5, 
	348.2, 352.8, 357.2, 0.9, 5.5, 10.1, 13.9, 18.4, 
	22.5, 26.6, 30.6, 34.7, 39.2, 43.8, 48.3, 52.2,
	56.5, 60.5, 64.7, 68.8, 73.5, 77.3, 81.8, 86.4,
	90.0, 94.5, 98.9, 102.7, 107.3, 111.2, 115.3, 120.2,
	124.2, 127.8, 132.5, 136.1, 140.8, 145.3, 149.8, 153.9,
	157.9, 161.8, 166.3, 170.1, 174.3, 179.1, 182.9, 187.2,
	191.5, 195.6, 200.3, 204.1, 208.1, 212.1, 216.6, 221.0,
	225.0, 228.9, 233.3, 237.8, 241.9, 245.9, 249.8, 254.5,
	258.4, 262.8, 267.3, 270.9, 275.5, 279.9, 283.7, 288.2,
	292.1, 296.1, 300.2, 304.7, 309.2, 313.8, 317.5, 322.2,
	325.8, 329.8, 334.7, 338.8, 342.7, 347.2, 351.1, 355.5};

double effective_radius_from_angle_byte[256] =
{
1.000, 1.000, 1.002, 1.004, 1.010, 1.017, 1.022, 1.034, 
1.039, 1.052, 1.061, 1.078, 1.085, 1.106, 1.115, 1.101, 
1.094, 1.090, 1.080, 1.075, 1.070, 1.067, 1.065, 1.061, 
1.061, 1.059, 1.058, 1.058, 1.058, 1.058, 1.058, 1.058, 
1.055, 1.058, 1.058, 1.058, 1.058, 1.058, 1.058, 1.059, 
1.061, 1.061, 1.065, 1.067, 1.070, 1.075, 1.080, 1.090, 
1.094, 1.101, 1.115, 1.106, 1.085, 1.078, 1.061, 1.052, 
1.039, 1.034, 1.022, 1.017, 1.010, 1.004, 1.002, 1.000, 
1.000, 1.001, 1.003, 1.007, 1.010, 1.018, 1.028, 1.035, 
1.047, 1.054, 1.071, 1.079, 1.096, 1.108, 1.111, 1.103, 
1.092, 1.087, 1.082, 1.074, 1.073, 1.068, 1.066, 1.064, 
1.062, 1.062, 1.061, 1.061, 1.061, 1.061, 1.061, 1.061, 
1.058, 1.061, 1.061, 1.061, 1.061, 1.061, 1.061, 1.062, 
1.063, 1.065, 1.068, 1.071, 1.074, 1.080, 1.085, 1.093, 
1.100, 1.105, 1.115, 1.103, 1.085, 1.078, 1.060, 1.052, 
1.038, 1.033, 1.022, 1.017, 1.010, 1.004, 1.002, 1.000, 
1.000, 1.001, 1.002, 1.007, 1.010, 1.018, 1.028, 1.034, 
1.045, 1.054, 1.068, 1.079, 1.096, 1.104, 1.115, 1.107, 
1.097, 1.090, 1.087, 1.079, 1.076, 1.072, 1.069, 1.067, 
1.065, 1.064, 1.064, 1.064, 1.063, 1.064, 1.064, 1.063, 
1.061, 1.063, 1.064, 1.064, 1.063, 1.064, 1.064, 1.064, 
1.065, 1.067, 1.069, 1.072, 1.076, 1.079, 1.087, 1.090, 
1.097, 1.107, 1.115, 1.104, 1.096, 1.079, 1.068, 1.054, 
1.045, 1.034, 1.028, 1.018, 1.010, 1.007, 1.002, 1.001, 
1.000, 1.000, 1.002, 1.004, 1.010, 1.017, 1.022, 1.033, 
1.038, 1.052, 1.060, 1.078, 1.085, 1.103, 1.115, 1.105, 
1.100, 1.093, 1.085, 1.080, 1.074, 1.071, 1.068, 1.065, 
1.063, 1.062, 1.061, 1.061, 1.061, 1.061, 1.061, 1.061, 
1.058, 1.061, 1.061, 1.061, 1.061, 1.061, 1.061, 1.062, 
1.062, 1.064, 1.066, 1.068, 1.073, 1.074, 1.082, 1.087, 
1.092, 1.103, 1.111, 1.108, 1.096, 1.079, 1.071, 1.054, 
1.047, 1.035, 1.028, 1.018, 1.010, 1.007, 1.003, 1.001 };



Tracked_Object::Tracked_Object(void)
{
	m_radius = 0.0;
	m_velocity_known	= false;
	m_is_active			= false;
	m_velocity_known	= false;
	m_easy_prey			= false;
	m_pos.abs.set_xy(0.0, 0.0);
	m_pos.abs_predicted.set_xy(0.0, 0.0);
	m_vel.abs.set_xy(0.0, 0.0);
	m_vel.abs_averaged.set_xy(0.0, 0.0);
	m_dir.current.set_xy(0.0, 0.0);
	m_dir.last[0].set_xy(0.0, 0.0);
	m_dir.last[1].set_xy(0.0, 0.0);
	m_dir.last[2].set_xy(0.0, 0.0);
	m_dir.accurate.set_xy(0.0, 0.0);
	m_dir.target.set_xy(0.0, 0.0);
	m_dir.last_turn[0] = 0;
	m_dir.last_turn[1] = 0;
	m_dir.last_turn[2] = 0;
	m_dir.angle_byte = 0;
	m_time_to_hit = 0;
	m_incoming_shots = 0;
	m_shot_travel_time = 0.0;
	m_ship_turn_time = 0.0;
}

void Tracked_Object::set_inactive(void)
{
	m_radius = 0.0;
	m_velocity_known	= false;
	m_is_active			= false;
	m_velocity_known	= false;
	m_easy_prey			= false;
	m_pos.abs.set_xy(0.0, 0.0);
	m_pos.abs_predicted.set_xy(0.0, 0.0);
	m_vel.abs.set_xy(0.0, 0.0);
	m_vel.abs_averaged.set_xy(0.0, 0.0);
	m_dir.current.set_xy(0.0, 0.0);
	m_dir.last[0].set_xy(0.0, 0.0);
	m_dir.last[1].set_xy(0.0, 0.0);
	m_dir.last[2].set_xy(0.0, 0.0);
	m_dir.accurate.set_xy(0.0, 0.0);
	m_dir.target.set_xy(0.0, 0.0);
	m_dir.last_turn[0] = 0;
	m_dir.last_turn[1] = 0;
	m_dir.last_turn[2] = 0;
	m_dir.angle_byte = 0;
	m_time_to_hit = 0;
	m_incoming_shots = 0;
	m_shot_travel_time = 0.0;
	m_ship_turn_time = 0.0;
}

void Tracked_Object::track(Asteroid *p_asteroid)
{
	CVector screen_pos;

	screen_pos.set_xy(p_asteroid->x, p_asteroid->y);

//	New tracked object
	if(!m_is_active)
		{
		m_velocity_known = false;
		m_type = (Object_Types) p_asteroid->get_object_type();
		switch(p_asteroid->sf)
			{
			case 0:	// groer Asteroid
				m_radius = ASTEROID_RADIUS_LARGE;
				break;
			case 14:	// kleiner Asteroid
				m_radius = ASTEROID_RADIUS_SMALL;
				break;
			case 15:	// mittlerer Asteroid
				m_radius = ASTEROID_RADIUS_MEDIUM;
				break;
			}
		}
//	Already known object
	else
		{
		m_vel.abs = screen_pos - m_pos.abs;
		m_vel.abs.torus_wrap(TORUS_X, TORUS_Y);
		if(m_vel.abs.length() > 50.0)
			{	printf("Abnormal asteroid velocity after torus wrap. \n");	}
//	Initialize average velocity
		if(m_velocity_known == false)
			{	m_vel.abs_averaged = m_vel.abs;	}
		else
			{
			m_vel.abs_averaged = m_vel.abs_averaged * VELOCITY_AVERAGING_COEFFICIENT +
				m_vel.abs * (1.0 - VELOCITY_AVERAGING_COEFFICIENT);
			}
		m_velocity_known = true;
		}

	m_pos.abs				= screen_pos;
	m_is_updated			= true;
	m_is_active				= true;
	p_asteroid->is_tracked	= true;
}

void Tracked_Object::track(Shot *p_shot, int shot_idx, Tracked_Object &ship, Tracked_Object &saucer)
{
	CVector screen_pos;
	Tracked_Object *origin;
	double vel;

	screen_pos.set_xy(p_shot->x, p_shot->y);

//	New object
	if(!m_is_active)
		{
		m_pos.abs = screen_pos;
		m_velocity_known = false;
		m_radius = SHOT_RADIUS;

//		if(distance_from(ship) < distance_from(saucer))
		if((distance_from(ship) < 25.0) && (shot_idx < 4))
			{
			m_type = player_shot;
			origin = &ship;
			}
		else
			{
			m_type = saucer_shot;
			origin = &saucer;
			}
		m_vel.abs = m_pos.abs - origin->m_pos.abs;
		m_vel.abs.torus_wrap(TORUS_X, TORUS_Y);
		vel = m_vel.abs.length();
		m_vel.abs = m_vel.abs * (SHOT_VELOCITY / vel);
		if(origin->m_velocity_known)
			{	m_vel.abs = m_vel.abs + origin->m_vel.abs_averaged;	}
		m_vel.abs_averaged = m_vel.abs;
		}
//	Already known object
	else
		{
		m_vel.abs = screen_pos - m_pos.abs;
		m_pos.abs = screen_pos;
		m_vel.abs.torus_wrap(TORUS_X, TORUS_Y);
		m_vel.abs_averaged = m_vel.abs_averaged * VELOCITY_AVERAGING_COEFFICIENT +
			m_vel.abs * (1.0 - VELOCITY_AVERAGING_COEFFICIENT);
		}
	m_is_updated = true;
	m_is_active = true;
	m_velocity_known = true;
	p_shot->is_tracked = true;
}

void Tracked_Object::track(Saucer *p_saucer)
{
	CVector screen_pos;

	screen_pos.set_xy(p_saucer->x, p_saucer->y);

//	New tracked object
	if(!m_is_active)
		{
		m_velocity_known = false;
		m_type = (p_saucer->size == 15) ? saucer_large : saucer_small;
		m_radius = (p_saucer->size == 15) ? SAUCER_RADIUS_LARGE : SAUCER_RADIUS_SMALL;
		}
//	Already known object
	else
		{
		m_vel.abs = screen_pos - m_pos.abs;
		m_vel.abs.torus_wrap(TORUS_X, TORUS_Y);
//	Initialize average velocity
		if(m_velocity_known == false)
			{	m_vel.abs_averaged = m_vel.abs;	}
		else
			{
			m_vel.abs_averaged = m_vel.abs_averaged * VELOCITY_AVERAGING_COEFFICIENT +
				m_vel.abs * (1.0 - VELOCITY_AVERAGING_COEFFICIENT);
			}
		m_velocity_known = true;
		}

	m_pos.abs			= screen_pos;
	m_is_updated		= true;
	m_is_active			= true;
}

void Tracked_Object::track(Ship *p_ship)
{
	CVector screen_pos;

	screen_pos.set_xy(p_ship->x, p_ship->y);

//	New tracked object
	if(!m_is_active)
		{
		m_velocity_known = false;
		m_type = player_ship;
		m_radius = PLAYER_SHIP_RADIUS;
		}
//	Already known object
	else
		{
		m_vel.abs = screen_pos - m_pos.abs;
		m_vel.abs.torus_wrap(TORUS_X, TORUS_Y);
//	Initialize average velocity
		if(m_velocity_known == false)
			{	m_vel.abs_averaged = m_vel.abs;	}
		else
			{
			m_vel.abs_averaged = m_vel.abs_averaged * VELOCITY_AVERAGING_COEFFICIENT +
				m_vel.abs * (1.0 - VELOCITY_AVERAGING_COEFFICIENT);
			}
		m_velocity_known = true;
		}

	m_pos.abs			= screen_pos;
	m_dir.current.set_xy(p_ship->dir_x, p_ship->dir_y);
	m_dir.current.normalize();
	m_is_updated			= true;
	m_is_active			= true;
}

bool Tracked_Object::enough_shots(void)
{
	if(m_type == saucer_small)
		{
		if(m_incoming_shots > 0)
			{	return(true);	}
		return(false);
		}
	if(m_type == saucer_large)
		{
		if(m_incoming_shots > 0)
			{	return(true);	}
		return(false);
		}

	if((m_radius <= ASTEROID_RADIUS_SMALL) && (m_incoming_shots > 0))
		{	return(true);	}
	if((m_radius <= ASTEROID_RADIUS_MEDIUM) && (m_incoming_shots > 1))
		{	return(true);	}
	if(m_incoming_shots > 2)
	{	return(true);	}
	return(false);
}

bool Tracked_Object::can_hit(Tracked_Object *target)
{
	double angle_diff, angle_hit;

	angle_diff	= abs(m_dir.predicted.angle_to(target->m_dir.target));
	angle_hit	= atan((target->m_radius * effective_radius_from_angle_byte[m_dir.angle_byte_predicted])/ 
				target->m_pos.intercept.distance_to(m_pos.abs_predicted));
	angle_hit	= abs(angle_hit);
	angle_hit	= ACCURACY_FACTOR * angle_hit;

	if(angle_diff <= angle_hit)
	{	return(true);	}
	return(false);
}

double Tracked_Object::current_distance_from(Tracked_Object &other_object)
{
	CVector diff;

	diff = other_object.m_pos.abs - m_pos.abs;
	diff.torus_wrap(TORUS_X, TORUS_Y);
	return(diff.length());
}

double Tracked_Object::current_distance_from(double x0, double y0)
{
	CVector diff;

	diff.set_xy(x0, y0);
	diff = diff - m_pos.abs;
	diff.torus_wrap(TORUS_X, TORUS_Y);

	return(diff.length());
}

double Tracked_Object::distance_from(Tracked_Object &other_object)
{
	CVector dist_vec;

	dist_vec = other_object.m_pos.abs - m_pos.abs;
	dist_vec.torus_wrap(TORUS_X, TORUS_Y);

	return(dist_vec.length());
}
double Tracked_Object::distance_from(double x0, double y0)
{
	CVector dist_vec;

	dist_vec.set_xy(x0, y0);
	dist_vec = dist_vec - m_pos.abs;
	dist_vec.torus_wrap(TORUS_X, TORUS_Y);

	return(dist_vec.length());
}

double Tracked_Object::minimum_distance_from(Tracked_Object &other_object)
{
	CVector rel_pos, rel_vel, closest_pos;

	if(!m_is_active)
		{	return(0.0);	}

	rel_pos = m_pos.abs - other_object.m_pos.abs;
	rel_pos.torus_wrap(TORUS_X, TORUS_Y);

	rel_vel = m_vel.abs_averaged - other_object.m_vel.abs_averaged;

	closest_pos.set_x(rel_vel.get_y() * rel_pos.cross_prod(rel_vel) / rel_vel.dot_prod(rel_vel));
	closest_pos.set_y(- rel_vel.get_x() * rel_pos.cross_prod(rel_vel) / rel_vel.dot_prod(rel_vel));
	closest_pos.torus_wrap(TORUS_X, TORUS_Y);

	return(closest_pos.length());
}

double Tracked_Object::time_to_minimum_distance_from(Tracked_Object &other_object)
{
	CVector rel_vel;	

	if(!m_is_active)
		{	return(0.0);	}

	rel_vel = m_vel.abs_averaged - other_object.m_vel.abs_averaged;

//	avoid division by zero
	if(rel_vel.length() == 0.0)
		{	return(0.0);	}

	return((distance_from(other_object) - minimum_distance_from(other_object)) / rel_vel.length());
}


double Tracked_Object::predicted_distance_from(Tracked_Object &other_object)
{
	CVector dist_vec;

	dist_vec = other_object.m_pos.abs_predicted - m_pos.abs_predicted;
	dist_vec.torus_wrap(TORUS_X, TORUS_Y);

	return(dist_vec.length());
}

double Tracked_Object::predicted_distance_from(double x0, double y0)
{
	CVector dist_vec;

	dist_vec.set_xy(x0, y0);
	dist_vec = dist_vec - m_pos.abs_predicted;
	dist_vec.torus_wrap(TORUS_X, TORUS_Y);

	return(dist_vec.length());
}

void Tracked_Object::predict_position(void)
{
	if(!m_is_active)
		{	return;	}

	m_pos.abs_predicted = m_pos.abs;
	if(m_velocity_known)
		{	m_pos.abs_predicted = m_pos.abs_predicted + m_vel.abs_averaged;	}
}

int map_angle_byte_to_displayed_orientation[256] =
{0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 
4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 
8, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10, 10, 11, 11, 11, 11, 
12, 12, 12, 12, 13, 13, 13, 13, 14, 14, 14, 14, 15, 15, 15, 15, 
16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 
20, 21, 21, 21, 21, 22, 22, 22, 22, 23, 23, 23, 23, 24, 24, 24, 
24, 25, 25, 25, 25, 26, 26, 26, 26, 27, 27, 27, 27, 28, 28, 28, 
28, 29, 29, 29, 29, 30, 30, 30, 30, 31, 31, 31, 31, 32, 32, 32, 
32, 32, 32, 32, 33, 33, 33, 33, 34, 34, 34, 34, 35, 35, 35, 35, 
36, 36, 36, 36, 37, 37, 37, 37, 38, 38, 38, 38, 39, 39, 39, 39, 
40, 40, 40, 40, 41, 41, 41, 41, 42, 42, 42, 42, 43, 43, 43, 43, 
44, 44, 44, 44, 45, 45, 45, 45, 46, 46, 46, 46, 47, 47, 47, 47, 
48, 49, 49, 49, 49, 50, 50, 50, 50, 51, 51, 51, 51, 52, 52, 52, 
52, 53, 53, 53, 53, 54, 54, 54, 54, 55, 55, 55, 55, 56, 56, 56, 
56, 57, 57, 57, 57, 58, 58, 58, 58, 59, 59, 59, 59, 60, 60, 60, 
60, 61, 61, 61, 61, 62, 62, 62, 62, 63, 63, 63, 63, 0, 0, 0};

double angle_byte_to_angle_old[256] = { 0.0, 0.90938, 2.72631, 3.63295, 5.44, 7.125, 8.1301, 10.0607, 
11.0, 12.7, 13.9, 15.5, 16.5, 18.4, 19.3, 21.2, 
22.5, 23.4, 25.2, 26.6, 28.1, 29.5, 30.6, 32.8, 
33.5, 34.7, 36.7, 37.8, 39.2, 40.6, 41.7, 43.8, 
45.0, 46.2, 48.3, 49.4, 50.8, 52.2, 53.3, 55.3, 
56.5, 57.2, 59.4, 60.5, 61.9, 63.4, 64.7, 66.6, 
67.5, 68.8, 70.7, 71.6, 73.5, 74.5, 76.1, 77.3, 
79.0, 79.9, 81.8, 82.8, 84.5, 86.4, 87.2, 89.1, 
90.0, 91.9, 92.8, 94.5, 95.5, 97.2, 98.9, 100.1, 
101.8, 102.7, 104.7, 105.5, 107.3, 108.4, 110.2, 111.2, 
113.4, 114.1, 115.3, 117.3, 118.1, 120.2, 121.4, 122.8, 
124.2, 125.6, 126.7, 127.8, 130.0, 131.1, 132.5, 133.8, 
135.0, 136.1, 138.3, 139.4, 140.8, 142.2, 143.3, 145.3, 
146.5, 147.9, 149.8, 151.0, 151.9, 153.9, 155.0, 156.6, 
157.9, 158.8, 160.7, 161.8, 163.6, 164.5, 166.3, 167.3, 
169.2, 170.1, 171.8, 172.8, 174.3, 176.6, 177.3, 179.1, 
180.0, 181.9, 182.9, 184.4, 185.5, 187.2, 188.9, 189.9, 
191.5, 192.7, 194.5, 195.6, 197.3, 198.2, 200.3, 201.2, 
203.0, 204.1, 205.0, 206.9, 208.1, 209.7, 211.0, 212.1, 
214.2, 215.4, 216.6, 217.8, 220.0, 221.0, 222.5, 223.8, 
225.0, 226.2, 227.5, 228.9, 230.0, 232.2, 233.3, 234.6, 
235.9, 237.8, 239.0, 240.3, 241.9, 243.1, 245.1, 245.9, 
247.0, 248.8, 249.8, 251.8, 252.7, 254.5, 255.5, 257.3, 
258.4, 260.1, 261.1, 262.8, 264.5, 265.6, 267.3, 268.1, 
270.0, 270.9, 272.7, 273.6, 275.5, 277.2, 278.2, 279.9, 
280.8, 282.7, 283.7, 285.5, 286.5, 288.2, 289.3, 291.2, 
292.1, 293.4, 295.0, 296.1, 298.1, 299.0, 300.2, 302.2, 
303.5, 304.7, 306.7, 307.8, 309.2, 310.6, 311.7, 313.8, 
315.0, 316.2, 317.5, 318.9, 320.0, 322.2, 323.3, 324.5, 
325.8, 327.2, 328.6, 329.8, 331.9, 332.7, 334.7, 335.9, 
336.6, 338.8, 339.8, 341.6, 342.7, 344.5, 345.3, 347.2, 
348.2, 349.9, 351.1, 352.8, 354.5, 355.5, 357.2, 358.1};

double angle_byte_to_angle[256] = 
{
0.0000, 0.9094, 2.7263, 3.6330, 5.4403, 7.2369, 8.1301, 10.0607,
10.9541, 12.7244, 13.8150, 15.5725, 16.4404, 18.4349, 19.2900, 21.2974,
22.4794, 23.3177, 25.3462, 26.5651, 28.1786, 29.4072, 30.6507, 32.6806,
33.4399, 34.6952, 36.6897, 37.9542, 39.2257, 40.5028, 41.7845, 43.7270,
45.0000, 46.2730, 48.2155, 49.4972, 50.7743, 52.0458, 53.3103, 55.3048,
56.5601, 57.3194, 59.3493, 60.5928, 61.8214, 63.4349, 64.6538, 66.6823,
67.5206, 68.7026, 70.7100, 71.5651, 73.5596, 74.4275, 76.1850, 77.2756,
79.0459, 79.9393, 81.8699, 82.7631, 84.5597, 86.3670, 87.2737, 89.0906,
90.0000, 91.8183, 92.7263, 94.5378, 95.4403, 97.2369, 99.0193, 100.0607,
101.8421, 102.7244, 104.6973, 105.5725, 107.3005, 108.4349, 110.1363, 111.2974,
113.3177, 114.1455, 115.3462, 117.3777, 118.1786, 120.1916, 121.4296, 122.6806,
124.1861, 125.4333, 126.6897, 127.9542, 129.9204, 131.1859, 132.4552, 133.7270,
135.0000, 136.2730, 138.2155, 139.4972, 140.7743, 142.0458, 143.3103, 145.3048,
146.5601, 147.8043, 149.8084, 151.0323, 151.8214, 153.8384, 155.0372, 156.6823,
157.8645, 158.7026, 160.7100, 161.8473, 163.5596, 164.4275, 166.3995, 167.2756,
169.2157, 170.0958, 171.8699, 172.7631, 174.5597, 176.4237, 177.3162, 179.1048,
180.0000, 181.7899, 182.6838, 184.4672, 185.4403, 187.2369, 189.0193, 189.9042,
191.6593, 192.7244, 194.4703, 195.5725, 197.3005, 198.1527, 200.1363, 201.2974,
202.9638, 204.1455, 204.9628, 206.9657, 208.1786, 209.7449, 210.9638, 212.1957,
214.1861, 215.4333, 216.6897, 217.9542, 219.9204, 221.1859, 222.4552, 223.7270,
225.0000, 226.2730, 227.5448, 228.8141, 230.0796, 232.0458, 233.3103, 234.5667,
235.8139, 237.8043, 239.0362, 240.2551, 241.8214, 243.0343, 245.0372, 245.8545,
247.0362, 248.7026, 249.8637, 251.8473, 252.6995, 254.4275, 255.5297, 257.2756,
258.3407, 260.0958, 260.9807, 262.7631, 264.5597, 265.5328, 267.3162, 268.2101,
270.0000, 270.8952, 272.6838, 273.5763, 275.4403, 277.2369, 278.1301, 279.9042,
280.7843, 282.7244, 283.6005, 285.5725, 286.4404, 288.1527, 289.2900, 291.2974,
292.1355, 293.3177, 294.9628, 296.1616, 298.1786, 298.9677, 300.1916, 302.1957,
303.4399, 304.6952, 306.6897, 307.9542, 309.2257, 310.5028, 311.7845, 313.7270,
315.0000, 316.2730, 317.5448, 318.8141, 320.0796, 322.0458, 323.3103, 324.5667,
325.8139, 327.3194, 328.5704, 329.8084, 331.8214, 332.6223, 334.6538, 335.8545,
336.6823, 338.7026, 339.8637, 341.5651, 342.6995, 344.4275, 345.3027, 347.2756,
348.1579, 349.9393, 350.9807, 352.7631, 354.5597, 355.4622, 357.2737, 358.1817,
};

void Tracked_Object::update_angle(void)
{
	int index_display, predicted_angle_byte, predicted_index_display;
	double angle_display;
	double accurate_angle;

// 1. Bestimme Ausrichtungsindex (0-64) aus angezeigter
// Blickrichtung (dx, dy).
	angle_display = atan2(m_dir.current.get_y(), m_dir.current.get_x());
	angle_display = angle_display * (360.0 / (2 * 3.141592));
	angle_display = angle_display * (64.0 / 360.0);
	angle_display = floor(angle_display + 0.5);
	index_display = (int) angle_display;
	index_display = index_display & 0x0000003F;

// 2. Prfe, ob Ausrichtungsindex mit dem berechneten
// (letztes Winkelbyte + (-3/0/3)) bereinstimmt.
	predicted_angle_byte = m_dir.angle_byte;
	if(m_dir.last_turn[1] == LEFT_TURN)
		{	predicted_angle_byte += 3;	}
	else if(m_dir.last_turn[1] == RIGHT_TURN)
		{	predicted_angle_byte -= 3;	}
	predicted_angle_byte = predicted_angle_byte & 0x000000FF;
	predicted_index_display = map_angle_byte_to_displayed_orientation[predicted_angle_byte];

// 3. Wenn nein: Passe internes Winkelbyte an
	if(index_display != predicted_index_display)
		{
		if(abs(index_display - predicted_index_display) > 1)
			{
				printf("ANGLE_BYTE: Complete resync.\n");
				predicted_angle_byte = 4 * index_display;
			}
		else
			{
			while(map_angle_byte_to_displayed_orientation[predicted_angle_byte] != index_display)
				{
				printf("ANGLE_BYTE: Small resync.\n");
				predicted_angle_byte += (index_display - predicted_index_display);
				predicted_angle_byte = predicted_angle_byte & 0x000000FF;
				}
			}
		}
	m_dir.angle_byte = predicted_angle_byte;
// 4. Berechne "genaue" Blickrichtung aus internem Winkelbyte
	accurate_angle = angle_byte_to_angle[m_dir.angle_byte];
	if(accurate_angle > 180.0)
	{	accurate_angle = accurate_angle - 360.0;	}
	accurate_angle = accurate_angle * ((2 * 3.141592) / 360.0);

	m_dir.accurate.set_x(cos(accurate_angle));
	m_dir.accurate.set_y(sin(accurate_angle));

// 5. Nun Update der vorhergesagten richtung
	predicted_angle_byte = m_dir.angle_byte;
	if(m_dir.last_turn[0] == LEFT_TURN)
		{	predicted_angle_byte += 3;	}
	else if(m_dir.last_turn[0] == RIGHT_TURN)
		{	predicted_angle_byte -= 3;	}
	predicted_angle_byte = predicted_angle_byte & 0x000000FF;
	m_dir.angle_byte_predicted = predicted_angle_byte;

	accurate_angle = angle_byte_to_angle[predicted_angle_byte];
	if(accurate_angle > 180.0)
	{	accurate_angle = accurate_angle - 360.0;	}
	accurate_angle = accurate_angle * PI / 180.0;

	m_dir.predicted.set_x(cos(accurate_angle));
	m_dir.predicted.set_y(sin(accurate_angle));
//	6. update buffers

	m_dir.last_turn[2] = m_dir.last_turn[1];
	m_dir.last_turn[1] = m_dir.last_turn[0];

	m_dir.last[2] = m_dir.last[1];
	m_dir.last[1] = m_dir.last[0];

	m_dir.last[0] = m_dir.current;
}
/*
bool Tracked_Object::sim_firing_solution(Tracked_Object *target)
{
	int angle_byte_tmp, n_sim;
	CVector rel_pos, rel_vel, intercept_pos, shot_vel;
	double angle_diff;

	rel_pos = target->m_pos.abs_predicted - m_pos.abs_predicted;
	rel_pos.torus_wrap(TORUS_X, TORUS_Y);
	rel_vel = target->m_vel.abs_averaged - m_vel.abs_averaged;
	angle_byte_tmp = m_dir.angle_byte;

	for(n_sim = 0; n_sim < 60; n_sim++)
		{
		angle_diff = angle_byte_to_angle[angle_byte_tmp];
		angle_diff = rel_pos.angle_abs() * 180.0 / PI;
		while(angle_diff > 180.0)
		{	angle_diff -= 360.0;	}
		while(angle_diff <= -180.0)
		{	angle_diff +=	360.0;	}
		angle_hit = ACCURACY_FACTOR * abs(atan(target->m_radius / (t_var.t_fly * SHOT_VELOCITY)));
		angle_hit = angle_hit * 180 / PI;


		angle_diff

		}
}
*/

void Tracked_Object::backup_firing_solution_to(Tracked_Object *target)
{
	CVector rel_pos, rel_vel, intercept_pos;
	CVector ship_pos, ship_angle, target_pos, target_angle;
	double a, b, c, radikand, z1, z2, t1, t2;
	double step_turn_time, shot_travel_time;
	bool calculation_done = false;
	int n_iteration = 0;

	ship_pos = m_pos.abs_predicted;
	target_pos = target->m_pos.abs_predicted;
	ship_angle = m_dir.predicted;

//	rel_pos = m_pos.abs_predicted - target->m_pos.abs_predicted;
	while(calculation_done == false)
		{
		rel_pos = target_pos - ship_pos;
		rel_pos.torus_wrap(TORUS_X, TORUS_Y);

		rel_vel = target->m_vel.abs_averaged - m_vel.abs_averaged;
		rel_pos.torus_wrap(TORUS_X, TORUS_Y);

		a = rel_pos.dot_prod(rel_pos);
		b = - 2 * rel_pos.dot_prod(rel_vel);
		c = rel_vel.dot_prod(rel_vel) - (SHOT_VELOCITY * SHOT_VELOCITY);
		radikand = (b * b) - (4 * a * c);

//	Solve polynmial
		if(	(a == 0) || (radikand < 0) ) //	Should never happen. Point ship directly at target
			{
			printf("ERROR 1 in FIRING SOLUTION !!!! \n");
			shot_travel_time = rel_pos.length() / SHOT_VELOCITY;
			target_angle = rel_pos;
			calculation_done = true;
			}
		else					// one or two solutions
			{
			z1 =  (- b + sqrt(radikand)) / (2 * a);
			z2 =  (- b - sqrt(radikand)) / (2 * a);
			t1 = 1 / z1;
			t2 = 1 / z2;
			if((t1 <= 0) && (t2 <= 0))	// Should never happen. Point ship directly at target
				{
				printf("ERROR 2 in FIRING SOLUTION !!!! \n");
				shot_travel_time = rel_pos.length() / SHOT_VELOCITY;
				target_angle = rel_pos;
				calculation_done = true;
				}
			else
				{
				if((t1 < 0) && (t2 > 0))
					{	t1 = t2;	}
				else if((t1 > t2) && (t2 > 0))
					{	t1 = t2;	}
//		TODO: Calculate point of interception, WRAP the point of interception,
//		point ship at wrapped point
				intercept_pos = target->m_pos.abs_predicted + target->m_vel.abs_averaged * t1;
				intercept_pos.torus_wrap_absolute();
				target_angle = intercept_pos - m_pos.abs_predicted;
				target_angle.torus_wrap(TORUS_X, TORUS_Y);
				shot_travel_time = t1;
				}
			}
//	Calculate turn_time,
		target_angle.normalize();
		step_turn_time = abs(ship_angle.angle_to(target_angle)) / SHIP_TURN_PER_FRAME_IN_RADIANS;
		target_pos = target_pos + target->m_vel.abs_averaged * step_turn_time;
		ship_pos = ship_pos + m_vel.abs_averaged * step_turn_time;
		ship_angle = target_angle;
		target_pos.torus_wrap_absolute();
		ship_pos.torus_wrap_absolute();
		if(step_turn_time < 1.0)
		{	calculation_done = true;	}
		if(n_iteration > 4)
		{	calculation_done = true;	}
		n_iteration++;
		}

	m_dir.target = target_angle;
	m_shot_travel_time	=	shot_travel_time;
	m_ship_turn_time	=	step_turn_time;
	m_pos.intercept = intercept_pos;
    target->m_dir.target =	target_angle;
	target->m_shot_travel_time = shot_travel_time;
	target->m_ship_turn_time = step_turn_time;
	target->m_pos.intercept = intercept_pos;
	update_easy_prey(target);
}

double dot_prod(double x1, double y1, double x2, double y2);
double cross_prod(double x1, double y1, double x2, double y2);
double v_length(double x1, double y1);

#define MAX_NEWTON_ITERATIONS	30

void Tracked_Object::old_newton_firing_solution_to(Tracked_Object *target)
{
    t_var_struct t_var, t_var_tmp;
    t_const_struct t_con;
	t_var_struct delta_t_var;


    double trgt_dist;

	bool continue_iteration = true, wraparound = false;
	int iteration_count = 0; 
	static int statistics[50] = {	0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
									0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
									0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
									0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
									0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	static int t_greater_zero = 0, t_lessthan_zero= 0;

	CVector rel_pos, rel_vel, intercept_pos, shot_vel;

	backup_firing_solution_to(target);

	rel_pos = target->m_pos.abs_predicted - m_pos.abs_predicted;
	rel_pos.torus_wrap(TORUS_X, TORUS_Y);
	rel_vel = target->m_vel.abs_averaged - m_vel.abs_averaged;

	for(;;)
		{
		t_con.x_ship  = 0.0;
		t_con.y_ship  = 0.0;
		t_con.fx_ship = m_dir.predicted.get_x();
		t_con.fy_ship = m_dir.predicted.get_y();
		t_con.x_trgt  = rel_pos.get_x();
		t_con.y_trgt  = rel_pos.get_y();
		t_con.vx_trgt = rel_vel.get_x();
		t_con.vy_trgt = rel_vel.get_y();
		t_con.v_shot  = SHOT_VELOCITY;
		t_con.r_turn  = SHIP_TURN_PER_FRAME_IN_RADIANS;

//    Initialize variables
	    trgt_dist = v_length((t_con.x_trgt - t_con.x_ship), (t_con.y_trgt - t_con.y_ship));

		t_var.x_shot = t_con.v_shot * (t_con.x_trgt - t_con.x_ship) / trgt_dist;
		t_var.y_shot = t_con.v_shot * (t_con.y_trgt - t_con.y_ship) / trgt_dist;
		t_var.t_turn = acos(dot_prod((t_con.x_trgt - t_con.x_ship), (t_con.y_trgt - t_con.y_ship),
								t_con.fx_ship, t_con.fy_ship) / trgt_dist) / t_con.r_turn;
		t_var.t_fly = trgt_dist / t_con.v_shot;

// Newton iteration:
		while(continue_iteration)
			{
			if(!newton_iteration(t_var, t_con, delta_t_var))
			{	printf("Newton iteration failed.\n");	}
// 5. Check for exit conditions
			iteration_count++;
			if( (abs(delta_t_var.x_shot) < 0.00001) &&
				(abs(delta_t_var.y_shot) < 0.00001) &&
				(abs(delta_t_var.t_turn) < 0.0001) &&
				(abs(delta_t_var.t_fly) < 0.0001)    )
				{    continue_iteration = false;    }
			if(iteration_count > 39)
				{	continue_iteration = false;	}
			}
		statistics[iteration_count - 1]++;
		if(iteration_count > 29)
			{
			printf("NEWTON_ALGO: Iteration limit exceeded !!!\n");
			backup_firing_solution_to(target);
			return;
			}

		if(wraparound == true)
		{	break;	}
		shot_vel.set_xy(t_var.x_shot, t_var.y_shot);
		intercept_pos = m_pos.abs_predicted + 
						m_vel.abs_averaged * (t_var.t_fly + t_var.t_turn) +
						shot_vel * t_var.t_fly;
		if(intercept_pos.get_x() > 1023)
			{	
			rel_pos.set_x(rel_pos.get_x() - 1024.0);
			wraparound = true;
			}
		else if(intercept_pos.get_x() < 0)
			{
			rel_pos.set_x(rel_pos.get_x() + 1024.0);
			wraparound = true;
			}
		if(intercept_pos.get_y() > 895)
			{
			rel_pos.set_y(rel_pos.get_y() + 768.0);
			wraparound = true;
			}
		else if(intercept_pos.get_y() < 128)
			{
			rel_pos.set_y(rel_pos.get_y() - 768.0);
			wraparound = true;
			}
		if(wraparound == false)
			{	break;	}
		else
			{
			t_var_tmp.x_shot	= t_var.x_shot;
			t_var_tmp.y_shot	= t_var.y_shot;
			t_var_tmp.t_turn	= t_var.t_turn;
			t_var_tmp.t_fly		= t_var.t_fly;
			}
		}
	if(wraparound == true)
		{
		if((t_var_tmp.t_turn + t_var_tmp.t_fly) < (t_var.t_turn + t_var.t_fly) )
			{
			t_var.x_shot	= t_var_tmp.x_shot;
			t_var.y_shot	= t_var_tmp.y_shot;
			t_var.t_turn	= t_var_tmp.t_turn;
			t_var.t_fly		= t_var_tmp.t_fly;
			}
		}
	if(t_var.t_turn > 200.0 || t_var.t_fly > 2000.0)
		{
		backup_firing_solution_to(target);
		printf("NEWTON_ALGO: t_fly or t_turn exceed limits !!!\n");
		return;
		}
	if(t_var.t_turn < 0 || t_var.t_fly < 0)
		{
		t_lessthan_zero++;	
		backup_firing_solution_to(target);
		printf("NEWTON_ALGO: time less than zero !!!\n");
		return;
		}
	else
		{	t_greater_zero++;	}


	intercept_pos = m_pos.abs_predicted + 
					m_vel.abs_averaged * (t_var.t_fly + t_var.t_turn) +
					shot_vel * t_var.t_fly;

	m_dir.target.set_xy(t_var.x_shot, t_var.y_shot);
	m_dir.target.normalize();
	m_shot_travel_time	=	t_var.t_fly;
	m_ship_turn_time	=	t_var.t_turn;
	m_pos.intercept = intercept_pos;
    target->m_dir.target.set_xy(t_var.x_shot, t_var.y_shot);
	target->m_dir.target.normalize();
	target->m_shot_travel_time = t_var.t_fly;
	target->m_ship_turn_time = t_var.t_turn;
	target->m_pos.intercept = intercept_pos;
	update_easy_prey(target);
}

void Tracked_Object::newton_firing_solution_to(Tracked_Object *target)
{
    t_var_struct t_var[4][4];
	t_var_struct *final_result;
	int n_iterations[4][4];
    t_const_struct t_con;
	t_var_struct delta_t_var;
	int iteration_count, wrap_idx, shot_idx;

    double trgt_dist;

	bool continue_iteration;

	CVector rel_pos[4], shot_init[4], rel_vel, intercept_pos, shot_vel;

	rel_vel = target->m_vel.abs_averaged - m_vel.abs_averaged;
	rel_pos[0] = target->m_pos.abs_predicted - m_pos.abs_predicted;
	rel_pos[0].torus_wrap(TORUS_X, TORUS_Y);
	rel_pos[1] = rel_pos[0];
	rel_pos[2] = rel_pos[0];
	rel_pos[3] = rel_pos[0];

//	Calculate the three nearest wraparound positions
	if(rel_pos[0].get_x() > 0)
		{	
		rel_pos[1].set_x(rel_pos[1].get_x() - 1024.0);
		rel_pos[3].set_x(rel_pos[3].get_x() - 1024.0);
		}
	else
		{	
		rel_pos[1].set_x(rel_pos[1].get_x() + 1024.0);
		rel_pos[3].set_x(rel_pos[3].get_x() + 1024.0);
		}

	if(rel_pos[0].get_y() > 0)
		{	
		rel_pos[2].set_y(rel_pos[2].get_y() - 768.0);
		rel_pos[3].set_y(rel_pos[3].get_y() - 768.0);
		}
	else
		{	
		rel_pos[2].set_y(rel_pos[2].get_y() + 768.0);
		rel_pos[3].set_y(rel_pos[3].get_y() + 768.0);
		}

	for(wrap_idx = 0; wrap_idx < 4; wrap_idx++)
		{
//	Calculate 4 initial conditions for shot velocity
		shot_init[0] = rel_pos[wrap_idx];
		shot_init[0].normalize();
		shot_init[0] = shot_init[0] * SHOT_VELOCITY;
		shot_init[1] = shot_init[0];
		shot_init[2] = shot_init[0];
		shot_init[3] = shot_init[0];
		shot_init[1].set_x(- shot_init[1].get_x());
		shot_init[3].set_x(- shot_init[3].get_x());
		shot_init[2].set_y(- shot_init[2].get_y());
		shot_init[3].set_y(- shot_init[3].get_y());

		for(shot_idx = 0; shot_idx < 4; shot_idx++)
			{
// Initialize constants
			t_con.x_ship  = 0.0;	
			t_con.y_ship  = 0.0;
			t_con.fx_ship = m_dir.predicted.get_x();
			t_con.fy_ship = m_dir.predicted.get_y();
			t_con.x_trgt  = rel_pos[wrap_idx].get_x();
			t_con.y_trgt  = rel_pos[wrap_idx].get_y();
			t_con.vx_trgt = rel_vel.get_x();
			t_con.vy_trgt = rel_vel.get_y();
			t_con.v_shot  = SHOT_VELOCITY;
			t_con.r_turn  = SHIP_TURN_PER_FRAME_IN_RADIANS;
//	Initialize variables
		    trgt_dist = rel_pos[wrap_idx].length();
			t_var[wrap_idx][shot_idx].x_shot = shot_init[shot_idx].get_x();
			t_var[wrap_idx][shot_idx].y_shot = shot_init[shot_idx].get_y();
			t_var[wrap_idx][shot_idx].t_turn = abs(m_dir.predicted.angle_to(shot_init[shot_idx])) /
												SHIP_TURN_PER_FRAME_IN_RADIANS;
			t_var[wrap_idx][shot_idx].t_fly = trgt_dist / t_con.v_shot;
//	Newton iteration
			iteration_count = 0;
			continue_iteration = true;
			while(continue_iteration)
				{
				if(!newton_iteration(t_var[wrap_idx][shot_idx], t_con, delta_t_var))
				{	
					printf("Newton iteration %i failed.\n", iteration_count);
					iteration_count = MAX_NEWTON_ITERATIONS;
					continue_iteration = false;
				}
				iteration_count++;
				if( (abs(delta_t_var.x_shot) < 0.000001) &&
					(abs(delta_t_var.y_shot) < 0.000001) &&
					(abs(delta_t_var.t_turn) < 0.00001) &&
					(abs(delta_t_var.t_fly) < 0.00001)    )
					{    continue_iteration = false;    }
				if(iteration_count >= MAX_NEWTON_ITERATIONS)
					{	continue_iteration = false;	}
				}
			n_iterations[wrap_idx][shot_idx] = iteration_count;
			}
		}	

	final_result = &t_var[0][0];
	for(wrap_idx = 0; wrap_idx < 4; wrap_idx++)
		{
		for(shot_idx = 0; shot_idx < 4; shot_idx++)
			{
			if(	(t_var[wrap_idx][shot_idx].t_turn < 0) ||
				(t_var[wrap_idx][shot_idx].t_fly < 0)	)
				{	continue;	}
			if(		(final_result->t_turn < 0) ||
					(final_result->t_fly < 0)	)
				{	final_result = &t_var[wrap_idx][shot_idx];	}
			if(	(n_iterations[wrap_idx][shot_idx] < MAX_NEWTON_ITERATIONS) &&
				(t_var[wrap_idx][shot_idx].t_fly <= 69.0) &&
				((t_var[wrap_idx][shot_idx].t_fly + t_var[wrap_idx][shot_idx].t_turn) <
				 (final_result->t_fly + final_result->t_turn) ) )
				{	final_result = &t_var[wrap_idx][shot_idx];	}
			}
		}
	if(	(final_result->t_turn < 0) ||
		(final_result->t_fly < 0)	)
		{
		printf("NEWTON_ALGO: No firing solution with positive t found.\n");
		backup_firing_solution_to(target);
		return;
		}


	shot_vel.set_xy(final_result->x_shot, final_result->y_shot);
	intercept_pos = m_pos.abs_predicted + 
					m_vel.abs_averaged * (final_result->t_fly + final_result->t_turn) +
					shot_vel * final_result->t_fly;
	m_dir.target = shot_vel;;
	m_dir.target.normalize();
	m_shot_travel_time	=	final_result->t_fly;
	m_ship_turn_time	=	final_result->t_turn;
	m_pos.intercept = intercept_pos;
    target->m_dir.target = shot_vel;
	target->m_dir.target.normalize();
	target->m_shot_travel_time = final_result->t_fly;
	target->m_ship_turn_time = final_result->t_turn;
	target->m_pos.intercept = intercept_pos;
	update_easy_prey(target);
}

void Tracked_Object::update_easy_prey(Tracked_Object *target)
{
	int angle_byte_tmp, angle_byte_inc;
	double angle_hit, target_angle, angle_diff, angle_diff_old;

	//	Check if angle can be easily reached (angle byte)
	m_easy_prey				= false;
	target->m_easy_prey		= false;
// Calculate angle necessary to hit and allowed tolerance
	angle_hit = ACCURACY_FACTOR * 
		abs(atan((target->m_radius * effective_radius_from_angle_byte[m_dir.angle_byte_predicted])
		/ (target->m_shot_travel_time * SHOT_VELOCITY)));
	angle_hit = angle_hit * 180 / PI;
	target_angle = m_dir.target.angle_abs() * 180.0 / PI;
//	Check if target can be reached in < 1/2 turn
	if(m_dir.predicted.angle_to(m_dir.target) >= 0)
		{	angle_byte_inc = 3;		}
	else
		{	angle_byte_inc = -3;	}
	angle_byte_tmp	= m_dir.angle_byte_predicted;
	angle_diff		= angle_byte_to_angle[angle_byte_tmp];
	if(angle_diff > 180.0)
	{	angle_diff = angle_diff - 360.0;	}
	angle_diff = angle_diff - target_angle;
	if(abs(angle_diff) <= angle_hit)
		{
		m_easy_prey				= true;
		target->m_easy_prey		= true;
		}
	while(angle_byte_tmp != m_dir.angle_byte)
		{
		angle_diff_old = angle_diff;
		angle_byte_tmp = (angle_byte_tmp + angle_byte_inc) & 0x000000FF;
		angle_diff = angle_byte_to_angle[angle_byte_tmp];
		if(angle_diff > 180.0)
		{	angle_diff = angle_diff - 360.0;	}
		angle_diff = angle_diff - target_angle;
		if(abs(angle_diff) > abs(angle_diff_old))
			{	return;	}
		angle_hit = ACCURACY_FACTOR * 
			abs(atan((target->m_radius * effective_radius_from_angle_byte[angle_byte_tmp])
			/ (target->m_shot_travel_time * SHOT_VELOCITY)));
		if(abs(angle_diff) <= angle_hit)
			{
			m_easy_prey				= true;
			target->m_easy_prey		= true;
			}
		}
}

bool Tracked_Object::newton_iteration(	t_var_struct &t_var,
										t_const_struct &t_con,
										t_var_struct &delta_t_var)
{
  	double quotient, deriv_acos, shot_speed_tweak;
	double J[4][4];
	double fun_val[4];
	double tmp_val;

	if(t_con.v_shot == 0.0)
	{	return(false);	}
 
// 1. Assemble Jacobi matrix J
	deriv_acos = (t_var.x_shot * t_con.fx_ship + t_var.y_shot * t_con.fy_ship) / t_con.v_shot;
	deriv_acos = deriv_acos * deriv_acos;
	if(abs(deriv_acos) >= 1)
	{	deriv_acos = 0.99999;	}
	deriv_acos = sqrt(1 - deriv_acos);
	J[0][0] = t_var.t_fly;
	J[0][1] = 0.0;
	J[0][2] = - t_con.vx_trgt;
	J[0][3] = t_var.x_shot - t_con.vx_trgt;
	J[1][0] = 0.0;
	J[1][1] = t_var.t_fly;
	J[1][2] = - t_con.vy_trgt;
	J[1][3] = t_var.y_shot - t_con.vy_trgt;
	if(deriv_acos != 0.0)
		{
		J[2][0] = (t_con.fx_ship / t_con.v_shot) * (- 1.0 / deriv_acos);
		J[2][1] = (t_con.fy_ship / t_con.v_shot) * (- 1.0 / deriv_acos);
		}
	else // Apparently, setting these to 0 anyway will numerically stablize the algo
		{
		printf("NEWTON: deriv_acos == 0 !!!\n");
		J[2][0] = 0.0;
		J[2][1] = 0.0;
		}
	J[2][2] = - t_con.r_turn;
	J[2][3] = 0.0;
	J[3][0] = 2.0 * t_var.x_shot;
	J[3][1] = 2.0 * t_var.y_shot;
	J[3][2] = 0.0;
	J[3][3] = 0.0;
// 2. Update function values fun_val
	fun_val[0]	=	(t_con.x_ship - t_con.x_trgt) -
					(t_var.t_turn + t_var.t_fly) * t_con.vx_trgt +
						t_var.t_fly * t_var.x_shot;
	fun_val[1]	=	(t_con.y_ship - t_con.y_trgt) -
					(t_var.t_turn + t_var.t_fly) * t_con.vy_trgt +
						t_var.t_fly * t_var.y_shot;
	tmp_val		=	dot_prod(t_var.x_shot, t_var.y_shot, t_con.fx_ship, t_con.fy_ship) / t_con.v_shot;
	if(tmp_val > 1)
	{	tmp_val = 1.0;	}
	else if(tmp_val < - 1.0)
	{	tmp_val = - 1.0;	}
	tmp_val		=	acos(tmp_val);
	fun_val[2]	=	tmp_val - t_con.r_turn * t_var.t_turn;
	fun_val[3]	=	t_var.x_shot * t_var.x_shot +
					t_var.y_shot * t_var.y_shot -
					t_con.v_shot * t_con.v_shot;
	fun_val[0] = - fun_val[0];
	fun_val[1] = - fun_val[1];
	fun_val[2] = - fun_val[2];
	fun_val[3] = - fun_val[3];
// 3. Calculate delta_t_var: J * delta_t_var = fun_val
//    Population: X 0 x x
//                0 X x x
//                x x X 0
//                x x 0 0
// X: always != 0 (t_fly and r_turn)
//
	if(J[0][0] == 0.0) 
		{	
		printf("NEWTON: J[0][0] == 0!\n");
		return(false);
		}
	quotient = J[2][0] / J[0][0];
	J[2][0] = 0.0;
	J[2][1] = J[2][1];
	J[2][2] = J[2][2] - J[0][2] * quotient;
	J[2][3] = J[2][3] - J[0][3] * quotient;
	fun_val[2] = fun_val[2] - fun_val[0] * quotient;
	quotient = J[3][0] / J[0][0];
	J[3][0] = 0.0;
	J[3][1] = J[3][1];
	J[3][2] = J[3][2] - J[0][2] * quotient;
	J[3][3] = J[3][3] - J[0][3] * quotient;
	fun_val[3] = fun_val[3] - fun_val[0] * quotient;
//
	if(J[1][1] == 0.0) 
		{	
		printf("NEWTON: J[1][1] == 0!\n");
		return(false);
		}
	quotient = J[2][1] / J[1][1];
	J[2][0] = 0.0;
	J[2][1] = 0.0;
	J[2][2] = J[2][2] - J[1][2] * quotient;
	J[2][3] = J[2][3] - J[1][3] * quotient;
	fun_val[2] = fun_val[2] - fun_val[1] * quotient;
	quotient = J[3][1] / J[1][1];
	J[3][0] = 0.0;
	J[3][1] = 0.0;
	J[3][2] = J[3][2] - J[1][2] * quotient;
	J[3][3] = J[3][3] - J[1][3] * quotient;
	fun_val[3] = fun_val[3] - fun_val[1] * quotient;
//
	if(J[2][2] == 0.0)
		{	
		printf("NEWTON: J[2][2] == 0!\n");
		return(false);
		}
	quotient = J[3][2] / J[2][2];
	J[3][0] = 0.0;
	J[3][1] = 0.0;
	J[3][2] = 0.0;
	J[3][3] = J[3][3] - J[2][3] * quotient;
	fun_val[3] = fun_val[3] - fun_val[2] * quotient;
//
	if(J[3][3] == 0.0) 
		{	
		printf("NEWTON: J[3][3] == 0!\n");
		return(false);
		}
	delta_t_var.t_fly	= fun_val[3] / J[3][3];
	delta_t_var.t_turn	= (fun_val[2] - J[2][3] * delta_t_var.t_fly) /
							J[2][2];
	delta_t_var.y_shot	= (fun_val[1] - J[1][2] * delta_t_var.t_turn -
							J[1][3] * delta_t_var.t_fly) / J[1][1];
	delta_t_var.x_shot	= (fun_val[0] - J[0][1] * delta_t_var.y_shot -
							J[0][2] * delta_t_var.t_turn -
							J[0][3] * delta_t_var.t_fly) / J[0][0];

// 4. Update t_var: t_var = t_var + delta_t_var
	t_var.x_shot    += delta_t_var.x_shot;
	t_var.y_shot    += delta_t_var.y_shot;
	t_var.t_turn    += delta_t_var.t_turn;
	t_var.t_fly     += delta_t_var.t_fly;


	shot_speed_tweak = v_length(t_var.x_shot, t_var.y_shot);
	if(shot_speed_tweak == 0.0) 
		{	
		printf("NEWTON: shot_speed_tweak == 0!\n");
		return(false);
		}
	t_var.x_shot    = t_con.v_shot * t_var.x_shot / shot_speed_tweak;
	t_var.y_shot    = t_con.v_shot * t_var.y_shot / shot_speed_tweak;

	return(true);
}

double dot_prod(double x1, double y1, double x2, double y2)
{
    double dp;

    dp = x1 * x2 + y1 * y2;

    return(dp);
}

double cross_prod(double x1, double y1, double x2, double y2)
{
    double cp;

    cp = x1 * y2 - x2 * y1;

    return(cp);
}

double v_length(double x1, double y1)
{
    double length;

    length = x1 * x1 + y1 * y1;
    length = sqrt(length);

    return(length);
}
