package de.fhr.asteroids;

import java.io.DataInputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;

/**
 * The main class of the asteroids client. It connects to a running asteroids game
 * to play.
 * <p>
 *
 * <pre>
 * Usage: java de.fhr.asteroids.Client [-d] [-t] host [port]
 *        -d  Show display
 *        -t  Socket operations should time out (1s)
 * </pre>
 *
 * </p>
 * @author Florian Lutz
 * @version 1.1
 */
public final class Client extends Thread {

  /**
   * The internet address of the asteroids host.
   */
  private final InetAddress addr;

  /**
   * If the display should be shown.
   */
  private final boolean displ;

  /**
   * Array to save the sent control keys.
   */
  private final int[] keys;

  /**
   * The number of the last received frame.
   */
  private int lastframe;

  /**
   * The port to connect to at the asteroids host.
   */
  private final int port;

  /**
   * If the client has quit plaing.
   */
  private boolean quit;

  /**
   * The frame with endian decoded vector data.
   */
  private final Frame recvframe;

  /**
   * The received raw data paket from the host.
   */
  private final DatagramPacket recvpaket;

  /**
   * If the client should continue running.
   */
  private boolean run = true;

  /**
   * The control paket to sen to the host.
   */
  private final DatagramPacket sendpaket;

  /**
   * The network connected network socket.
   */
  private final DatagramSocket socket;

  /**
   * Create a new client instance that connects to the specified host.
   * @param host the host to connect to
   * @param port the port to connect to
   * @param displ id the display shoul be shown
   * @param to if the socket should have a timeout set
   * @throws UnknownHostException if the hostname can not be resolved
   * @throws SocketException if the connections fails
   */
  private Client(final String host, final int port, final boolean displ,
      final boolean to)
      throws UnknownHostException, SocketException {
    this.port = port;
    this.displ = displ;
    addr = InetAddress.getByName(host);
    socket = new DatagramSocket();
    socket.connect(addr, port);
    recvpaket = new DatagramPacket(new byte[1026], 1026);
    recvframe = new Frame();
    sendpaket = new DatagramPacket(new byte[8], 8);
    run = true;
    keys = new int[256];
    setPriority(10);
    System.out.println("Connected to " + addr + ":" + port);
    if (to) {
      socket.setSoTimeout(1000);
    }
  }

  /**
   * {@inheritDoc}
   * @see java.lang.Thread#run()
   */
  @Override
  public void run() {
    final Control control = new Control();
    final Control cforframe = new Control();
    final GameState state = new GameState();
    final Player player = new Player(state, control);
    Display display = null;

    int ping = 0;

    // saveKeysForFrame(ping, control);
    // send(ping, control);

    System.out.println("Playing asteroids, press ENTER to exit");
    System.out.println("Waiting for ship to appear...");

    while (run) {
      saveKeysForFrame(ping, control);
      send(ping, control);

      final Frame frame = receive();
      if (frame == null) {
        break;
      }

      control.clear();
      cforframe.setKeys(loadKeysForFrame(frame.ping));
      if (frame.ping != ping) {
        if (ping - frame.ping != 1 && ping + 256 - frame.ping != 1) {
          System.out.println("Wrong ping: " + frame.ping + " - " + ping);
        }
      }
      state.decode(frame, cforframe);

      player.play();

      if (displ) {
        if (display == null) {
          display = new Display(state, player);
        }
        display.repaint();
        if (display.aborted) {
          run = false;
          break;
        }
      }

      ping += 1;
      if (ping > 255) {
        ping = 0;
      }
    }

    state.printscore();
    quit(false, display);
  }

  /**
   * Load the keys data that has been sent with the specified ping.
   * @param ping the ping value
   * @return the corresponding keys
   */
  private int loadKeysForFrame(final int ping) {
    int n = ping - 1;
    if (n < 0) {
      n = 255;
    }
    return keys[n];
  }

  /**
   * Receives a frame of vector data from the remote host.
   * @return the received frame
   */
  private Frame receive() {
    try {
      socket.receive(recvpaket);
      final byte[] data = recvpaket.getData();

      if (data[0] == GAMEOVER[0] && Util.arraybegin(data, GAMEOVER)) {
        System.out.println("Received: game over");
        return null;
      }
      if (data[0] == BUSY[0] && Util.arraybegin(data, BUSY)) {
        System.out.println("Received: busy");
        return null;
      }

      recvframe.data(data);
      if (recvframe.number != lastframe + 1 && recvframe.number != 0
          && lastframe != 0) {
        System.out.println("lost frame(s) between " + lastframe + " and "
            + recvframe.number);
      }
      lastframe = recvframe.number;

      return recvframe;
    } catch (final IOException e) {
      if (run) {
        System.out.println("Failed to receive from " + addr + ":" + port);
        System.out.println(e.getMessage());
      }
      return null;
    }
  }

  /**
   * Saves the keys for the specified ping value.
   * @param ping the ping value
   * @param control the control paket with the keys
   */
  private void saveKeysForFrame(final int ping, final Control control) {
    keys[ping] = control.getKeys();
  }

  /**
   * Sends a control paket to the host.
   * @param ping the ping value to send within the paket
   * @param control the control paket to send
   */
  private void send(final int ping, final Control control) {
    sendpaket.setData(control.getData(ping));
    try {
      socket.send(sendpaket);
    } catch (final IOException e) {
      if (run) {
        System.out.println("Failed to send to " + addr + ":" + port);
        System.out.println(e.getMessage());
      }
    }
  }

  /**
   * Quit the running client.
   * @param shutdown id called from the shutdown hook
   * @param display the display to dispose
   */
  void quit(final boolean shutdown, final Display display) {
    try {
      if (!quit) {
        quit = true;
        run = false;
        Display.dispose(display);
        send(0, new Control());
        socket.close();
        System.out.println("Quit playing, connection closed");
        if (!shutdown) {
          System.exit(0);
        }
      }
    } catch (final Exception e) {
      // do nothing on quit
    } finally {
      quit = true;
    }
  }

  /**
   * The raw data of a BUSY paket from the host.
   */
  private static final byte[] BUSY;

  /**
   * The raw data of a GAME OVER paket from the host.
   */
  private static final byte[] GAMEOVER;

  static {
    GAMEOVER = new byte[] { 'g', 'a', 'm', 'e', ' ', 'o', 'v', 'e', 'r' };
    BUSY = new byte[] { 'b', 'u', 's', 'y' };
  }

  /**
   * The main method of the asteroids client.
   * @param args the command line arguments
   */
  public static void main(final String[] args) {
    boolean displ = false;
    boolean to = false;

    if (args.length > 0) {
      while (args[0] != null) {
        if (args[0].charAt(0) == '-') {
          if (args[0].equals("-d")) {
            displ = true;
          } else if (args[0].equals("-t")) {
            to = true;
          } else {
            System.out.println("Unknown option '" + args[0] + "'");
            usage();
            return;
          }
          System.arraycopy(args, 1, args, 0, args.length - 1);
          args[args.length - 1] = null;
        } else {
          break;
        }
      }
    }

    if (args.length < 1 || args[0] == null) {
      usage();
      return;
    }

    int port = 1979;
    if (args.length >= 2 && args[1] != null) {
      port = Integer.parseInt(args[1]);
    }

    try {
      final Client client = new Client(args[0], port, displ, to);

      Runtime.getRuntime().addShutdownHook(new Thread() {
        @Override
        public void run() {
          client.quit(true, null);
        }
      });

      client.start();

      try {
        final DataInputStream din = new DataInputStream(System.in);
        din.readChar();
      } catch (final Exception e) {
        // no error
      }

      client.run = false;

    } catch (final IOException e) {
      System.out.println("Unable to connect to " + args[0] + ":" + port);
    }
  }

  /**
   * Print the command line usage.
   */
  private static void usage() {
    System.out.println();
    System.out
        .println("Usage: java de.fhr.asteroids.Client [-d] [-t] host [port]");
    System.out.println("       -d  Show display");
    System.out.println("       -t  Socket operations should time out (1s)");
  }
}
