package de.fhr.asteroids;

/**
 * A control paket to send keys to the asteroids host.
 * @author Florian Lutz
 * @version 1.1
 */
final class Control {

  /**
   * The raw data of the paket.
   */
  private final byte[] data;

  /**
   * The keys to send.
   */
  private int keys;

  /**
   * Creates a new control paket.
   */
  Control() {
    data = new byte[] { 'c', 't', 'm', 'a', 'm', 'e', 0, 0 };
  }

  /**
   * Resets all keys.
   */
  synchronized void clear() {
    keys = 0;
  }

  /**
   * Sets the fire key.
   * @param press if the fire key should be pressed
   */
  synchronized void fire(final boolean press) {
    if (press) {
      keys |= 0x02;
    } else {
      keys &= ~0x02;
    }
  }

  /**
   * Returns the raw paket data.
   * @param ping the ping value to include
   * @return the raw paket data
   */
  synchronized byte[] getData(final int ping) {
    data[6] = (byte)keys;
    data[7] = (byte)ping;
    return data;
  }

  /**
   * Returns the keys value.
   * @return the keys value
   */
  synchronized int getKeys() {
    return keys;
  }

  /**
   * Sets the hyperspace jump key.
   * @param press if the hyperspace key should be pressed
   */
  synchronized void hyperspace(final boolean press) {
    if (press) {
      keys |= 0x01;
    } else {
      keys &= ~0x01;
    }
  }

  /**
   * Returns if hyperspace jump is pressed.
   * @return if hyperspace jump is pressed
   */
  synchronized boolean isHyper() {
    return (keys & 0x01) == 0x01;
  }

  /**
   * Returns if left run is pressed.
   * @return if left run is pressed
   */
  synchronized boolean isLeft() {
    return (keys & 0x10) == 0x10;
  }

  /**
   * Returns if right turn is pressed.
   * @return if right turn is pressed
   */
  synchronized boolean isRight() {
    return (keys & 0x08) == 0x08;
  }

  /**
   * Sets the key to turn left.
   * @param press if the left key should be pressed
   */
  synchronized void left(final boolean press) {
    if (press) {
      keys |= 0x10;
    } else {
      keys &= ~0x10;
    }
  }

  /**
   * Sets the key to turn right.
   * @param press if the right key should be pressed
   */
  synchronized void right(final boolean press) {
    if (press) {
      keys |= 0x08;
    } else {
      keys &= ~0x08;
    }
  }

  /**
   * Sets the keys value.
   * @param keys the value to set
   */
  synchronized void setKeys(final int keys) {
    this.keys = keys;
  }

  /**
   * Sets the thrust key.
   * @param press if the thrust key should be pressed
   */
  synchronized void thrust(final boolean press) {
    if (press) {
      keys |= 0x04;
    } else {
      keys &= ~0x04;
    }
  }
}
