package de.fhr.asteroids;

import java.awt.Color;
import java.awt.Graphics;

/**
 * A graphical context to draw at the game display. The coordinates are flipped
 * accross the y-axis to convert from asteroid vector coordinates to java screen
 * coordinates.
 * @author Florian Lutz
 * @version 1.1
 */
final class FlipGraphics {

  /**
   * The nested graphical context to do the real drawing with.
   */
  private Graphics g;

  /**
   * The height of the output area.
   */
  private final int h;

  /**
   * The height of the output area.
   */
  private final int w;

  /**
   * Creates a new graphical context for the specified dimensions.
   * @param w the width of the output area
   * @param h the height of the output area
   */
  FlipGraphics(final int w, final int h) {
    super();
    this.w = w;
    this.h = h;
  }

  /**
   * Internal function to draw rays and straights.
   * @param x the x-value of the base point
   * @param y the y-value of the base point
   * @param vx the x-value of the direction vector
   * @param vy the y-value of the direction vector
   * @param type specifies what to draw
   */
  private void drawInternal(final double x, final double y, final double vx,
      final double vy, final int type) {
    if (vx == 0 && vy == 0) {
      return;
    }
    final double a = vy / vx;
    final double t = (int)(y - a * x);
    double gx1 = 0;
    double gy1 = 0;
    double gx2 = 0;
    double gy2 = 0;
    if (Math.abs(a) < 0.75) {
      gx1 = 0;
      gy1 = a * gx1 + t;
      gx2 = w;
      gy2 = a * gx2 + t;
    } else {
      gy1 = 0;
      gy2 = h;
      if (Math.abs(a) == Double.POSITIVE_INFINITY || Math.abs(a) > 1E+14) {
        gx1 = gx2 = x;
      } else {
        gx1 = (gy1 - t) / a;
        gx2 = (gy2 - t) / a;
      }
    }
    if (type == 1 || type == 2) {
      double gx3 = 0;
      double gy3 = 0;
      if (Math.abs(a) < 0.75) {
        gx3 = vx < 0 ? gx1 : gx2;
        gy3 = vx < 0 ? gy1 : gy2;
      } else {
        gx3 = vy < 0 ? gx1 : gx2;
        gy3 = vy < 0 ? gy1 : gy2;
      }
      g.drawLine((int)x, (int)(h - y), (int)gx3, (int)(h - gy3));
      if (type == 2) {
        double hx3 = gx3 < w / 2 ? gx3 + w : gx3 - w;
        double hy3 = gy3 < h / 2 ? gy3 + h : gy3 - h;
        drawInternal(hx3, gy3, vx, vy, 3);
        drawInternal(gx3, hy3, vx, vy, 3);
        drawInternal(hx3, hy3, vx, vy, 3);
      }
    } else if (type == 3) {
      g.drawLine((int)gx1, (int)(h - gy1), (int)gx2, (int)(h - gy2));
    }
  }

  /**
   * Draws a circle.
   * @param x the x-value of the center
   * @param y the y-value of the center
   * @param r the radius of the circle
   */
  void drawCircle(final int x, final int y, final int r) {
    g.drawOval(x - r, (h - y) - r, 2 * r, 2 * r);
  }

  /**
   * Draws a line between two points.
   * @param x1 the x-value of the first point
   * @param y1 the y-value of the first point
   * @param x2 the x-value of the second point
   * @param y2 the y-value of the second point
   */
  void drawLine(final int x1, final int y1, final int x2, final int y2) {
    g.drawLine(x1, h - y1, x2, h - y2);
  }

  /**
   * Draws an oval.
   * @param x the x-value of the center
   * @param y the y-value of the center
   * @param width the width of the oval
   * @param height the height of the oval
   */
  void drawOval(final int x, final int y, final int width, final int height) {
    g.drawOval(x, h - y - height, width, height);
  }

  /**
   * Draws a ray to the border of the output area.
   * @param x the x-value of the start point
   * @param y the y-value of the start point
   * @param vx the x-value of the direction vector
   * @param vy the y-value of the direction vector
   */
  void drawRay(final double x, final double y, final double vx, final double vy) {
    drawInternal(x, y, vx, vy, 1);
  }

  /**
   * Draws a ray to the border of the output area. The ray warps around to the
   * opposite side once.
   * @param x the x-value of the start point
   * @param y the y-value of the start point
   * @param vx the x-value of the direction vector
   * @param vy the y-value of the direction vector
   */
  void drawRay2(final double x, final double y, final double vx, final double vy) {
    drawInternal(x, y, vx, vy, 2);
  }

  /**
   * Draws a straight across the output area.
   * @param x the x-value of the base point
   * @param y the y-value of the base point
   * @param vx the x-value of the direction vector
   * @param vy the y-value of the direction vector
   */
  void drawStraight(final double x, final double y, final double vx,
      final double vy) {
    drawInternal(x, y, vx, vy, 3);
  }

  /**
   * Draws a string at the specified position.
   * @param str the string to draw
   * @param x the x-value of the position
   * @param y the x-value of the position
   */
  void drawString(final String str, final int x, final int y) {
    g.drawString(str, x, h - y);
  }

  /**
   * Draws a filled circle.
   * @param x the x-value of the center
   * @param y the y-value of the center
   * @param r the radius of the circle
   */
  void fillCircle(final int x, final int y, final int r) {
    g.fillOval(x - r, h - y - r, 2 * r, 2 * r);
  }

  /**
   * Sets the color for following drawing operations.
   * @param color the color to set
   */
  void setColor(final Color color) {
    g.setColor(color);
  }

  /**
   * Sets the nested graphical context to use for drawing.
   * @param g the graphical context to use
   */
  void setGraphics(final Graphics g) {
    this.g = g;
  }
}
