package de.fhr.asteroids;

import java.awt.Color;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

/**
 * The interpretation of the asteroid game state as usable objects.
 * @author Florian Lutz
 * @version 1.1
 */
final class GameState implements Drawable {

  /**
   * Count of explosions.
   */
  private int explcount;

  /**
   * The explosions of the field.
   */
  private final List<Explosion> explosions;

  /**
   * Frames per second.
   */
  private int fps;

  /**
   * Count of decoded frames.
   */
  private int framecount;

  /**
   * Count of hyperjumps.
   */
  private int hyperjumps;

  /**
   * The most recent internal angle of the ship.
   */
  private int lastanglebyte;

  /**
   * The last time measurement.
   */
  private long lastmillis;

  /**
   * The current played level in the game.
   */
  private int level;

  /**
   * If the game is currently beweteen two levels.
   */
  private boolean levelpause;

  /**
   * Count of lives.
   */
  private int lives;

  /**
   * Saved score to detect overflow.
   */
  private int savedscore;

  /**
   * The currently displayed score.
   */
  private int score;

  /**
   * Count of players shots.
   */
  private int shotcount;

  /**
   * The score after the contest time.
   */
  private int timescore;

  /**
   * The asteroids on the field.
   */
  final List<Asteroid> asteroids;

  /**
   * The hostile saucer on the field.
   */
  Saucer saucer;

  /**
   * The players ship on the field.
   */
  Ship ship;

  /**
   * The shots on the field.
   */
  final List<Shot> shots;

  /**
   * Count of players shots on the field.
   */
  int shotsflying;

  /**
   * Creates a game state instance.
   */
  GameState() {
    super();
    asteroids = new ArrayList<Asteroid>();
    shots = new ArrayList<Shot>();
    explosions = new ArrayList<Explosion>();
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.Drawable#draw(de.fhr.asteroids.FlipGraphics)
   */
  public synchronized void draw(final FlipGraphics g) {
    g.setColor(Color.WHITE);

    for (Explosion e : explosions) {
      e.draw(g);
    }
    for (Asteroid a : asteroids) {
      a.draw(g);
    }
    for (Shot s : shots) {
      s.draw(g);
    }
    if (saucer != null) {
      saucer.draw(g);
    }
    if (ship != null) {
      ship.draw(g);
    }

    g.setColor(Color.WHITE);
    final int sc = shotcount - shotsflying;
    final String[] data =
        { framecount + "f", fps + "fps", level + "level", lives + "lives",
            hyperjumps + "hyper", score + "p",
            (framecount > 0 ? score * 60 / framecount : 0) + "pps",
            (framecount > 0 ? explcount * 600 / framecount : 0) + "hp10s",
            asteroids.size() + "ast", shotsflying + "fly", sc + "s",
            explcount + "h", (sc > 0 ? explcount * 1000 / sc : 0) + "‰",
            timescore > 0 ? timescore + "p" : "-", };

    for (int i = 0; i < data.length; i++) {
      g.drawString(data[i], 10 + i * 65, 748);
    }
  }

  /**
   * Called after a frame has been decoded for further operations.
   * @param cforframe the keys for this frame
   */
  private void afterDecode(final Control cforframe) {
    if (ship != null) {
      if (cforframe.isLeft()) {
        ship.left();
      }
      if (cforframe.isRight()) {
        ship.right();
      }
      ship.syncAngle();
      for (Shot s : shots) {
        if (s.ab < 0) {
          s.setab(ship.ab, ship.vx, ship.vy);
        }
      }
    }
    if (cforframe.isHyper()) {
      hyperjumps++;
    }
    while (score < savedscore) {
      score += 100000;
    }
    if (framecount > 0) {
      framecount++;
      if (framecount % 50 == 0) {
        final long curmillis = System.currentTimeMillis();
        fps = Math.round(50000 / (curmillis - lastmillis));
        lastmillis = curmillis;
      }
      if (framecount % 3600 == 0) {
        System.out.println((framecount / 3600) + " minutes, " + framecount
            + " frames, level " + level + ", score " + score);
      }
      if (asteroids.size() == 0) {
        if (!levelpause) {
          levelpause = true;
          level++;
        }
      } else {
        if (levelpause) {
          System.out.println(framecount + ": Level " + level);
          levelpause = false;
        }
      }
    }
    if (framecount == 18000) {
      timescore = score;
    }
  }

  /**
   * Clears the updated flag of all game objects.
   */
  private void clearUpdated() {
    if (ship != null) {
      ship.updated = false;
    }
    if (saucer != null) {
      saucer.updated = false;
    }
    for (Asteroid a : asteroids) {
      a.updated = false;
    }
    for (Shot s : shots) {
      s.updated = false;
    }
    for (Explosion e : explosions) {
      e.updated = false;
    }
  }

  /**
   * Process a JRSL jump in the vector ram.
   * @param addr the target address of the jump
   * @param px the x-value of the vector ahead of the jump
   * @param py the s-value of the vector ahead of the jump
   * @param gsf the scaling factor of the vector
   */
  private void jsrl(final int addr, final int px, final int py, final int gsf) {
    switch (addr) {
    case 0x8F3: // Asteroid 1
      procAsteroid(px, py, 1, gsf);
      break;
    case 0x8FF: // Asteroid 2
      procAsteroid(px, py, 2, gsf);
      break;
    case 0x90D: // Asteroid 3
      procAsteroid(px, py, 3, gsf);
      break;
    case 0x91A: // Asteroid 4
      procAsteroid(px, py, 4, gsf);
      break;
    case 0x929: // Saucer
      procSaucer(px, py, gsf);
      break;
    case 0x880: // Explosion (3)
      procExplosion(px, py, 3, gsf);
      break;
    case 0x896: // Explosion (2)
      procExplosion(px, py, 2, gsf);
      break;
    case 0x8B5: // Explosion (1)
      procExplosion(px, py, 1, gsf);
      break;
    case 0x8D0: // Explosion (0)
      procExplosion(px, py, 0, gsf);
      break;
    case 0xA6D: // Live
      procLive(px, py);
      break;
    default:
      for (int i = 0; i < Tables.JSRLADDR.length; i++) {
        if (Tables.JSRLADDR[i] == addr) {
          procChar(Tables.JSRLCHAR[i], px, py);
          break;
        }
      }
    }
  }

  /**
   * Proces an asteroid.
   * @param px the x-value of the position
   * @param py the y-value of the position
   * @param type the type of the asteroid
   * @param gsf the scaling factor
   */
  private void procAsteroid(final int px, final int py, final int type,
      final int gsf) {
    for (Asteroid a : asteroids) {
      if (!a.updated && a.distance2(px, py) <= 7 && a.sameTypeGsf(type, gsf)) {
        a.update(px, py);
        return;
      }
    }
    asteroids.add(new Asteroid(px, py, type, gsf));
  }

  /**
   * Proces a 'written' character.
   * @param c the character
   * @param px the x-value of the position
   * @param py the y-value of the position
   */
  private void procChar(final char c, final int px, final int py) {
    if (px == 100 && py == 748 && Character.isDigit(c)) {
      score = score * 10 + Integer.parseInt(c + "");
    } else if (c == 'S' && px == 400 && (py == 600 || py == 664)) {
      reset();
    }
  }

  /**
   * Proces an explosion.
   * @param px the x-value of the position
   * @param py the y-value of the position
   * @param type the type of the explosion
   * @param gsf the scaling factor
   */
  private void procExplosion(final int px, final int py, final int type,
      final int gsf) {
    for (Explosion e : explosions) {
      if (!e.updated && e.px == px && e.py == py) {
        e.update(px, py, type, gsf);
        return;
      }
    }
    explosions.add(new Explosion(px, py, type, gsf));
    explcount++;
  }

  /**
   * Proces a live symbol.
   * @param px the x-value of the position
   * @param py the y-value of the position
   */
  private void procLive(final int px, final int py) {
    if (px == 160 && py == 724) {
      lives++;
    }
  }

  /**
   * Proces a shot.
   * @param px the x-value of the position
   * @param py the y-value of the position
   * @param gsf the scaling factor
   */
  private void procSaucer(final int px, final int py, final int gsf) {
    if (saucer == null) {
      saucer = new Saucer(px, py, gsf);
    } else {
      saucer.update(px, py);
    }
  }

  /**
   * Proces the ship.
   * @param px the x-value of the position
   * @param py the y-value of the position
   * @param dx the x-value of the direction
   * @param dy the y-value of the direction
   */
  private void procShip(final int px, final int py, final int dx, final int dy) {
    if (ship == null) {
      ship = new Ship(px, py, dx, dy, lastanglebyte);
      if (framecount == 0) {
        lastmillis = System.currentTimeMillis();
        framecount++;
      }
      System.out.println(framecount + ": Ship on the field");
    } else {
      ship.update(px, py, dx, dy);
    }
  }

  /**
   * Process a shot.
   * @param px the x-value of the shot position
   * @param py the y-value of the shot position
   */
  private void procShot(final int px, final int py) {
    for (Shot s : shots) {
      if (!s.updated && s.distance2(px, py) <= 13.0) {
        s.update(px, py);
        return;
      }
    }
    final Shot shot = Shot.create(px, py);
    if (saucer != null && saucer.distance2(px, py) <= 23.0) {
      shot.hostile = true;
    } else {
      shotcount++;
      shotsflying++;
    }
    shots.add(shot);
  }

  /**
   * Removes old game objects that have not been updated.
   */
  private void removeOld() {
    if (ship != null && !ship.updated) {
      lastanglebyte = ship.ab;
      ship = null;
      System.out.println(framecount + ": Ship away");
    }
    if (saucer != null && !saucer.updated) {
      saucer = null;
    }
    final ListIterator<Asteroid> ait = asteroids.listIterator();
    while (ait.hasNext()) {
      if (!ait.next().updated) {
        ait.remove();
      }
    }
    final ListIterator<Shot> sit = shots.listIterator();
    while (sit.hasNext()) {
      final Shot s = sit.next();
      if (!s.updated) {
        if (!s.hostile) {
          shotsflying--;
        }
        Shot.remove(s);
        sit.remove();
      }
    }
    final ListIterator<Explosion> eit = explosions.listIterator();
    while (eit.hasNext()) {
      if (!eit.next().updated) {
        eit.remove();
      }
    }
  }

  /**
   * Resets the game state.
   */
  private void reset() {
    score = 0;
    savedscore = 0;
    lastanglebyte = 0;
    lastmillis = 0;
    framecount = 0;
    shotcount = 0;
    explcount = 0;
    hyperjumps = 0;
    level = 0;
    levelpause = false;
  }

  /**
   * Decodes a frame of the vector ram.
   * @param frame the frame to decode
   * @param cforframe the related keys that have been sent
   */
  synchronized void decode(final Frame frame, final Control cforframe) {
    try {
      savedscore = score;
      score = 0;
      lives = 0;
      clearUpdated();

      int pos = 1;
      int px = 0;
      int py = 0;
      int dx = 0;
      int dy = 0;
      int dx1 = 0;
      int dy1 = 0;
      int gsf = 0;
      int z = 0;

      while (pos < frame.data.length) {
        final int op = frame.data[pos] >> 12;

        switch (op) {
        case 0xa: // LABS
          py = (frame.data[pos] & 0x3ff) - 128;
          px = frame.data[pos + 1] & 0x3ff;
          gsf = frame.data[pos + 1] >> 12;
          pos += 2;
          break;

        case 0xb: // HALT
          pos++;
          return;

        case 0xc: // JSRL
          jsrl(frame.data[pos] & 0xfff, px, py, gsf);
          pos++;
          break;

        case 0xd: // RTSL
          pos++;
          return;

        case 0xe: // JMPL
          return;

        case 0xf: // SVEC
          pos++;
          break;

        default: // VECTR 0 - 9
          dy = frame.data[pos] & 0x3ff;
          if ((frame.data[pos] & 0x400) != 0) {
            dy = -dy;
          }

          dx = frame.data[pos + 1] & 0x3ff;
          if ((frame.data[pos + 1] & 0x400) != 0) {
            dx = -dx;
          }

          z = frame.data[pos + 1] >> 12;

          if (dx == 0 && dy == 0 && z == 15) {
            procShot(px, py);
          } else if (op == 6 && z == 12 && dx != 0 && dy != 0) {
            if (dx1 == 0 && dy1 == 0) {
              dx1 = dx;
              dy1 = dy;
            } else {
              procShip(px, py, dx1 - dx, dy1 - dy);
            }
          } else {
            dx1 = 0;
            dy1 = 0;
          }

          pos += 2;
          break;
        }
      }
    } finally {
      removeOld();
      afterDecode(cforframe);
    }
  }

  /**
   * Print the score to the console.
   */
  synchronized void printscore() {
    if (timescore > 0) {
      System.out.println("Score " + timescore);
    } else if (score > 0) {
      System.out.println("Score " + score);
    }
  }
}
