package de.fhr.asteroids;

import java.awt.Color;
import java.util.Arrays;

/**
 * The asteroids players ship.
 * @author Florian Lutz
 * @version 1.1
 */
final class Ship extends GameObject {

  /**
   * The cached point the ship should aim at.
   */
  private final Point aimpoint;

  /**
   * The cached point that can be fired at.
   */
  private Point firepoint;

  /**
   * The most recent value of the internal angel.
   */
  private int lastab;

  /**
   * The count of saved direction values.
   */
  private int saven;

  /**
   * Saved x-values of direction vector.
   */
  private final int[] savesdx;

  /**
   * Saved y-values of direction vector.
   */
  private final int[] savesdy;

  /**
   * Saved turn operations.
   */
  private final int[] saveturn;

  /**
   * Uncorrected x-value of direction vector.
   */
  private int screendx;

  /**
   * Uncorrected y-value of direction vector.
   */
  private int screendy;

  /**
   * Synchronized internal angle of the ship.
   */
  int ab;

  /**
   * Corrected x-value of direction vector.
   */
  double dx;

  /**
   * Corrected y-value of direction vector.
   */
  double dy;

  /**
   * Creates a new ship instance with the specified parameters.
   * @param px the x-value of the ship position
   * @param py the y-value of the ship direction
   * @param dx the x-value of the ship position
   * @param dy the ý-value of the ship direction
   * @param ab the internal angle
   */
  Ship(final int px, final int py, final int dx, final int dy, final int ab) {
    super(px, py);
    aimpoint = new Point(px, py);
    firepoint = new Point(px, py);
    this.ab = ab;
    lastab = ab;
    savesdx = new int[5];
    savesdy = new int[5];
    saveturn = new int[5];
    saven = 0;
    update(px, py, dx, dy);
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.Drawable#draw(de.fhr.asteroids.FlipGraphics)
   */
  public void draw(final FlipGraphics g) {
    final int r = radius();

    g.setColor(Color.GRAY);
    g.drawRay(px, py, dx, dy);
    g.drawString("" + ab, px - 24, py - r - 24);

    final int ddx = (int)Util.vxscale(screendx, screendy, r);
    final int ddy = (int)Util.vyscale(screendx, screendy, r);
    final int x1 = px + ddx;
    final int y1 = py + ddy;
    final int x2 = px - ddy / 2;
    final int y2 = py + ddx / 2;
    final int x3 = px + ddy / 2;
    final int y3 = py - ddx / 2;
    final int x4 = x2 - ddx;
    final int y4 = y2 - ddy;
    final int x5 = x3 - ddx;
    final int y5 = y3 - ddy;

    g.setColor(Color.WHITE);
    g.fillCircle(px, py, 2);
    g.drawLine(x2, y2, x1, y1);
    g.drawLine(x3, y3, x1, y1);
    g.drawLine(x2, y2, x3, y3);
    g.drawLine(x2, y2, x4, y4);
    g.drawLine(x3, y3, x5, y5);
    g.drawLine(x4, y4, x5, y5);

    g.setColor(Color.GREEN);
    drawSpeed(g);

    String v = (Math.round(Util.vlen(vx, vy) * 100.0) / 100.0) + "";
    v = v.length() < 4 ? v + "0" : v;
    g.drawString(v, px - 12, py - radius() - 12);
  }

  /**
   * Calculates the point to aim at with the specified shotspeed.
   * @param target the target to aim at
   * @param shotspeed the shotspeed for calculation
   */
  private void aimpoint(final Target target, final double shotspeed) {
    final int spx = Util.warpx(px, target.px);
    final int spy = Util.warpy(py, target.py);
    final double v = target.speed;
    final double w = shotspeed;
    final double c = target.shipdist;
    final double vwq = (v * v) / (w * w);
    final double cosb =
        ((spx - target.px) * target.vx + (spy - target.py) * target.vy)
            / (v * Util.vlen(spx - target.px, spy - target.py));
    final double b =
        (-2 * (v / w) * c * cosb + Math
            .sqrt(4 * c * c * (vwq * cosb + 1 - vwq)))
            / (2 - 2 * vwq);
    final double a = v * b / w;
    aimpoint.px = (int)(target.px + target.vx * a / v);
    aimpoint.py = (int)(target.py + target.vy * a / v);
  }

  /**
   * Saves a turn the ship made.
   * @param turn the turn
   */
  private void saveturn(final int turn) {
    if (saven < 5) {
      saven++;
    }
    for (int i = saven - 1; i >= 1; i--) {
      savesdx[i] = savesdx[i - 1];
      savesdy[i] = savesdy[i - 1];
      saveturn[i] = saveturn[i - 1];
    }
    savesdx[0] = screendx;
    savesdy[0] = screendy;
    saveturn[0] = turn;
  }

  /**
   * Calculates the point to aim at for the given target.
   * @param target the target to aim at
   * @return the calculated holdpoint
   */
  Point aimpoint(final Target target) {
    if (target.speed < 0.01) {
      aimpoint.px = target.px;
      aimpoint.py = target.py;
    } else {
      aimpoint(target, Tables.SHOTSPEED);
      //final int nab = pointab(aimpoint.px, aimpoint.py);
      //aimpoint(target, Util.vlen(Util.sspx(nab, vx), Util.sspy(nab, vy)));
    }
    aimpoint.ab = pointab(aimpoint.px, aimpoint.py);
    return aimpoint;
  }

  /**
   * Calculates the firepoint for the given aimpoint.
   * @param aim the calculated aimpoint
   * @return the point that can be fired at
   */
  Point firepoint(final Point aim) {
    final double a1 = Math.tan(Tables.ANGLE[aim.ab] * Math.PI / 180);
    if (Math.abs(a1) == Double.POSITIVE_INFINITY || Math.abs(a1) > 1E+14) {
      firepoint.px = px;
      firepoint.py = aim.py;
    } else {
      final double t1 = Util.warpy(py, aim.py) - a1 * Util.warpx(px, aim.px);
      final double a2 = -1 / a1;
      final double t2 = aim.py - a2 * aim.px;
      firepoint.px = (int)Math.round((t2 - t1) / (a1 - a2));
      firepoint.py = (int)Math.round(a1 * firepoint.px + t1);
    }
    return firepoint;
  }

  /**
   * The ship has been turned left.
   */
  void left() {
    saveturn(1);
    ab++;
    if (ab > 255) {
      ab = 0;
    }
  }

  /**
   * Calculates the next nearest internal angle that could reach the specified
   * point.
   * @param ax the x-value of the target vector
   * @param ay the y-value of the target vector
   * @return the nearest value for the internal angle
   */
  int pointab(final int ax, final int ay) {
    double a =
        Util.vangle(100, 0, Util.warpx(ax, px) - px, Util.warpy(ay, py) - py);
    if (a < 0) {
      a += 360;
    }
    int nab = ab;
    for (int i = ab + 1; i <= ab + 42; i++) {
      final int j = i > 255 ? i - 256 : i;
      if (Util.anglediff(Tables.ANGLE[j], a) < Util.anglediff(
          Tables.ANGLE[nab], a)) {
        nab = j;
      } else {
        break;
      }
    }
    for (int i = ab - 1; i >= ab - 42; i--) {
      final int j = i < 0 ? i + 256 : i;
      if (Util.anglediff(Tables.ANGLE[j], a) < Util.anglediff(
          Tables.ANGLE[nab], a)) {
        nab = j;
      } else {
        break;
      }
    }
    return nab;
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.ScreenObject#radius()
   */
  @Override
  int radius() {
    return 16;
  }

  /**
   * The ship has been turned right.
   */
  void right() {
    saveturn(-1);
    ab--;
    if (ab < 0) {
      ab = 255;
    }
  }

  /**
   * The internal angle is synched with the displayed direction.
   */
  void syncAngle() {
    final int[] goodab = Tables.goodab(screendx, screendy);
    if (Arrays.binarySearch(goodab, ab) < 0) {
      int[] isect = null;
      if (saven > 0) {
        isect = Tables.goodab(savesdx[saven - 1], savesdy[saven - 1]);
        for (int i = saven - 2; i >= 0; i--) {
          Util.arrayadd(isect, saveturn[i]);
          final int[] abn = Tables.goodab(savesdx[i], savesdy[i]);
          isect = Util.arrayisect(isect, abn);
        }
      }
      if (isect != null && isect.length == 1) {
        ab = isect[0];
      } else {
        int nab = 0;
        for (int i = 0; i < goodab.length; i++) {
          if (Math.abs(goodab[i] - lastab) < Math.abs(nab - lastab)) {
            nab = goodab[i];
          }
        }
        if (nab == 0) {
          nab = goodab[0];
        }
        ab = nab;
      }
    }

    dx = Util.vectorx(Tables.ANGLE[ab], 100);
    dy = Util.vectory(Tables.ANGLE[ab], 100);
    lastab = ab;
  }

  /**
   * Updates the ship's position and direction.
   * @param npx the new x-value for the position
   * @param npy the new y-value for the position
   * @param ndx the new x-value for the direction vector
   * @param ndy the new y-value for the direction vector
   */
  void update(final int npx, final int npy, final int ndx, final int ndy) {
    update(npx, npy);
    screendx = ndx;
    screendy = ndy;
  }
}
