package de.fhr.asteroids;

import java.awt.Color;

/**
 * Superclass of targets to shoot at.
 * @author Florian Lutz
 * @version 1.1
 */
abstract class Target extends GameObject implements Comparable<Target> {

  /**
   * The point to aim at to hit the target.
   */
  final Point aimpoint;

  /**
   * The point that can be fired at.
   */
  final Point firepoint;

  /**
   * The scaling factor of the target object.
   */
  final int gsf;

  /**
   * If the target has been shot at.
   */
  int hasbeenshot;

  /**
   * The targets distance to the ship.
   */
  double shipdist;

  /**
   * The evaluation value of the target.
   */
  int value;

  /**
   * If the the target will be hit by a shot.
   */
  boolean willbehit;

  /**
   * If the target will collide with the ship.
   */
  boolean willcollide;

  /**
   * Creates a new target object with the specified values.
   * @param px the x-value of the position
   * @param py the y-value of the position
   * @param gsf the scaling factor
   */
  Target(final int px, final int py, final int gsf) {
    super(px, py);
    this.gsf = gsf;
    aimpoint = new Point(px, py);
    firepoint = new Point(px, py);
  }

  /**
   * {@inheritDoc}
   * @see java.lang.Comparable#compareTo(java.lang.Object)
   */
  @Override
  public int compareTo(final Target t) {
    if (value == t.value) {
      return 0;
    }
    return value < t.value ? -1 : 1;
  }

  /**
   * Called by supclasses to draw the common values of target objects.
   * @param g the graphical context for drwaing
   * @param ox the x-offset from the objects position
   * @param oy the y-offset from the objects position
   */
  final void drawValues(final FlipGraphics g, final int ox, final int oy) {
    final String v = (Math.round(speed * 10.0) / 10.0) + "";
    g.drawString(value + "  " + v, px + ox, py + oy);
    if (shipdist > 0) {
      g.drawString(Math.round(shipdist) + "", px + ox + 7, py + oy - 12);
    }
    g.setColor(Color.GREEN);
    if (aimpoint.px != px || aimpoint.py != py) {
      aimpoint.draw(g);
    }
    if (firepoint.px != aimpoint.px || firepoint.py != aimpoint.py) {
      g.drawCircle(firepoint.px, firepoint.py, 4);
    }
  }
}
