package de.fhr.asteroids;

/**
 * Static utility class for vector calculations and other operations.
 * @author Florian Lutz
 * @version 1.1
 */
final class Util {

  /**
   * Private default constructor.
   */
  private Util() {
    // no initialisation needed
  }

  /**
   * Temporary array to calculate intersections.
   */
  private static final int[] TMPAR;

  static {
    TMPAR = new int[10];
  }

  /**
   * Calculates the difference of two byte values, warping around zero.
   * @param ab1 the first byte value
   * @param ab2 the second byte value
   * @return the absolute difference of the values
   */
  static int abdiff(final int ab1, final int ab2) {
    if (ab1 < ab2) {
      return Math.min(ab2 - ab1, 256 + ab1 - ab2);
    }
    return Math.min(ab1 - ab2, 256 + ab2 - ab1);
  }

  /**
   * Returns the unoriented difference between two absolute angles.
   * @param a1 the first angle in degrees
   * @param a2 the second angle in the degrees
   * @return the difference between 0 and 180
   */
  static double anglediff(final double a1, final double a2) {
    final double b1 = a1 % 360;
    final double b2 = a2 % 360;
    if (b1 < b2) {
      return Math.min(b2 - b1, 360 + b1 - b2);
    }
    return Math.min(b1 - b2, 360 + b2 - b1);
  }

  /**
   * Adds a constant value to each element of an array.
   * @param ar the array
   * @param v the value to add
   */
  static void arrayadd(final int[] ar, final int v) {
    for (int i = 0; i < ar.length; i++) {
      ar[i] += v;
    }
  }

  /**
   * Looks for specific content at the beginning of a byte array.
   * @param base the byte array to be checked
   * @param cont the content to be found
   * @return if the content could be found
   */
  static boolean arraybegin(final byte[] base, final byte[] cont) {
    if (base.length < cont.length) {
      return false;
    }
    for (int i = 0; i < cont.length; i++) {
      if (base[i] != cont[i]) {
        return false;
      }
    }
    return true;
  }

  /**
   * Returns the intersection of two array.
   * @param ar1 the first array
   * @param ar2 the second array
   * @return the intersecion
   */
  static int[] arrayisect(final int[] ar1, final int[] ar2) {
    int n = 0;
    for (int i = 0; i < ar1.length; i++) {
      for (int j = 0; j < ar2.length; j++) {
        if (ar1[i] == ar2[j]) {
          boolean e = false;
          for (int k = 0; k < n; k++) {
            if (ar1[i] == TMPAR[k]) {
              e = true;
              break;
            }
          }
          if (!e) {
            TMPAR[n++] = ar1[i];
          }
        }
      }
    }
    final int[] isect = new int[n];
    System.arraycopy(TMPAR, 0, isect, 0, n);
    return isect;
  }

  /**
   * Calculates the distance between two points.
   * @param x1 the x-value of the first point
   * @param y1 the y-value of the first point
   * @param x2 the x-value of the second point
   * @param y2 the y-value of the second point
   * @return the distance
   */
  static double distance(final double x1, final double y1, final double x2,
      final double y2) {
    return Util.vlen(x2 - x1, y2 - y1);
  }

  /**
   * Calculates the distance between two points, warping around the field if the
   * distance is shorter.
   * @param x1 the x-value of the first point
   * @param y1 the y-value of the first point
   * @param x2 the x-value of the second point
   * @param y2 the y-value of the second point
   * @return the distance
   */
  static double distance2(final double x1, final double y1, final double x2,
      final double y2) {
    double ax = 0;
    double ay = 0;
    if (x1 > x2) {
      ax = Math.min(x1 - x2, x2 + Tables.EXTENT_X - x1);
    } else {
      ax = Math.min(x2 - x1, x1 + Tables.EXTENT_X - x2);
    }
    if (y1 > y2) {
      ay = Math.min(y1 - y2, y2 + Tables.EXTENT_Y - y1);
    } else {
      ay = Math.min(y2 - y1, y1 + Tables.EXTENT_Y - y2);
    }
    return Util.vlen(ay, ax);
  }

  /**
   * Calculates the x-value of the velocity vector of a shot in relation the the
   * ships speed.
   * @param ab the internal angle of the shot
   * @param svx the x-value of the ships velocity vector
   * @return the x-value of the shots velocity vector
   */
  static double sspx(final int ab, final double svx) {
    return Util.vectorx(Tables.ANGLE[ab], Tables.SHOTSPEED) + svx;
  }

  /**
   * Calculates the y-value of the velocity vector of a shot in relation the the
   * ships speed.
   * @param ab the internal angle of the shot
   * @param svy the y-value of the ships velocity vector
   * @return the y-value of the shots velocity vector
   */
  static double sspy(final int ab, final double svy) {
    return Util.vectory(Tables.ANGLE[ab], Tables.SHOTSPEED) + svy;
  }

  /**
   * Calculates the angle between two vectors.
   * @param vx1 the x-avlue of the first vector
   * @param vy1 the y-avlue of the first vector
   * @param vx2 the x-avlue of the second vector
   * @param vy2 the y-avlue of the second vector
   * @return the angle in degrees between -180 and 180
   */
  static double vangle(final double vx1, final double vy1, final double vx2,
      final double vy2) {
    final double vlen1 = vlen(vx1, vy1);
    final double vlen2 = vlen(vx2, vy2);
    final double a = (vx1 * vx2 + vy1 * vy2) / (vlen1 * vlen2);
    return Math.acos(a) * Math.signum((vx1 * vy2) - (vx2 * vy1)) * 180
        / Math.PI;
  }

  /**
   * Calculates the x-value of a vector from polar coordinates.
   * @param angle the absolute angle of the vector
   * @param len the length of the vector
   * @return the x-value of the vector
   */
  static double vectorx(final double angle, final double len) {
    return Math.cos(angle * Math.PI / 180) * len;
  }

  /**
   * Calculates the y-value of a vector from polar coordinates.
   * @param angle the absolute angle of the vector
   * @param len the length of the vector
   * @return the y-value of the vector
   */
  static double vectory(final double angle, final double len) {
    return Math.sin(angle * Math.PI / 180) * len;
  }

  /**
   * Returns the length of a vector.
   * @param x the x-value of the vector
   * @param y the y-value of the vector
   * @return the length of the vector
   */
  static double vlen(final double x, final double y) {
    return Math.sqrt(x * x + y * y);
  }

  /**
   * Returns the rotated x-value of a vector.
   * @param vx the original x-value of the vector
   * @param vy the original y-value of the vector
   * @param angle the rotation angle in degrees
   * @return the rotated x-value
   */
  static double vxrotate(final double vx, final double vy, final double angle) {
    final double arc = angle * Math.PI / 180;
    return vx * Math.cos(arc) - vy * Math.sin(arc);
  }

  /**
   * Returns the scaled x-value of a vector.
   * @param vx the original x-value of the vector
   * @param vy the original y-value of the vector
   * @param len the new length of the vector
   * @return the scaled x-value
   */
  static double vxscale(final double vx, final double vy, final double len) {
    return vx * len / vlen(vx, vy);
  }

  /**
   * Returns the rotated y-value of a vector.
   * @param vx the original x-value of the vector
   * @param vy the original y-value of the vector
   * @param angle the rotation angle in degrees
   * @return the rotated y-value
   */
  static double vyrotate(final double vx, final double vy, final double angle) {
    final double arc = angle * Math.PI / 180;
    return vx * Math.sin(arc) + vy * Math.cos(arc);
  }

  /**
   * Returns the scaled y-value of a vector.
   * @param vx the original x-value of the vector
   * @param vy the original y-value of the vector
   * @param len the new length of the vector
   * @return the scaled y-value
   */
  static double vyscale(final double vx, final double vy, final double len) {
    return vy * len / vlen(vx, vy);
  }

  /**
   * Warps a point around the field if the distance to a related point gets
   * shorter.
   * @param x the x-value of the point to warp around
   * @param relx the x-value of the related point
   * @return the warped x-value of the point
   */
  static int warpx(final int x, final int relx) {
    if (relx < x) {
      return x - relx < relx + Tables.EXTENT_X - x ? x : x - Tables.EXTENT_X;
    }
    return relx - x < x + Tables.EXTENT_X - relx ? x : x + Tables.EXTENT_X;
  }

  /**
   * Warps a point around the field if the distance to a related point gets
   * shorter.
   * @param y the x-value of the point to warp around
   * @param rely the x-value of the related point
   * @return the warped x-value of the point
   */
  static int warpy(final int y, final int rely) {
    if (rely < y) {
      return y - rely < rely + Tables.EXTENT_Y - y ? y : y - Tables.EXTENT_Y;
    }
    return rely - y < y + Tables.EXTENT_Y - rely ? y : y + Tables.EXTENT_Y;
  }
}
